package com.bxm.egg.user.login.strategy.support;

import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.convert.EggSixParamConverter;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserWechatGrantService;
import com.bxm.egg.user.integration.EggSixEnjoyIntegrationService;
import com.bxm.egg.user.login.strategy.AbstractUserLoginStrategy;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.WechatBindMobileOneKeyLoginParam;
import com.bxm.egg.user.onekey.OneKeyService;
import com.bxm.newidea.component.bo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Optional;

/**
 * 一键登录绑定手机号登录策略
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/17 5:48 下午
 */
@Slf4j
@Service
public class WechatOneKeyBindMobileLoginStrategy extends AbstractUserLoginStrategy<WechatBindMobileOneKeyLoginParam> {

    @Resource
    private UserInfoService userInfoService;

    @Resource
    private UserExtendService userExtendService;

    /**
     * 极光一键登录service
     */
    @Resource
    private OneKeyService oneKeyService;

    @Resource
    private UserWechatGrantService userWechatGrantService;

    @Resource
    private EggSixEnjoyIntegrationService eggSixEnjoyIntegrationService;

    @Override
    protected Message beforeLoginAuthentication(WechatBindMobileOneKeyLoginParam loginParam) {
        Message message = oneKeyService.checkToken(loginParam.getToken());
        //一键绑定是否正确
        if (!message.isSuccess()) {
            return Message.build(false, "一键登录绑定失败");
        }

        //设置手机号码
        String phone = message.getParam("phone");
        loginParam.setPhone(phone);

        //验证是否授权
        Optional<UserGrantWechatAppEntity> userUnionIdExist = userWechatGrantService.userInfoUnionIdExist(loginParam.getUnionId());

        if (!userUnionIdExist.isPresent()) {
            return Message.build(false, "微信用户未授权");
        }


        return Message.build(true);
    }

    @Override
    protected Long checkExists(WechatBindMobileOneKeyLoginParam loginParam) {

        //判断用户绑定的手机号是否已经存在
        Optional<Long> userPhoneExist = userInfoService.userPhoneExist(loginParam.getPhone());

        //如果已经存在
        if (!userPhoneExist.isPresent()) {
            return null;
        }
        loginParam.setUserId(userPhoneExist.get());
        return userPhoneExist.get();
    }

    @Override
    public UserLoginTypeEnum type() {
        return UserLoginTypeEnum.WECHAT_BIND_MOBILE_ONE_KEY_LOGIN;
    }

    @Override
    protected boolean saveIdentity(WechatBindMobileOneKeyLoginParam loginParam, Long userId) {
        return true;
    }

    @Override
    protected void afterLogin(LoginResultDTO resultDTO, WechatBindMobileOneKeyLoginParam loginParam) {
        //理论登录绑定手机号只会执行一次，所以放到这个方法里面也可以
        //登录后绑定手机号
        userInfoService.bindPhone(resultDTO.getUserId(), loginParam.getPhone());
        //绑定手机号为用户名
        userInfoService.bindUsername(resultDTO.getUserId(), loginParam.getPhone());
        //修改绑定用户
        userWechatGrantService.updateWechatGrantBindUserId(resultDTO.getUserId(), loginParam.getUnionId());
        //微信登录绑定手机号也要初始化一下资料完成度
        userExtendService.initComplete(resultDTO.getUserId());

        //调用六享微信绑定奖励
        eggSixEnjoyIntegrationService.bind(EggSixParamConverter.convert(loginParam, resultDTO.getUserId()));
    }

    @Override
    protected void fillUserInfo(WechatBindMobileOneKeyLoginParam loginParam, UserInfoEntity newUser) {
        String newHeadImg = userExtendService.uploadWechatImage(loginParam.getHeadImageUrl(), loginParam, newUser.getId());

        newUser.setHeadImg(newHeadImg);
        newUser.setNickname(loginParam.getNickName());
        newUser.setSex(loginParam.getSex());
    }

    @Override
    protected void updateExistUserInfo(Long userId, WechatBindMobileOneKeyLoginParam loginParam) {
        UserInfoEntity updateUserInfo = new UserInfoEntity();
        String newHeadImg = userExtendService.uploadWechatImage(loginParam.getHeadImageUrl(), loginParam, userId);

        updateUserInfo.setId(userId);
        updateUserInfo.setHeadImg(newHeadImg);
        updateUserInfo.setNickname(loginParam.getNickName());
        updateUserInfo.setSex(loginParam.getSex());

        userInfoService.updateUserInfo(updateUserInfo);
    }

}