package com.bxm.egg.user.support.impl;

import com.bxm.egg.common.enums.UserTypeEnum;
import com.bxm.egg.user.attribute.UserTagService;
import com.bxm.egg.user.attribute.UserVisitService;
import com.bxm.egg.user.constant.DomainScene;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.enums.WarmValueEquityEnum;
import com.bxm.egg.user.follow.UserFollowService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserInformationService;
import com.bxm.egg.user.info.UserStatisticsService;
import com.bxm.egg.user.integration.DomainIntegrationService;
import com.bxm.egg.user.integration.PushMsgIntegrationService;
import com.bxm.egg.user.location.UserLocationService;
import com.bxm.egg.user.mapper.UserLoginHistoryMapper;
import com.bxm.egg.user.medal.UserMedalService;
import com.bxm.egg.user.model.UserEquityDTO;
import com.bxm.egg.user.model.dto.*;
import com.bxm.egg.user.model.dto.warmlevel.OneselfHomePageCoverDTO;
import com.bxm.egg.user.model.dto.warmlevel.UserWarmLevelInfoDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.UserInformationEntity;
import com.bxm.egg.user.model.entity.UserLocationEntity;
import com.bxm.egg.user.model.entity.UserStatisticsEntity;
import com.bxm.egg.user.model.vo.UserTag;
import com.bxm.egg.user.properties.HomePageConfigProperties;
import com.bxm.egg.user.properties.NativeUserProperties;
import com.bxm.egg.user.support.HomePageService;
import com.bxm.egg.user.utils.ConstellationUtils;
import com.bxm.egg.user.warmlevel.UserWarmLevelService;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.time.DateFormatUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.time.LocalDate;
import java.time.ZoneId;
import java.util.*;
import java.util.concurrent.ThreadLocalRandom;
import java.util.stream.Collectors;

import static java.lang.Boolean.TRUE;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

@Slf4j
@Service
@AllArgsConstructor(onConstructor_ = {@Autowired})
public class HomePageServiceImpl implements HomePageService {

    private final String RESOURCE = "homepage:systemmessage:lock:";

    private final HomePageConfigProperties homePageConfigProperties;

    private final RedisStringAdapter redisStringAdapter;

    private final UserLoginHistoryMapper userLoginHistoryMapper;

    private final PushMsgIntegrationService pushMsgIntegrationService;

    private final DistributedLock distributedLock;

    private final UserTagService userTagService;

    private final UserVisitService userVisitService;

    private final NativeUserProperties nativeUserProperties;

    private final DomainIntegrationService domainIntegrationService;

    private final UserWarmLevelService userWarmLevelService;

    private final UserMedalService userMedalService;

    private final UserInfoService userInfoService;

    private final UserInformationService userInformationService;

    private final UserStatisticsService userStatisticsService;

    private final UserLocationService userLocationService;

    @Override
    public OneselfHomePageDTO getOneselfHomePage(Long userId) {
        OneselfHomePageDTO result = new OneselfHomePageDTO();
        BaseHomePage baseHomePage = this.getBaseHomePage(userId, userId);
        BeanUtils.copyProperties(baseHomePage, result);

        //用户最近访问的头像
        result.setInterviewHeadImgs(userVisitService.listRecentVisitorHeadImg(userId));
        //今天增加的访问数量
        result.setTodayInterviewNum(userVisitService.getTodayVisitorNum(userId));
        //总访问量
        result.setAllInterviewNum(userVisitService.getInterviewNumByUserId(userId));

        //3.12.0 获取最近获得的三个勋章
        result.setMedalMainPageSimpleDTO(userMedalService.getMainPageMedalDetail(userId));
        //3.12.0 用户更改封面信息
        result.setCoverDTO(builderUserCoverInfo(userId, result));

        return result;
    }


    private OneselfHomePageCoverDTO builderUserCoverInfo(Long userId, OneselfHomePageDTO result) {
        OneselfHomePageCoverDTO oneselfHomePageCoverDTO = new OneselfHomePageCoverDTO();
        UserEquityDTO userWarmLevelDTO = result.getUserWarmLevelDTO();

        String[] equityType = userWarmLevelDTO.getEquityType();

        List<String> equityList = Collections.EMPTY_LIST;
        if (equityType != null) {
            equityList = Arrays.asList(equityType);
        }
        if (equityList.contains(WarmValueEquityEnum.USER_COVER.name())) {
            oneselfHomePageCoverDTO.setCustomMadeCoverLevel(true);
            UserCoverDTO userCoverDTO = redisStringAdapter.get(userCoverCache(userId), UserCoverDTO.class);
            oneselfHomePageCoverDTO.setShowCustomMadeCover(Objects.isNull(userCoverDTO));
            return oneselfHomePageCoverDTO;
        }
        oneselfHomePageCoverDTO.setCustomMadeCoverLevel(false);
        return oneselfHomePageCoverDTO;
    }

    @Override
    public CustomerHomePageDTO getCustomerHomePage(Long targetUserId, Long userId) {

        CustomerHomePageDTO result = new CustomerHomePageDTO();
        BaseHomePage baseHomePage = this.getBaseHomePage(userId, targetUserId);
        BeanUtils.copyProperties(baseHomePage, result);
        //判断是否关注
        result.setIsAttention(SpringContextHolder.getBean(UserFollowService.class).isFollowed(userId, targetUserId));
        //调用访客接口
        userVisitService.visit(userId, targetUserId);
        //返回是否展示,如果是马甲号则不展示
        baseHomePage.setShowFarm(Objects.equals(baseHomePage.getType(), UserTypeEnum.VIRTUAL.getCode()));

        //3.12.0 获取最近获得的三个勋章
        result.setMedalMainPageSimpleDTO(userMedalService.getMainPageMedalDetail(targetUserId));
        return result;
    }

    /**
     * 获取个人主页基本信息
     */
    private BaseHomePage getBaseHomePage(Long currUserId, Long targetUserId) {

        //获取用户信息
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(targetUserId);
        //获取用户资料
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(targetUserId);
        //获取用户统计信息
        UserStatisticsEntity userStatisticsEntity = userStatisticsService.selectUserStatisticsByUserId(targetUserId);
        //获取地区信息
        UserLocationEntity userLocationEntity = userLocationService.getUserLocationByUserId(targetUserId);
        BaseHomePage baseHomePage = new BaseHomePage();

        if (Objects.nonNull(userInfoEntity)) {
            BeanUtils.copyProperties(userInfoEntity, baseHomePage);
            BeanUtils.copyProperties(userInformationEntity, baseHomePage);
            BeanUtils.copyProperties(userStatisticsEntity, baseHomePage);
            BeanUtils.copyProperties(userLocationEntity, baseHomePage);
            //拷贝属性时id会被覆盖，重新赋值一下
            baseHomePage.setId(targetUserId);

            //填充默认个人简介
            if (Objects.equals(userInformationEntity.getIsDefaultPersonalProfile(), TRUE)) {
                baseHomePage.setPersonalProfile(nativeUserProperties.getDefaultPersonalProfile());
            }

            //处理星座年龄
            if (null == userInformationEntity.getBirthday()) {
                baseHomePage.setAge(null);
            } else {
                LocalDate birthday = userInformationEntity.getBirthday().toInstant().atZone(ZoneId.systemDefault()).toLocalDate();
                if (birthday.isAfter(LocalDate.now())) {
                    baseHomePage.setAge(null);
                } else {
                    baseHomePage.setConstellation(ConstellationUtils.calculateConstellation(birthday));
                    baseHomePage.setBirthday(birthday);
                    baseHomePage.setGeneration(getGeneration(birthday));
                    baseHomePage.setAge(LocalDate.now().getYear() - birthday.getYear() + 1);
                }
            }
            String backgroundImg = null;
            UserCoverDTO userCoverDTO = redisStringAdapter.get(userCoverCache(targetUserId), UserCoverDTO.class);
            //如果有审核通过的封面
            if (Objects.nonNull(userCoverDTO)) {
                backgroundImg = userCoverDTO.getAuditCover();
            }
            //如果是自己看自己
            if (Objects.equals(currUserId, targetUserId)) {
                //优先展示用户提交审核的
                if (Objects.nonNull(userCoverDTO) && StringUtils.isNotBlank(userCoverDTO.getNoAuditCover())) {
                    backgroundImg = userCoverDTO.getNoAuditCover();
                }
            }
            //用户自己的封面不存在
            if (StringUtils.isBlank(backgroundImg)) {
                // 图片不存在随机选择背景图片,并保存到数据库
                if (StringUtils.isBlank(baseHomePage.getBackgroundImgUrl())) {
                    List<String> urls = homePageConfigProperties.getDefaultBackgroundImgUrls();
                    int index = ThreadLocalRandom.current().nextInt(homePageConfigProperties.getDefaultBackgroundImgUrls().size());
                    String backgroundImgUrl = urls.get(index);
                    userInformationService.updateBackgroundImgUrlByUserId(targetUserId, backgroundImgUrl);
                    baseHomePage.setBackgroundImgUrl(backgroundImgUrl);
                }
            } else {
                baseHomePage.setBackgroundImgUrl(backgroundImg);
            }
            //设置分享副标题
            if (Objects.equals(userInformationEntity.getIsDefaultPersonalProfile(), TRUE)) {
                baseHomePage.setShareSubTitle(homePageConfigProperties.getShareSubTitle());
            } else {
                baseHomePage.setShareSubTitle(baseHomePage.getPersonalProfile());
            }

            //{nickName} 分享标题中用户昵称的占位符
            if (isNotBlank(homePageConfigProperties.getShareTitle()) && isNotBlank(userInfoEntity.getNickname())) {
                baseHomePage.setShareTitle(homePageConfigProperties.getShareTitle().replace("{nickName}", userInfoEntity.getNickname()));
            }

            // 分享链接 域名服务获取base url 进行拼接
            String baseUrl = domainIntegrationService.getOutSideShareBaseUrl(DomainScene.DomainViewScene.CONTENT_VIEW);
            baseHomePage.setShareUrl(StringUtils.join(baseUrl, "/shareUserHome.html"));

            //填充兴趣标签列表
            List<UserTag> userTagList = userTagService.getUserTagFromRedisDb(targetUserId);
            baseHomePage.setUserTagList(
                    userTagList.stream().filter(e -> e.getDeleteFlag() == 0).map(UserTag::getLabel).collect(Collectors.toList()));

            UserWarmLevelInfoDTO userWarmInfoCache = userWarmLevelService.getUserWarmInfoCache(targetUserId);
            UserEquityDTO equityDTO = userWarmInfoCache.getEquityDTO();
            baseHomePage.setUserWarmLevelDTO(equityDTO);
        }
        return baseHomePage;
    }


    @Override
    public Boolean sendSystemMessage(Long userId) {
        KeyGenerator keyGenerator = RedisConfig.USER_ACCESS_MESSAGE.copy().appendKey(userId);
        String lockkey = RESOURCE + userId;
        //分布式锁,接口幂等
        if (distributedLock.lock(lockkey)) {
            try {
                Long timestamp = redisStringAdapter.get(keyGenerator, Long.class);
                boolean flag = Objects.isNull(timestamp)
                        || !new Date(timestamp).toInstant().atZone(ZoneId.systemDefault()).toLocalDate().equals(LocalDate.now());
                //执行发系统消息逻辑
                if (flag) {
                    Date sendDate;
                    if (Objects.isNull(timestamp)) {
                        sendDate = userLoginHistoryMapper.selectMinLoginTime(userId);
                    } else {
                        sendDate = new Date(timestamp);
                    }
                    int count = userVisitService.getInterviewNumByTime(userId, sendDate);
                    if (count > 0) {
                        log.debug("homepage systemmessage ,userId={},sendTime={}", userId,
                                DateFormatUtils.format(new Date(), DateUtils.DATE_TIME_FORMAT));
//                        LocationUserInfoDTO location = userService.getLocationUserInfo(userId);
                        LocationUserInfoDTO location = null;
                        pushMsgIntegrationService.sendSystemMessage(userId, count, location);
                        redisStringAdapter.set(keyGenerator, System.currentTimeMillis());
                    }
                }
            } finally {
                distributedLock.unlock(lockkey);
            }
        }
        return TRUE;
    }

    private String getGeneration(LocalDate birthday) {
        String generationStr = null;
        if (null != birthday) {
            int year = birthday.getYear();
            int decade = year / 10;
            String s = String.valueOf(decade);
            generationStr = s.substring(s.length() - 1) + "0后";
        }
        return generationStr;
    }


    private KeyGenerator userCoverCache(Long userId) {
        return RedisConfig.USER_COVER_INFO.copy().appendKey(userId);
    }
}
