package com.bxm.egg.user.timer;

import com.bxm.component.elasticjob.job.ISimplifyElasticJob;
import com.bxm.egg.mq.common.param.DingtalkMessage;
import com.bxm.egg.user.dto.LocationDTO;
import com.bxm.egg.user.info.UserRegCounterService;
import com.bxm.egg.user.integration.LocationIntegrationService;
import com.bxm.egg.user.integration.MessageFacadeIntegrationService;
import com.bxm.egg.user.model.vo.UserStatisticsBean;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.Interval;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.tools.inner.IntervalMap;
import lombok.extern.slf4j.Slf4j;
import org.apache.shardingsphere.elasticjob.api.ShardingContext;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.util.List;

/**
 * 创建即发布的周期定时任务
 */
@Component
@Slf4j
public class RegStatisticsTask implements ISimplifyElasticJob {

    @Resource
    private UserRegCounterService userRegCounterService;

    @Resource
    private MessageFacadeIntegrationService messageFacadeIntegrationService;

    @Resource
    private LocationIntegrationService locationIntegrationService;

    @Override
    public void execute(ShardingContext shardingContext) {
        IntervalMap intervalMap = DateUtils.getInterval(Interval.YESTERDAY);
        List<UserStatisticsBean> regUserList = userRegCounterService.regUserCounter(intervalMap.getStartTime(), intervalMap.getEndTime());
        List<UserStatisticsBean> migrationList = userRegCounterService.migrationUserCounter(intervalMap.getStartTime(), intervalMap.getEndTime());

        for (UserStatisticsBean userStatisticsBean : regUserList) {
            handleStatisticsData(userStatisticsBean);
        }

        for (UserStatisticsBean userStatisticsBean : migrationList) {
            handleStatisticsData(userStatisticsBean);
        }
        // 各个区域新增用户
        String statisticsDate = DateUtils.formatDate(intervalMap.getStartTime());
        StringBuilder content = new StringBuilder();
        content.append(statisticsDate).append(" 各地区注册用户汇总：\n");
        int sum = 0;

        for (UserStatisticsBean areaGroup : regUserList) {
            sum += areaGroup.getTotal();
            content.append(areaGroup.getLocationName()).append(" : ").append(areaGroup.getTotal()).append("\n");
        }
        content.append("共：[ ").append(sum).append(" ]人\n");

        content.append(statisticsDate).append(" 用户迁移情况（从蛋蛋佳到新版本）：\n");
        sum = 0;

        for (UserStatisticsBean areaGroup : migrationList) {
            sum += areaGroup.getTotal();
            content.append(areaGroup.getLocationName()).append(" : ").append(areaGroup.getTotal()).append("\n");
        }
        content.append("共：[ ").append(sum).append(" ]人\n");

        DingtalkMessage message = new DingtalkMessage();
        message.setScene("reg");
        message.setContent(content.toString());

        messageFacadeIntegrationService.sendDingtalk(message);
    }

    /**
     * 处理统计数据
     *
     * @param userStatisticsBean 处理数据
     */
    private void handleStatisticsData(UserStatisticsBean userStatisticsBean) {
        if (StringUtils.isNotBlank(userStatisticsBean.getLocationCode())) {
            LocationDTO locationByGeocode = locationIntegrationService.getLocationByGeocode(userStatisticsBean.getLocationCode());
            userStatisticsBean.setLocationName(locationByGeocode.getName());
        } else {
            userStatisticsBean.setLocationName("所属地区为空");
            userStatisticsBean.setLocationCode("-1");
        }
    }

    @Override
    public String cron() {
        return "0 0 9 * * ?";
    }

    @Override
    public String description() {
        return "每日早上9点统计注册人数和迁移人数";
    }
}
