package com.bxm.egg.user.info.impl;

import cn.hutool.core.date.DateUtil;
import cn.hutool.core.date.TimeInterval;
import com.alibaba.fastjson.JSON;
import com.bxm.egg.common.enums.UserStatusEnum;
import com.bxm.egg.user.constant.RedisConfig;
import com.bxm.egg.user.equitylevelmedal.UserEquityLevelMedalService;
import com.bxm.egg.user.info.UserInfoCacheService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserInformationService;
import com.bxm.egg.user.location.UserLocationService;
import com.bxm.egg.user.medal.UserMedalService;
import com.bxm.egg.user.model.bo.UserCacheInfoBO;
import com.bxm.egg.user.model.bo.UserInformationBO;
import com.bxm.egg.user.model.bo.UserLocationBO;
import com.bxm.egg.user.model.dto.UserBaseInfoDTO;
import com.bxm.egg.user.model.dto.info.UserBriefInfoDTO;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.entity.UserInformationEntity;
import com.bxm.egg.user.model.entity.UserLocationEntity;
import com.bxm.egg.user.model.equitylevelmedal.UserEquityLevelMedalInfoDTO;
import com.bxm.egg.user.properties.HomePageConfigProperties;
import com.bxm.egg.user.properties.UserProperties;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.sync.core.CacheHolder;
import com.bxm.newidea.component.sync.core.SyncCacheAgent;
import com.bxm.newidea.component.sync.core.SyncCacheHolderFactory;
import com.google.common.collect.Lists;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListeningExecutorService;
import com.google.common.util.concurrent.MoreExecutors;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collection;
import java.util.List;
import java.util.concurrent.ExecutorService;

import static com.bxm.egg.user.constant.MemoryCacheKeyConfig.MEMORY_CACHE_KEY;
import static com.bxm.egg.user.constant.MemoryCacheKeyConfig.MEMORY_USER_BRIRF_CACHE_KEY;
import static org.apache.commons.lang3.StringUtils.isNotEmpty;
import static org.springframework.util.CollectionUtils.isEmpty;

/**
 * 用户信息缓存service类
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/17 3:00 下午
 */
@Slf4j
@Service
public class UserInfoCacheServiceImpl implements UserInfoCacheService {

    @Resource
    private UserInfoService userInfoService;

    @Resource
    private UserLocationService userLocationService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private UserInformationService userInformationService;

    @Resource
    private UserEquityLevelMedalService userEquityLevelMedalService;

    @Resource
    private UserProperties userProperties;

    @Resource
    private HomePageConfigProperties homePageConfigProperties;

    @Resource
    private CacheHolder cacheHolder;

    @Resource
    private ExecutorService executor;

    @Override
    public UserCacheInfoBO load(Long userId) {
        SyncCacheAgent<Long, UserCacheInfoBO> cacheAgent = SyncCacheHolderFactory.build(MEMORY_CACHE_KEY, this::loadUserToMemory);
        return cacheAgent.get(userId);
    }

    @Override
    public List<UserCacheInfoBO> getBatchUserInfo(Collection<Long> userIds) {
        List<UserCacheInfoBO> userInfos = Lists.newArrayList();

        if (!isEmpty(userIds)) {
            userIds.forEach(userId -> {
                UserCacheInfoBO userInfoBO = load(userId);
                if (null != userInfoBO) {
                    userInfos.add(userInfoBO);
                }
            });
        }
        return userInfos;
    }

    @Override
    public void clearUserCache(Long userId) {
        //清空二级缓存
        cacheHolder.sendEvictCmd(MEMORY_CACHE_KEY, userId);
        //清空redis缓存
        redisStringAdapter.remove(buildUserKey(userId));
    }

    /**
     * 获取用户信息通过Redis或者DB
     *
     * @param userId 用户id
     * @return 用户信息缓存对象
     */
    private UserCacheInfoBO loadUserToMemory(Long userId) {
        //先从redis里面拿
        UserCacheInfoBO userCacheInfo = getUserInfoFromRedis(userId);

        if (log.isDebugEnabled()) {
            log.debug("用户信息：{}", JSON.toJSONString(userCacheInfo));
        }

        //用户不存在或者已被注销
        if (null == userCacheInfo || userCacheInfo.getUserId() == -1) {
            return buildInvalidUserCacheInfo(userId);
        }

        return userCacheInfo;
    }


    /**
     * 用户信息查询从Redis缓存中获取
     *
     * @param userId 用户id
     * @return 用户缓存信息
     */
    private UserCacheInfoBO getUserInfoFromRedis(Long userId) {

        UserCacheInfoBO userCacheInfoBO = redisStringAdapter.get(buildUserKey(userId), UserCacheInfoBO.class);

        //如果存在则直接返回
        if (userCacheInfoBO != null && userCacheInfoBO.getV() >= UserCacheInfoBO.CACHE_VERSION) {
            return userCacheInfoBO;
        }
        //如果用户缓存信息为空，则查询数据库
        userCacheInfoBO = getUserInfoFromDB(userId);

        if (userCacheInfoBO == null) {
            //放一个无效防止缓存穿透
            UserCacheInfoBO invalidUserInfo = new UserCacheInfoBO();
            invalidUserInfo.setUserId(-1L);

            log.warn("加载不存在的用户,用户id:{}", userId);

            redisStringAdapter.set(buildUserKey(userId), invalidUserInfo, userProperties.getTempUserCacheExpiredSeconds());

            return invalidUserInfo;
        } else {
            redisStringAdapter.set(buildUserKey(userId), userCacheInfoBO, userProperties.getUserCacheExpiredSeconds());
        }

        return userCacheInfoBO;

    }

    /**
     * 获取用户缓存信息从DB
     *
     * @param userId 用户id
     * @return 用户缓存信息
     */
    private UserCacheInfoBO getUserInfoFromDB(Long userId) {
        UserInfoEntity userInfoEntity = userInfoService.selectAllUserById(userId);

        if (userInfoEntity == null) {
            return null;
        }
        TimeInterval timer = DateUtil.timer();
        UserCacheInfoBO userCacheInfoBO = new UserCacheInfoBO();

        userCacheInfoBO.setUserId(userId);

        timer.restart();
        ListeningExecutorService listeningExecutorService = MoreExecutors.listeningDecorator(executor);

        fillUserInfo(userCacheInfoBO, userInfoEntity);
        //填充用户资料信息
        ListenableFuture<UserCacheInfoBO> fillUserInformationInfo = listeningExecutorService.submit(() -> fillUserInformationInfo(userCacheInfoBO));
        //填充用户定位信息
        ListenableFuture<UserCacheInfoBO> fillUserLocationInfo = listeningExecutorService.submit(() -> fillUserLocationInfo(userCacheInfoBO));
        //填充用户的勋章信息和等级信息
        ListenableFuture<UserCacheInfoBO> fillUserMedalInfoAndLevel = listeningExecutorService.submit(() -> fillUserMedalInfoAndLevel(userCacheInfoBO));

        ListenableFuture<List<UserCacheInfoBO>> listListenableFuture = Futures.successfulAsList(fillUserInformationInfo,
                fillUserLocationInfo,
                fillUserMedalInfoAndLevel);
        try {
            List<UserCacheInfoBO> userInfo = listListenableFuture.get();
            log.debug("异步处理,花费时间：{}", timer.intervalMs());
            timer.restart();
            userInfo.forEach(userBaseInfo -> {
                if (userBaseInfo.getUserInformationBO() != null) {
                    userCacheInfoBO.setUserInformationBO(userBaseInfo.getUserInformationBO());
                }
                if (userBaseInfo.getUserLocationBO() != null) {
                    userCacheInfoBO.setUserLocationBO(userBaseInfo.getUserLocationBO());
                }
                if (userBaseInfo.getUserWarmLevelDTO() != null) {
                    userCacheInfoBO.setUserWarmLevelDTO(userBaseInfo.getUserWarmLevelDTO());
                }
                if (userBaseInfo.getWearMedalList() != null) {
                    userCacheInfoBO.setWearMedalList(userBaseInfo.getWearMedalList());
                }
            });
            log.debug("数据拼装处理,花费时间：{}", timer.intervalMs());
        } catch (Exception e) {
            log.error("");
        }
        return userCacheInfoBO;

    }

    /**
     * 填充用户的基本信息
     *
     * @param userCacheInfoBO 用户缓存对象
     * @param userInfoEntity  用户基本信息实体类
     * @ 用户缓存信息
     */
    private void fillUserInfo(UserCacheInfoBO userCacheInfoBO, UserInfoEntity userInfoEntity) {
        BeanUtils.copyProperties(userInfoEntity, userCacheInfoBO);

        userCacheInfoBO.setUserId(userInfoEntity.getId());
    }


    /**
     * 填充用户的勋章和等级信息
     *
     * @param userCacheInfoBO 用户缓存对象
     */
    private UserCacheInfoBO fillUserMedalInfoAndLevel(UserCacheInfoBO userCacheInfoBO) {
        UserEquityLevelMedalInfoDTO equityLevelMedalInfo = userEquityLevelMedalService.getEquityLevelMedalInfo(userCacheInfoBO.getUserId());

        //如果不存在则直接返回
        if (equityLevelMedalInfo == null) {
            return userCacheInfoBO;
        }
        userCacheInfoBO.setWearMedalList(equityLevelMedalInfo.getWearMedalDTOList());
        userCacheInfoBO.setUserWarmLevelDTO(equityLevelMedalInfo.getUserEquityDTO());
        return userCacheInfoBO;
    }

    /**
     * 填充用户的定位信息
     *
     * @param userCacheInfoBO 用户缓存对象
     */
    private UserCacheInfoBO fillUserLocationInfo(UserCacheInfoBO userCacheInfoBO) {
        UserLocationEntity userLocationEntity = userLocationService.getUserLocationByUserId(userCacheInfoBO.getUserId());

        //如果不存在则直接返回
        if (userLocationEntity == null) {
            return userCacheInfoBO;
        }
        UserLocationBO userLocationBO = new UserLocationBO();

        userLocationBO.setLocationCode(userLocationEntity.getLocationCode());
        userLocationBO.setLocationName(userLocationEntity.getLocationName());
        userCacheInfoBO.setUserLocationBO(userLocationBO);
        return userCacheInfoBO;
    }

    /**
     * 填充用户的资料信息
     *
     * @param userCacheInfoBO 用户缓存对象
     */
    private UserCacheInfoBO fillUserInformationInfo(UserCacheInfoBO userCacheInfoBO) {
        UserInformationEntity userInformationEntity = userInformationService.getUserInformationByUserId(userCacheInfoBO.getUserId());

        //如果不存在则直接返回
        if (userInformationEntity == null) {
            return userCacheInfoBO;
        }

        UserInformationBO userInformationBO = new UserInformationBO();
        userInformationBO.setPersonalProfile(userInformationEntity.getPersonalProfile());
        userInformationBO.setBirthday(userInformationEntity.getBirthday());

        //设置默认个人简介
        if (isNotEmpty(userInformationBO.getPersonalProfile())) {
            userInformationBO.setIsDefaultPersonalProfile(Boolean.FALSE);
        } else {
            userInformationBO.setIsDefaultPersonalProfile(Boolean.TRUE);
            userInformationBO.setPersonalProfile(homePageConfigProperties.getDefaultPersonalProfile());
        }

        userCacheInfoBO.setUserInformationBO(userInformationBO);

        if (log.isDebugEnabled()) {
            log.debug("用户资料信息：{}", JSON.toJSONString(userInformationBO));
        }
        return userCacheInfoBO;
    }

    /**
     * 构造一个失效的用户缓存信息
     *
     * @param userId 用户id
     * @return 用户信息
     */
    private UserCacheInfoBO buildInvalidUserCacheInfo(Long userId) {
        UserCacheInfoBO userCacheInfoBO = new UserCacheInfoBO();

        userCacheInfoBO.setHeadImg(userProperties.getDefaultHeadImageUrl());
        userCacheInfoBO.setUserId(userId);
        userCacheInfoBO.setNickname("未知用户");
        userCacheInfoBO.setState(UserStatusEnum.DISABLE.getCode());

        return userCacheInfoBO;
    }


    private KeyGenerator buildUserKey(Long userId) {
        return RedisConfig.USER_INFO_CACHE_KEY.copy().appendKey(userId);
    }


    @Override
    public UserBriefInfoDTO loadBriefInfo(Long userId) {
        SyncCacheAgent<Long, UserBriefInfoDTO> cacheAgent = SyncCacheHolderFactory.build(MEMORY_USER_BRIRF_CACHE_KEY, this::loadUserBriefInfo);
        return cacheAgent.get(userId);
    }


    private KeyGenerator buildUserBriefKey(Long userId) {
        return RedisConfig.USER_BRIEF_INFO_CACHE_KEY.copy().appendKey(userId);
    }


    private UserBriefInfoDTO loadUserBriefInfo(Long userId) {
        //先从redis里面拿
        UserBriefInfoDTO userCacheInfo = getUserBriefInfoFromRedis(userId);

        if (log.isDebugEnabled()) {
            log.debug("用户信息：{}", JSON.toJSONString(userCacheInfo));
        }

        //用户不存在或者已被注销
        if (userCacheInfo.getUserId() == -1) {
            UserBriefInfoDTO userBriefInfoDTO = new UserBriefInfoDTO();
            userBriefInfoDTO.setNickname("未知用户");
            userBriefInfoDTO.setUserId(userId);
            return userBriefInfoDTO;
        }

        return userCacheInfo;
    }

    private UserBriefInfoDTO getUserBriefInfoFromRedis(Long userId) {
        UserBriefInfoDTO userCacheInfoBO = redisStringAdapter.get(buildUserBriefKey(userId), UserBriefInfoDTO.class);

        //如果存在则直接返回
        if (userCacheInfoBO != null) {
            return userCacheInfoBO;
        }
        //如果用户缓存信息为空，则查询数据库
        UserInfoEntity userInfoEntity = userInfoService.selectUserById(userId);
        if (userInfoEntity == null) {
            //放一个无效防止缓存穿透
            UserBriefInfoDTO invalidUserInfo = new UserBriefInfoDTO();
            invalidUserInfo.setUserId(-1L);
            log.warn("加载不存在的用户,用户id:{}", userId);
            redisStringAdapter.set(buildUserBriefKey(userId), invalidUserInfo, userProperties.getTempUserCacheExpiredSeconds());
            return invalidUserInfo;
        } else {
            userCacheInfoBO = new UserBriefInfoDTO();
            userCacheInfoBO.setUserId(userInfoEntity.getId());
            userCacheInfoBO.setHeadImg(userInfoEntity.getHeadImg());
            userCacheInfoBO.setNickname(userInfoEntity.getNickname());
            userCacheInfoBO.setState(userInfoEntity.getState());
            redisStringAdapter.set(buildUserBriefKey(userId), userCacheInfoBO, userProperties.getUserCacheExpiredSeconds());
        }

        return userCacheInfoBO;

    }
}