package com.bxm.egg.user.info.impl;

import cn.hutool.core.collection.CollectionUtil;
import com.baomidou.mybatisplus.core.conditions.query.LambdaQueryWrapper;
import com.bxm.egg.user.constant.BaseDistributeKeyConstant;
import com.bxm.egg.user.constant.UserBizConfigProperties;
import com.bxm.egg.user.convert.EggSixParamConverter;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoCacheService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserWechatGrantService;
import com.bxm.egg.user.info.msg.UserInfoChangeSender;
import com.bxm.egg.user.integration.EggSixEnjoyIntegrationService;
import com.bxm.egg.user.integration.SixEnjoyUserIntegrationService;
import com.bxm.egg.user.mapper.UserGrantWechatAppMapper;
import com.bxm.egg.user.model.bo.UserCacheInfoBO;
import com.bxm.egg.user.model.bo.UserWechatBindBO;
import com.bxm.egg.user.model.dto.info.UserBindWxDTO;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.UserBindWxParam;
import com.bxm.egg.user.reward.UserRewardService;
import com.bxm.newidea.component.JSON;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.config.SequenceHolder;
import com.bxm.sync.facade.dto.UserRewardDTO;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

/**
 * @author wzy
 * @version 1.0
 * @date 2021/10/8 5:50 下午
 */
@Slf4j
@Service
@AllArgsConstructor
public class UserWechatGrantServiceImpl implements UserWechatGrantService {

    private final UserGrantWechatAppMapper userGrantWechatAppMapper;

    private final DistributedLock distributedLock;

    private final UserInfoService userInfoService;

    private final UserInfoCacheService userInfoCacheService;

    private final UserInfoChangeSender userInfoChangeSender;

    private final UserExtendService userExtendService;

    private final EggSixEnjoyIntegrationService eggSixEnjoyIntegrationService;

    private final SixEnjoyUserIntegrationService enjoyUserIntegrationService;

    private final UserBizConfigProperties userBizConfigProperties;

    private final UserRewardService userRewardService;

    @Override
    public Optional<UserGrantWechatAppEntity> userInfoUnionIdExist(String unionId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUnionId, unionId);
        queryWrapper.eq(UserGrantWechatAppEntity::getDeleted, 0);
        queryWrapper.orderByDesc(UserGrantWechatAppEntity::getCreateTime);

        List<UserGrantWechatAppEntity> userGrantWechatAppEntity = userGrantWechatAppMapper.selectList(queryWrapper);

        if (CollectionUtil.isNotEmpty(userGrantWechatAppEntity)) {
            return Optional.of(userGrantWechatAppEntity.get(0));
        }

        return Optional.empty();
    }

    @Override
    public UserGrantWechatAppEntity getUserWxInfoByUserId(Long userId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUserId, userId);
        queryWrapper.eq(UserGrantWechatAppEntity::getDeleted, 0);
        queryWrapper.orderByDesc(UserGrantWechatAppEntity::getCreateTime);
        List<UserGrantWechatAppEntity> userGrantWechatAppEntity = userGrantWechatAppMapper.selectList(queryWrapper);

        if (CollectionUtil.isNotEmpty(userGrantWechatAppEntity)) {
            return userGrantWechatAppEntity.get(0);
        }

        return null;
    }

    @Override
    public int updateWechatGrantBindUserId(Long userId, String unionId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUnionId, unionId);

        UserGrantWechatAppEntity updateEntity = new UserGrantWechatAppEntity();

        updateEntity.setUserId(userId);

        return userGrantWechatAppMapper.update(updateEntity, queryWrapper);

    }

    @Override
    public Message userBindWx(UserBindWxParam userBindWxParam) {

        String resource = buildBindWxUserLockKey(userBindWxParam.getUserId());

        if (!distributedLock.lock(resource)) {
            log.info("出现重复点击，请求参数：{}", JSON.toJSONString(userBindWxParam));
            return null;
        }

        //判断用户是否已经绑定微信
        if (userHasBindWx(userBindWxParam.getUserId())) {
            distributedLock.unlock(resource);
            return Message.build(false, "该用户已经绑定过微信，请勿重复绑定");
        }

        //判断用户是否已经绑定微信
        Optional<UserGrantWechatAppEntity> userGrantWechatAppEntity = userInfoUnionIdExist(userBindWxParam.getUnionId());
        if (userGrantWechatAppEntity.isPresent()) {
            UserGrantWechatAppEntity grantWechatAppEntity = userGrantWechatAppEntity.get();
            distributedLock.unlock(resource);
            return Message.build(false, "该绑定过微信已绑定" + grantWechatAppEntity.getUserId() + "，请勿重复绑定");
        }


        UserWechatBindBO userWechatBindBO = new UserWechatBindBO();
        BeanUtils.copyProperties(userBindWxParam, userWechatBindBO);
        //添加微信授权记录
        grantBindWxRecord(userWechatBindBO);


        //微信授权后更新用户头像等信息
        updateUserWxBindRelationInfo(userBindWxParam);

        //清空用户信息缓存
        userInfoChangeSender.sendUserChangeMsg(userBindWxParam.getUserId());

        //初始化资料完成度
        userExtendService.initComplete(userBindWxParam.getUserId());

        if (StringUtils.isGrateOrEqualThan(userBindWxParam.getCurVer(), "2.0.1")) {
            userBindWxHandler(userBindWxParam);
        } else {
            //调用六享微信绑定奖励
            eggSixEnjoyIntegrationService.bind(EggSixParamConverter.convert(userBindWxParam));
        }

        distributedLock.unlock(resource);
        return Message.build(true);
    }

    /**
     * 用户绑定微信后续操作
     *
     * @param userBindWxParam 用户绑定信息
     */
    @Override
    public void userBindWxHandler(UserBindWxParam userBindWxParam) {

        if (userBindWxParam.getPhone() == null) {
            UserCacheInfoBO load = userInfoCacheService.load(userBindWxParam.getUserId());
            userBindWxParam.setPhone(load.getPhone());
        }
        UserRewardDTO userRewardInfo = enjoyUserIntegrationService.getUserRewardInfo(userBindWxParam.getUserId(),
                userBindWxParam.getPhone(),
                userBindWxParam.getDevcId());

        if (userRewardInfo == null) {
            log.error("获取用户信息失败，请检验用户在六享是否存在：{}", userBindWxParam.getUserId());
            return;
        }
        if (Objects.equals(userRewardInfo.getRegisterReward(), Boolean.TRUE) &&
                Objects.equals(userRewardInfo.getWxReward(), Boolean.TRUE)) {
            log.debug("用户已发放过奖励：userBindWxParam：{},userRewardInfo:{}", JSON.toJSONString(userBindWxParam), JSON.toJSONString(userRewardInfo));
            return;
        }
        userRewardService.sendUserBindWxReward(userBindWxParam, userRewardInfo);
    }

    /**
     * 微信授权后更新用户头像等信息
     *
     * @param userBindWxParam 微信绑定入参
     */
    private void updateUserWxBindRelationInfo(UserBindWxParam userBindWxParam) {
        UserInfoEntity updateUserInfo = new UserInfoEntity();

        String newHeadImg = userExtendService.uploadWechatImage(userBindWxParam.getHeadImageUrl(),
                userBindWxParam.getUserId());

        userBindWxParam.setHeadImageUrl(newHeadImg);
        updateUserInfo.setId(userBindWxParam.getUserId());
        updateUserInfo.setHeadImg(newHeadImg);
        updateUserInfo.setNickname(userBindWxParam.getNickName());
        updateUserInfo.setSex(userBindWxParam.getSex());

        userInfoService.updateUserInfo(updateUserInfo);
    }

    /**
     * 判断用户是否已经绑定微信
     *
     * @param userId 用户id
     * @return 是否绑定
     */
    @Override
    public Boolean userHasBindWx(Long userId) {
        LambdaQueryWrapper<UserGrantWechatAppEntity> queryWrapper = new LambdaQueryWrapper<>();

        queryWrapper.eq(UserGrantWechatAppEntity::getUserId, userId);
        queryWrapper.eq(UserGrantWechatAppEntity::getDeleted, 0);

        List<UserGrantWechatAppEntity> grantWechatAppEntityList = userGrantWechatAppMapper.selectList(queryWrapper);

        return !CollectionUtils.isEmpty(grantWechatAppEntityList);
    }

    @Override
    public Boolean grantBindWxRecord(UserWechatBindBO userWechatBindBO) {
        // 保留微信授权信息
        UserGrantWechatAppEntity entity = new UserGrantWechatAppEntity();
        entity.setId(SequenceHolder.nextLongId());

        entity.setCountry(userWechatBindBO.getCountry());
        entity.setProvince(userWechatBindBO.getProvince());
        entity.setCity(userWechatBindBO.getCity());
        entity.setNickName(userWechatBindBO.getNickName());
        entity.setHeadImg(userWechatBindBO.getHeadImageUrl());
        entity.setOpenId(userWechatBindBO.getOpenId());
        entity.setUnionId(userWechatBindBO.getUnionId());
        entity.setSex(userWechatBindBO.getSex());

        entity.setUserId(userWechatBindBO.getUserId());
        entity.setCreateTime(new Date());

        return userGrantWechatAppMapper.insert(entity) > 0;
    }

    /**
     * 构建用户绑定微信分布式锁，保证不重复请求
     *
     * @param userId 用户id
     * @return 分布式锁key
     */
    private String buildBindWxUserLockKey(Long userId) {
        return BaseDistributeKeyConstant.USER_BIND_WX_LOCK_KEY.copy().appendKey(userId).gen();
    }

    @Override
    public UserBindWxDTO getUserHasBindWx(Long userId, Long devcId) {
        //如果用户id没穿过来，则给个默认值
        if (Objects.isNull(userId)) {
            return UserBindWxDTO.builder()
                    .isBind(false)
                    .foodsNum(userBizConfigProperties.getRegisterWxAndReward())
                    .build();
        }
        Boolean userHasBindWx = userHasBindWx(userId);
        if (userHasBindWx) {
            return UserBindWxDTO.builder()
                    .isBind(true)
                    .build();
        }

        UserRewardDTO userRewardInfo = enjoyUserIntegrationService.getUserRewardInfo(userId, null, devcId);
        return UserBindWxDTO.builder()
                .foodsNum(userRewardInfo != null && userRewardInfo.getRegisterReward() ?
                        userBizConfigProperties.getWxBindReward() : userBizConfigProperties.getRegisterWxAndReward())
                .isBind(false)
                .build();
    }
}