package com.bxm.egg.user.login.strategy.support;

import com.bxm.egg.mq.common.constant.SmsTemplateEnum;
import com.bxm.egg.user.constant.UserLoginTypeEnum;
import com.bxm.egg.user.convert.EggSixParamConverter;
import com.bxm.egg.user.info.UserExtendService;
import com.bxm.egg.user.info.UserInfoService;
import com.bxm.egg.user.info.UserWechatGrantService;
import com.bxm.egg.user.integration.EggSixEnjoyIntegrationService;
import com.bxm.egg.user.integration.SmsIntegrationService;
import com.bxm.egg.user.login.strategy.AbstractUserLoginStrategy;
import com.bxm.egg.user.model.bo.UserWechatBindBO;
import com.bxm.egg.user.model.dto.login.LoginResultDTO;
import com.bxm.egg.user.model.entity.UserGrantWechatAppEntity;
import com.bxm.egg.user.model.entity.UserInfoEntity;
import com.bxm.egg.user.model.param.UserBindWxParam;
import com.bxm.egg.user.model.param.login.WechatBindMobileLoginParam;
import com.bxm.newidea.component.bo.Message;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Objects;
import java.util.Optional;

/**
 * 微信绑定手机号登录策略
 *
 * @author wzy
 * @version 1.0
 * @date 2021/9/14 8:03 下午
 */
@Slf4j
@Service
public class WechatBindMobileLoginStrategy extends AbstractUserLoginStrategy<WechatBindMobileLoginParam> {
    @Resource
    private UserInfoService userInfoService;

    @Resource
    private SmsIntegrationService smsIntegrationService;

    @Resource
    private UserExtendService userExtendService;

    @Resource
    private UserWechatGrantService userWechatGrantService;

    @Resource
    private EggSixEnjoyIntegrationService eggSixEnjoyIntegrationService;

    @Override
    protected Message beforeLoginAuthentication(WechatBindMobileLoginParam loginParam) {

        //验证码是否正确
        if (!smsIntegrationService.verifyByType(SmsTemplateEnum.BINDING_PHONE, loginParam.getPhone(), loginParam.getSmsCode())) {
            return Message.build(false, "验证码错误");
        }

        //WX登录不上，才会跳转到这里登录，如果没授权，则客户端不会有此数据
        if (Objects.isNull(loginParam.getUnionId())) {
            return Message.build(false, "微信用户未授权");
        }

        return Message.build(true);
    }

    @Override
    protected Long checkExists(WechatBindMobileLoginParam loginParam) {
        //判断用户绑定的手机号是否已经存在
        Optional<Long> userPhoneExist = userInfoService.userPhoneExist(loginParam.getPhone());

        //判断手机号是否有绑定的用户，如果有则不用创建新用户直接登录，如果没有则要创建新用户
        if (!userPhoneExist.isPresent()) {
            return null;
        }
        loginParam.setUserId(userPhoneExist.get());
        return userPhoneExist.get();
    }

    @Override
    public UserLoginTypeEnum type() {
        return UserLoginTypeEnum.WECHAT_BIND_MOBILE_LOGIN;
    }

    @Override
    protected boolean saveIdentity(WechatBindMobileLoginParam loginParam, Long userId) {
        // 保留微信授权原始积累
        UserWechatBindBO userWechatBindBO = new UserWechatBindBO();
        BeanUtils.copyProperties(loginParam, userWechatBindBO);

        userWechatBindBO.setUserId(userId);

        return userWechatGrantService.grantBindWxRecord(userWechatBindBO);
    }

    @Override
    protected void fillUserInfo(WechatBindMobileLoginParam loginParam, UserInfoEntity newUser) {
        String newHeadImg = userExtendService.uploadWechatImage(loginParam.getHeadImageUrl(), newUser.getId());

        newUser.setHeadImg(newHeadImg);
        newUser.setNickname(loginParam.getNickName());
        newUser.setSex(loginParam.getSex());
    }

    @Override
    protected void updateExistUserInfo(Long userId, WechatBindMobileLoginParam loginParam) {
        UserInfoEntity updateUserInfo = new UserInfoEntity();
        String newHeadImg = userExtendService.uploadWechatImage(loginParam.getHeadImageUrl(), userId);

        updateUserInfo.setEquipmentId(loginParam.getDevcId());
        updateUserInfo.setId(userId);
        updateUserInfo.setHeadImg(newHeadImg);
        updateUserInfo.setNickname(loginParam.getNickName());
        updateUserInfo.setSex(loginParam.getSex());

        userInfoService.updateUserInfo(updateUserInfo);

        UserGrantWechatAppEntity userWxInfoByUserId = userWechatGrantService.getUserWxInfoByUserId(userId);

        //如果用户的微信不存在，则绑定微信信息
        if (Objects.isNull(userWxInfoByUserId)) {
            saveIdentity(loginParam, userId);
        }
    }
}