package com.bxm.localnews.activity.controller;

import com.bxm.localnews.activity.config.CommandRedPacketProperties;
import com.bxm.localnews.activity.dto.*;
import com.bxm.localnews.activity.param.ObtainHistoryParam;
import com.bxm.localnews.activity.param.RedPacketAddParam;
import com.bxm.localnews.activity.param.RedPacketUpdateParam;
import com.bxm.localnews.activity.param.WallRedPacketParam;
import com.bxm.localnews.activity.service.redpacket.command.CommandGradeService;
import com.bxm.localnews.activity.service.redpacket.command.RedPacketService;
import com.bxm.localnews.activity.vo.GradeResultBean;
import com.bxm.localnews.activity.vo.RedPacketTotalVo;
import com.bxm.localnews.activity.vo.WechatUserInfoVo;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.vo.User;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;

import java.util.List;

/**
 * <p>
 * 口令红包接口
 * </p>
 *
 * @Author: JandMin
 * @since: 1.0.0
 * @Date: 2019/4/17
 */
@Api(tags = "2-12 口令红包相关接口", description = "口令红包的相关操作接口操作")
@RestController
@RequestMapping("api/redpacket")
public class RedPacketController {
    private static final Logger LOG = LoggerFactory.getLogger(RedPacketController.class);
    @Autowired
    private RedPacketService redPacketService;
    @Autowired
    private CommandGradeService commandGradeService;

    @Autowired
    private CommandRedPacketProperties commandRedPacketProperties;

    @ApiOperation(value = "2-12-1 红包记录金额汇总接口", notes = "获取用户发红包记录和领红包记录的总金额和总记录数")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "owner", value = "拥有权限(1：我发的，2：我领的)", dataType = "int", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true)
    })
    @GetMapping("/record/total")
    public Json<RedPacketTotalVo> getTotalRecord(@RequestParam("owner") Integer owner,
                                                 @RequestParam("userId") Long userId) {
        return Json.build(redPacketService.getTotalRecordByUser(owner,userId));
    }

    @ApiOperation(value = "2-12-2 我发送的红包记录接口", notes = "获取用户发过的红包信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "current", value = "当前页", dataType = "int", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每页大小", dataType = "int", required = true)
    })
    @GetMapping("/record/sent")
    public Json<PageWarper<RedPacketSentDTO>> getSentRecord(@RequestParam("userId") Long userId,
                                                            @RequestParam("current")Integer current,
                                                            @RequestParam("pageSize")Integer pageSize) {
        return  Json.build(redPacketService.listSentRecord(userId,current,pageSize));
    }

    @ApiOperation(value = "2-12-3 我领取的红包记录接口", notes = "获取用户领取过的红包信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "current", value = "当前页", dataType = "int", required = true),
            @ApiImplicitParam(name = "pageSize", value = "每页大小", dataType = "int", required = true)
    })
    @GetMapping("/record/gotten")
    public Json<PageWarper<RedPacketGottenDTO>> getGottenRecord(@RequestParam("userId") Long userId,
                                                                @RequestParam("current")Integer current,
                                                                @RequestParam("pageSize")Integer pageSize) {
        return  Json.build(redPacketService.listGottenRecord(userId,current,pageSize));
    }

    @ApiOperation(value = "2-12-4 新增红包接口（返回红包id）", notes = "添加口令红包信息（返回红包id）")
    @PostMapping("/add")
    public Json<Long> addRedPacket(RedPacketAddParam redPacketAddParam) {
        return Json.build(redPacketService.insertRedPacket(redPacketAddParam));
    }

    @ApiOperation(value = "2-12-5 修改红包接口", notes = "修改口令红包信息")
    @PostMapping("/update")
    public Json<Boolean> updateRedPacket(RedPacketUpdateParam redPacketUpdateParam,Integer platform) {
        Boolean result = redPacketService.updateRedPacket(redPacketUpdateParam,platform);
        return  Json.build(result);
    }

    @ApiOperation(value = "2-12-6 获取单个红包修改信息接口", notes = "获取单个红包信息用于修改")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true)
    })
    @PostMapping("/get/rpfu")
    public Json<RedPacketUpdateDTO> getRedPacketById(@RequestParam("rpId") Long rpId,
                                                   @RequestParam("userId") Long userId) {
        RedPacketUpdateDTO result = redPacketService.getRedPacketById(rpId,userId);
        if(null == result){
            return Json.build(RespCode.BAD_REQUEST,"red packet not exist");
        }
        return Json.build(result);
    }

    @ApiOperation(value = "2-12-7 获取口令模板接口", notes = "获取所红包口令模板")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true)
    })
    @PostMapping("/get/content")
    public Json<List<RedPacketContentDTO>> getContentList(@RequestParam("userId") Long userId) {
        List<RedPacketContentDTO> result = redPacketService.listRedPacketContent(userId);
        if(null == result){
            return Json.build(RespCode.BAD_REQUEST,"the word of command is empty");
        }
        return Json.build(result);
    }

    @ApiOperation(value = "2-12-8 获取红包配置接口", notes = "获取可发送的红包金和和数量信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true)
    })
    @PostMapping("/get/item")
    public Json<List<RedPacketItemDTO>> getItemList(@RequestParam("userId") Long userId) {
        List<RedPacketItemDTO> result = redPacketService.listRedPacketItem(userId);
        if(null == result){
            return Json.build(RespCode.BAD_REQUEST,"the template is empty");
        }
        return Json.build(result);
    }

    @ApiOperation(value = "2-12-9 获取分享信息接口", notes = "获取用于分享的信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "type", value = "类型（1：未参与、2：已参与）", dataType = "int", required = true),
            @ApiImplicitParam(name = "nickname", value = "分享人昵称", dataType = "string", required = true)
    })
    @PostMapping("/get/share")
    public Json<RedPacketShareDTO> getShareInfo(@RequestParam("rpId") Long rpId,
                                                @RequestParam("userId") Long userId,
                                                @RequestParam("type") Integer type,
                                                @RequestParam("nickname") String nickname) {
        RedPacketShareDTO result = redPacketService.getShareInfo(rpId,userId,type, nickname);
        if(null == result){
            return Json.build(RespCode.BAD_REQUEST,"share info is empty");
        }
        return Json.build(result);
    }


    @ApiOperation(value = "2-12-11 红包墙-获取红包列表", notes = "获取当前区域的红包列表，根据用户筛选红包及设置红包状态")
    @ApiImplicitParams({
    })
    @GetMapping("/wall")
    public Json<PageWarper<WallRedPacketDTO>> getWall(WallRedPacketParam wallRedPacketParam) {
        return Json.build(redPacketService.getReadPacketWall(wallRedPacketParam));
    }

    @ApiOperation(value = "2-12-12 获取红包详情信息", notes = "获取红包详情信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true)
    })
    @GetMapping("/info")
    public Json<RedPacketDTO> getRedPacketInfo(@RequestParam("userId") Long userId, @RequestParam("rpId") Long rpId) {
        return Json.build(redPacketService.getReadPacketDetail(userId, rpId));
    }

    @ApiOperation(value = "2-12-13 获取红包的最近领取记录")
    @ApiImplicitParams({
    })
    @GetMapping("/obtainHistory")
    public Json<PageWarper<PacketHistoryDTO>> getObtainHistory(ObtainHistoryParam obtainHistoryParam) {
        return Json.build(redPacketService.getObtainHistory(obtainHistoryParam));
    }

    @ApiOperation(value = "2-12-14 获取语音评分排名")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true)
    })
    @GetMapping("/ranklist")
    public Json<List<PacketHistoryDTO>> getRankList(@RequestParam("rpId") Long rpId) {
        return Json.build(redPacketService.getRanking(rpId));
    }

    @ApiOperation(value = "2-12-15 口令评分")
    @ApiImplicitParams({
    })
    @PostMapping("/markScore")
    public Json<CommandResultDTO> markScore(MarkScoreDTO markScoreDTO,Integer platform) {
        return Json.build(redPacketService.markScore(markScoreDTO,platform));
    }

    @ApiOperation(value = "2-12-16 支付查询接口", notes = "查看支付是否成功")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "paymentNum", value = "订单编号 返回状态(1待付款，2取消付款，3付款成功，4付款失败，5失败超时)", dataType = "String", required = true)
    })
    @GetMapping("/query/order")
    public Json<Byte> queryOrder(@RequestParam("rpId") Long rpId, @RequestParam("userId") Long userId, @RequestParam("paymentNum") String paymentNum, BasicParam basicParam) {
        return Json.build(redPacketService.queryOrder(rpId,userId,paymentNum, basicParam.getPlatform()));
    }


    @ApiOperation(value = "2-12-17 评分测试接口(前端勿调)", notes = "评分测试接口(前端勿调)")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "content", value = "目前录音内容", dataType = "String", required = true),
            @ApiImplicitParam(name = "path", value = "文件地址", dataType = "String", required = true),
            @ApiImplicitParam(name = "len", value = "录音时长", dataType = "Integer", required = true)
    })
    @GetMapping("/test/record")
    public Json<GradeResultBean> testRecord(@RequestParam("content") String content, @RequestParam("path") String path, @RequestParam("len") Integer len) {
        return Json.build(commandGradeService.testRecord(content,path,len));
    }

    @ApiOperation(value = "2-12-18 口令评分（站外）")
    @ApiImplicitParams({
    })
    @PostMapping("/public/markScore")
    public Json<CommandResultDTO> publicMarkScore(MarkScorePublicDTO markScorePublicDTO,Integer platform) {
        return Json.build(redPacketService.markScore(markScorePublicDTO,platform));
    }
    @ApiOperation(value = "2-12-19 口令最大允许长度")
    @GetMapping("/commandMaxLength")
    public Json<Integer> commandMaxLength(){
        return Json.build(commandRedPacketProperties.getMaxLength());
    }

    @ApiOperation(value = "2-12-20 获取红包详情信息（站外）", notes = "获取红包详情信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "当前用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "shareUserId", value = "分享人id", dataType = "long", required = true),
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true)
    })
    @GetMapping("/public/info")
    public Json<RedPacketPublicDTO> getPublicRedPacketInfo(@RequestParam("userId") Long userId,@RequestParam("shareUserId") Long shareUserId, @RequestParam("rpId") Long rpId) {
        return Json.build(redPacketService.getPublicReadPacketDetail(userId,shareUserId, rpId));
    }

    @ApiOperation(value = "2-12-21 微信评分测试接口(前端勿调)", notes = "评分测试接口(前端勿调)")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "target", value = "语音文字内容", dataType = "String", required = true),
            @ApiImplicitParam(name = "mediaId", value = "微信媒体文件id", dataType = "String", required = true),
    })
    @GetMapping("/test/recordWx")
    public Json<ScoreDto> testRecordWx(@RequestParam("target") String target, @RequestParam("mediaId") String mediaId) {
        return Json.build(commandGradeService.grade(target,mediaId));
    }

    @ApiOperation(value = "2-12-22 当前区域是否为口令红包活动区域", notes = "根据红包所属区域与用户定位判断是否为红包活动区域")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", required = true),
            @ApiImplicitParam(name = "city", value = "市级信息", required = false),
            @ApiImplicitParam(name = "district", value = "区县级信息", required = false)
    })
    @GetMapping("/public/isActiveArea")
    public Json<Boolean> isActiveArea(@RequestParam(name = "rpId") Long rpId,
                                      @RequestParam(name = "city") String city,
                                      @RequestParam(name = "district") String district) {
        return ResultUtil.genSuccessResult(redPacketService.isActiveArea(rpId,city, district));
    }

    @ApiOperation(value = "2-12-23 红包墙-获取单个红包", notes = "通过用户id和红包id 获取该用户与该红包的状态，及红包的属性")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "当前用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true)
    })
    @GetMapping("/wall/single")
    public Json<WallRedPacketDTO> getSingleWall(Long userId,Long rpId) {
        return Json.build(redPacketService.getSingleWallReadPacket(userId,rpId));
    }

    @ApiOperation(value = "2-12-24 获取红包的最近领取记录(站外)")
    @ApiImplicitParams({
    })
    @GetMapping("/public/obtainHistory")
    public Json<PageWarper<PacketHistoryDTO>> getObtainHistoryPublic(ObtainHistoryParam obtainHistoryParam) {
        return Json.build(redPacketService.getObtainHistory(obtainHistoryParam));
    }


    @ApiOperation(value = "2-12-25 获取语音评分排名(站外)")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true)
    })
    @GetMapping("/public/ranklist")
    public Json<List<PacketHistoryDTO>> getRankListPublic(@RequestParam("rpId") Long rpId) {
        return Json.build(redPacketService.getRanking(rpId));
    }

    @ApiOperation(value = "2-12-26 获取分享信息接口(站外)", notes = "站外获取用于分享的信息")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true),
            @ApiImplicitParam(name = "userId", value = "用户id", dataType = "long", required = true),
            @ApiImplicitParam(name = "type", value = "类型（1：未参与、2：已参与）", dataType = "int", required = true),
            @ApiImplicitParam(name = "nickname", value = "分享人昵称", dataType = "string", required = true)
    })
    @PostMapping("/get/public/share")
    public Json<RedPacketShareDTO> getShareInfoPublic(@RequestParam("rpId") Long rpId,
                                                @RequestParam("userId") Long userId,
                                                @RequestParam("type") Integer type,
                                                @RequestParam("nickname") String nickname) {
        RedPacketShareDTO result = redPacketService.getShareInfo(rpId,userId,type, nickname);
        if(null == result){
            return Json.build(RespCode.BAD_REQUEST,"share info is empty");
        }
        return Json.build(result);
    }

    @ApiOperation(value = "2-12-27 查询红包支付状态(H5调)", notes = "支付成功：true,支付失败:false")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "rpId", value = "红包id", dataType = "long", required = true)
    })
    @PostMapping("/get/isPaySuccess")
    public Json<Boolean> isPaySuccess(@RequestParam("rpId") Long rpId){
        return Json.build(redPacketService.isPaySuccess(rpId));
    }

}
