package com.bxm.localnews.activity.service.impl;

import com.bxm.localnews.activity.domain.DailyTaskMapper;
import com.bxm.localnews.activity.domain.NoviceTaskRecordMapper;
import com.bxm.localnews.activity.domain.SignRecordMapper;
import com.bxm.localnews.activity.dto.AppletAppTaskResultDTO;
import com.bxm.localnews.activity.dto.NewsMissionRewardDto;
import com.bxm.localnews.activity.param.TaskContext;
import com.bxm.localnews.activity.service.MissionService;
import com.bxm.localnews.activity.service.VipService;
import com.bxm.localnews.activity.strategy.TaskStrategyContext;
import com.bxm.localnews.activity.vo.*;
import com.bxm.localnews.base.service.AppVersionSupplyService;
import com.bxm.localnews.base.service.LocationFacadeService;
import com.bxm.localnews.common.config.NewsProperties;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.TaskEnum;
import com.bxm.localnews.common.constant.WxMiniSceneEnum;
import com.bxm.localnews.common.dto.LocationDetailDTO;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.localnews.integration.UserAuthIntegrationService;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.localnews.thirdparty.dto.AdvertDTO;
import com.bxm.localnews.thirdparty.service.AdvertService;
import com.bxm.localnews.thirdparty.service.WxMpFacadeService;
import com.bxm.localnews.vo.UserAuth;
import com.bxm.newidea.component.log.LogMarker;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.gexin.fastjson.JSON;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import static com.bxm.localnews.activity.vo.DailyTask.*;
import static com.bxm.localnews.activity.vo.UserMissionModel.TASK_STATE_NO;
import static com.bxm.localnews.activity.vo.UserMissionModel.TASK_STATE_YES;
import static com.bxm.localnews.common.constant.RedisConfig.*;
import static com.bxm.localnews.common.constant.TaskEnum.*;

/**
 * Created by Administrator on 2018/2/23.
 */
@Service("missionService")
public class MissionServiceImpl extends BaseService implements MissionService {

    private final static Logger logger = LoggerFactory.getLogger(MissionServiceImpl.class);

    @Resource
    private SignRecordMapper signRecordMapper;

    @Resource
    private DailyTaskMapper dailyTaskMapper;

    @Resource
    private UserAccountIntegrationService userAccountIntegrationService;

    @Resource
    private RedisStringAdapter redisStringAdapter;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private NoviceTaskRecordMapper noviceTaskRecordMapper;

    @Resource
    private TaskStrategyContext taskStrategyContext;

    @Resource
    private LocationFacadeService locationFacadeService;

    @Resource
    private NewsProperties newsProperties;

    @Resource
    private UserIntegrationService userIntegrationService;

    @Resource
    private VipService vipService;

    @Resource
    private AdvertService advertService;

    @Resource
    private AppVersionSupplyService appVersionSupplyService;

    @Resource
    private WxMpFacadeService wxMpFacadeService;

    @Resource
    private UserAuthIntegrationService userAuthIntegrationService;

    @Override
    public Json<MissionModel> listMissions(Long userId, String areaCode, String curVer, Integer platform) {
        MissionModel retModel = new MissionModel();

        //1.获取用户任务
        List<UserMissionModel> userTask = getUserTaskByPlatform(userId, platform);

        //2.根据任务进行分类处理
        List<UserMissionModel> newbieTaskList = userTask.stream()
                .filter(e -> NEWBIE_TASK.equals(e.getType()))
                .sorted(Comparator.comparingInt(UserMissionModel::getLevel))
                .collect(Collectors.toList());

        List<UserMissionModel> dailyTaskList = userTask.stream()
                .filter(e -> DAILY_TASK.equals(e.getType()))
                .sorted(Comparator.comparingInt(UserMissionModel::getLevel))
                .collect(Collectors.toList());

        //3.获得用户定位信息
        LocationDetailDTO locationDetailDTO = getLocation(userId, areaCode);

        //4.判断是否开通社区
        boolean isPost = checkPost(locationDetailDTO);

        //5.(分类处理)添加日常任务和新手任务，并做处理
        this.filterNewbieTask(newbieTaskList, retModel, userId, isPost, locationDetailDTO);
        this.filterDailyTask(dailyTaskList, retModel, userId, isPost, locationDetailDTO);

        logger.info("用户id为[{}]首次处理后的任务信息为:{}", userId, JSON.toJSONString(userTask));

        //6.添加广告类型任务
        this.supplementAdvertMission(curVer, retModel, areaCode, userId, isPost);

        //7.获得用户今日份完成任务获得的赏金
        Long gainGold = getRewardToday(userId);

        retModel.setGainGold(gainGold == null ? null : gainGold.intValue());
        retModel.setTotalGold(newsProperties.getGoldPerDay());

        return ResultUtil.genSuccessResult(retModel);

    }

    /**
     * 添加广告类型的任务
     *
     * @param curVer
     * @param retModel
     * @param areaCode
     * @param userId
     * @param isPost
     */
    private void supplementAdvertMission(String curVer, MissionModel retModel, String areaCode, Long userId, boolean isPost) {
        //任务广告-判断版本号,调用广告接口并做数据转换
        List<UserMissionModel> advertDailyTaskList = Lists.newArrayList();
        //查询广告类型的日常任务
        if (StringUtils.isNotBlank(curVer) && appVersionSupplyService.isHighVersion(curVer, "1.2.2") != -1) {
            List<AdvertDTO> advertList = advertService.queryAdByType((byte) 2, areaCode, userId,null);
            advertDailyTaskList = convertAdvert(advertList);
        }

        //备注：2.1.0版本之前日常任务广告放在前面 2.1.0版本日常任务广告放在后面
        //advertDailyTaskList.addAll(dailyTaskList);
        //retModel.setDailyTaskList(advertDailyTaskList);

        //每日签到放到第一的位置
        Optional<UserMissionModel> advertDailyTaskStream = advertDailyTaskList.stream().filter(e-> "每日签到".equals(e.getName())).findFirst();
        List<UserMissionModel> dailyTaskList = new ArrayList<>();
        if (advertDailyTaskStream.isPresent()) {
            UserMissionModel dailySignTask = advertDailyTaskStream.get();
            // 若用户已经签到
            SignRecord signRecord = signRecordMapper.getLastSignRecord(userId);
            if (signRecord != null && DateUtils.formatDate(new Date()).equals(DateUtils.formatDate(signRecord.getSignDate()))) {
                dailySignTask.setCompleted(true);
                dailySignTask.setAction("已完成");
            }
            dailyTaskList.add(dailySignTask);
            advertDailyTaskList.remove(dailySignTask);
        }

        dailyTaskList.addAll(retModel.getDailyTaskList());

        if (!isPost) {
            dailyTaskList = dailyTaskList.stream().filter(userMissionModel -> !userMissionModel.getName().
                    equals(TaskEnum.TASK_POST_INTIVATION.getDesc())).collect(Collectors.toList());
        }
        dailyTaskList.addAll(advertDailyTaskList);
        retModel.setDailyTaskList(dailyTaskList);
    }

    /**
     * 计算用户当天完成任务所获得的小红花数
     *
     * @param userId
     * @return
     */
    private Long getRewardToday(Long userId) {
        KeyGenerator userRewardSumKey = RedisConfig.USER_MISSION_REWARD_SUM_PER_DAY.copy()
                .appendKey(userId.toString()).appendKey(DateUtils.formatDate(new Date()));
        logger.debug("从redis获取用户[{}]当天完成任务获得奖励小红花总数", userId);
        return redisStringAdapter.getLong(userRewardSumKey);
    }

    /**
     * 新手任务处理
     *
     * @param taskList
     * @param retModel
     * @param userId
     * @param isPost
     */
    private void filterNewbieTask(List<UserMissionModel> taskList, MissionModel retModel, Long userId, boolean isPost, LocationDetailDTO locationDetailDTO) {
        //过滤不展示的任务--现需求在过滤新手任务,有需要可以移到if外层做所有任务的判断
        for (UserMissionModel e : taskList) {
            if (filterNotDisplayTaskByNoviceTask(e)) {
                logger.info("该任务[{}]不在前端展示", e.getName());
                continue;
            }

            //获得任务的枚举类
            TaskEnum taskEnum = TaskEnum.getTaskByType(e.getTaskType());
            e.setColor(taskEnum.getColor());

            //处理新手任务判断
            processNewbieTask(e, taskEnum, userId, isPost, locationDetailDTO);

            //当新手任务是完善资料或者首次发帖时，如果完成就隐藏
            if (TASK_IMPROVE_INFO.equals(taskEnum) || TASK_FIRST_POST_INTIVATION.equals(taskEnum)) {
                if (e.getCompleted()) {
                    continue;
                }
            }

            retModel.addNewbieTask(e);
        }
    }

    /**
     * 日常任务处理
     *
     * @param taskList
     * @param retModel
     * @param userId
     * @param isPost
     * @param locationDetailDTO
     */
    private void filterDailyTask(List<UserMissionModel> taskList,
                                 MissionModel retModel,
                                 Long userId,
                                 boolean isPost,
                                 LocationDetailDTO locationDetailDTO) {
        //获取用户已完成的任务
        Map<String, Byte> dailyTaskMap = this.getDailyTaskStatus(userId);

        for (UserMissionModel e : taskList) {
            //获得任务的枚举类
            TaskEnum taskEnum = TaskEnum.getTaskByType(e.getTaskType());
            e.setColor(taskEnum.getColor());

            //处理日常任务判断
            processDailyTask(e, taskEnum, userId, isPost, dailyTaskMap, locationDetailDTO);

            if (taskEnum.equals(TASK_ACTIVATION_VIP) && e.getCompleted()) {
                continue;
            }
            retModel.addDailyTask(e);
        }
    }

    /**
     * 日常任务广告转换为日常任务
     *
     * @param advertList
     * @return
     */
    private List<UserMissionModel> convertAdvert(List<AdvertDTO> advertList) {
        List<UserMissionModel> dailyTaskList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(advertList)) {
            return dailyTaskList;
        }

        advertList.forEach(advertDTO -> {
            UserMissionModel userMissionModel = new UserMissionModel();
            userMissionModel.setAction(advertDTO.getCopy());
//            userMissionModel.setHeadImg(advertDTO.getImgUrl());
            //2.1.0版本修改- 广告改为广告素材+投放的形式,图片从iconUrl中获取
            userMissionModel.setHeadImg(advertDTO.getIconUrl());
            userMissionModel.setName(advertDTO.getTaskMaintitle());
            userMissionModel.setRemark(advertDTO.getTaskSubtitle());
            userMissionModel.setAddress(advertDTO.getAddress());
            userMissionModel.setType(UserMissionModel.ADVERT_TASK);
            userMissionModel.setId(advertDTO.getId());
            userMissionModel.setMaterialId(advertDTO.getMaterialId());
            dailyTaskList.add(userMissionModel);
        });

        return dailyTaskList;
    }

    /**
     * 是否启用发帖任务
     *
     * @param locationDetailDTO
     * @return
     */
    private Boolean checkPost(LocationDetailDTO locationDetailDTO) {
        return locationDetailDTO != null && locationDetailDTO.getEnableCommunityContent() != 0;
    }

    /**
     * 是否显示vip任务
     *
     * @param userId
     */
    private Boolean checkVipActivation(Long userId, LocationDetailDTO locationDetailDTO) {
        return vipService.checkUserVip(userId) || isShowVipTask(locationDetailDTO);
    }

    /**
     * 获取地区信息
     *
     * @param areaCode
     * @return
     */
    private LocationDetailDTO getLocation(Long userId, String areaCode) {
        LocationDetailDTO locationDTO = null;
        if (StringUtils.isNotEmpty(areaCode)) {
            locationDTO = locationFacadeService.getLocationDetailByCode(areaCode);
        } else {
            //如果没有传地区编号，则去用户表里获取地区编号，如果查不到则说明是未开通地区
            LocationDetailDTO locationIntegrationDetailDTO = userIntegrationService.getLocationByUser(userIntegrationService.selectByPrimaryKey(userId));
            if (null != locationIntegrationDetailDTO) {
                locationDTO = new LocationDetailDTO();
                BeanUtils.copyProperties(locationIntegrationDetailDTO, locationDTO);
            }
        }

        return locationDTO;
    }

    /**
     * 如果地区不是开通地区，则隐藏
     *
     * @param locationDTO
     * @return
     */
    private boolean isShowVipTask(LocationDetailDTO locationDTO) {
        if (locationDTO == null) {
            return false;
        }

        return 0 == locationDTO.getEnableVip();
    }

    @Override
    public Json<BaskInfoMeta> baskInfo(Long userId) {
        BigDecimal totalCash = this.userAccountIntegrationService.getUserTotalCash(userId);

        BaskInfoMeta meta = new BaskInfoMeta();
        meta.setInviteCode(userId);
        meta.setTotalCoin(totalCash);
        return ResultUtil.genSuccessResult(meta);
    }

    @Override
    public void completeDailyTask(Long userId, Byte taskId) {
        Assert.notNull(taskId, "任务ID不能为空");
        logger.debug(LogMarker.BIZ, "[{}]完成了任务[{}]", userId, taskId);

        KeyGenerator key = getDailyTaskCacheKey(userId);

        this.redisHashMapAdapter.put(key, taskId.toString(), TASK_STATE_YES);
        this.redisHashMapAdapter.expire(key, DateUtils.getCurSeconds());
    }

    private KeyGenerator getDailyTaskCacheKey(Long userId) {
        return DAILY_TASK_COMPELE_STATUS.copy().setKey(DateUtils.formatDate(new Date()) + userId);
    }

    @Override
    public Map<String, Byte> getDailyTaskStatus(Long userId) {
        Assert.notNull(userId, "用户ID不能为空");

        Map<String, Byte> result = this.redisHashMapAdapter.entries(getDailyTaskCacheKey(userId), Byte.class);
        if (result == null) {
            result = Maps.newHashMap();
        }
        return result;
    }

    @Override
    public NewsMissionRewardDto completeTask(Long userId, TaskEnum taskEnum, String relationId, String content) {
        logger.debug("用户[{}]完成任务类型[{}]---建立邀请关系[{}]", userId, taskEnum.name(), relationId);

        TaskContext taskContext = new TaskContext();
        taskContext.setUserId(userId);
        taskContext.setTaskEnum(taskEnum);
        taskContext.setRelationId(relationId);
        taskContext.setContent(content);

        return taskStrategyContext.chooseStrategy(taskContext);
    }

    @Override
    public AppletAppTaskResultDTO completeSceneTask(Long userId, String scene) {
        TaskEnum taskEnum = WxMiniSceneEnum.getTaskEnumByScene(scene);
        if (null == taskEnum) {
            logger.warn("错误场景值[{}]传入,无法获取对应任务", scene);
            return new AppletAppTaskResultDTO();
        }
        //是否完成新人任务-必须在接下来的'完成任务'步骤前查询
        boolean isCompleteNoviceTask = isCompleteNoviceTask(userId, taskEnum);
        NewsMissionRewardDto taskReward = completeTask(userId, taskEnum, null, null);

        AppletAppTaskResultDTO appletAppTaskResultDTO = new AppletAppTaskResultDTO();
        //判断是'之前未完成的任务'且任务奖金大于0-则弹窗
        if (!isCompleteNoviceTask && taskReward.getGoldNum() != null && 0 < taskReward.getGoldNum()) {
            appletAppTaskResultDTO.setPopUpWindowFlag(true);
        }
        appletAppTaskResultDTO.setTaskReward(taskReward.getGoldNum());

        return appletAppTaskResultDTO;
    }

    /**
     * 根据平台获取用户任务
     *
     * @param userId   用户id
     * @param platform 平台号
     * @return 查询结果
     */
    private List<UserMissionModel> getUserTaskByPlatform(Long userId, Integer platform) {
        List<UserMissionModel> userTask;
        if (5 == platform) {
            //小程序任务
            userTask = this.dailyTaskMapper.listMissions(userId, APPLET_MINI_TYPE);
            logger.debug("从数据库中获取到的小程序任务为:{}", JSON.toJSONString(userTask));
        } else {
            userTask = this.dailyTaskMapper.listMissions(userId, APPLET_APP_TYPE);
            logger.debug("从数据库中获取到的客户端任务为:{}", JSON.toJSONString(userTask));
        }
        return userTask;
    }

    /**
     * 处理新手任务
     *
     * @param model  用户任务传输对象
     * @param userId 用户id
     * @param isPost 是否post
     * @return 处理结果 boolean结果暂不做判断
     */
    private void processNewbieTask(UserMissionModel model, TaskEnum taskEnum, Long userId, boolean isPost, LocationDetailDTO locationDetailDTO) {
        model.setCompleted(false);
        //新手任务中完善个人资料
        switch (taskEnum) {
            //完善个人资料
            case TASK_IMPROVE_INFO:
                model.setAddress("tt://perfectData");
                if (TASK_STATE_YES == model.getNewbiewTaskState()) {
                    model.setCompleted(true);
                }
                break;
            //首次发帖
            case TASK_FIRST_POST_INTIVATION:
                model.setAddress("tt://postInvitation");
                if (TASK_STATE_YES == model.getNewbiewTaskState() || !isPost) {
                    model.setCompleted(true);
                }
                model.setName(model.getName().replace("本地", locationDetailDTO.getName()).replace("市", "").replace("县", ""));
                break;
            //收藏任务判断
            case TASK_COLLECT_APPLET_MINI:
                NoviceTaskRecord noviceTaskRecord =
                        noviceTaskRecordMapper.findSelectiveByTaskType(model.getTaskType(), userId);

                if (null != noviceTaskRecord && 1 == noviceTaskRecord.getState()) {
                    model.setCompleted(true);
                }
                logger.info("收藏任务没记录或者未完成,任务记录为:{}", JSON.toJSONString(noviceTaskRecord));
                break;
            //关注微信公众号
            case TASK_FOCUS_WECAHT:
                UserAuth userAuth = userAuthIntegrationService.selectWechatUserAuthByUserId(userId);
                if (null != userAuth) {
                    if (wxMpFacadeService.subscribeWechat(userAuth.getIdentifier())) {
                        //若查询到已关注公众号但无任务完成记录，则触发完成任务功能-（前端无法触发关注公众号任务完成）
                        NoviceTaskRecord focusWechatTaskRecord = noviceTaskRecordMapper.findSelectiveByTaskType(
                                TaskEnum.TASK_FOCUS_WECAHT.getType(), userId);
                        if (focusWechatTaskRecord == null) {
                            logger.info("当前用户[{}]关注公众号----触发任务完成", userId);
                            completeTask(userId, TaskEnum.TASK_FOCUS_WECAHT, null, null);
                        }
                        model.setCompleted(true);
                    }
                }
                break;
            default:
                break;
        }

    }


    /**
     * 处理日常任务
     *
     * @param userMissionModel  用户任务类
     * @param userId            用户id
     * @param locationDetailDTO location传输对象
     * @param isPost            是否post
     * @param dailyTaskMap      日常任务map
     */
    private void processDailyTask(UserMissionModel userMissionModel, TaskEnum taskEnum, Long userId,
                                  boolean isPost, Map<String, Byte> dailyTaskMap, LocationDetailDTO locationDetailDTO) {
        userMissionModel.setState(dailyTaskMap.get(userMissionModel.getId().toString()) == null ? TASK_STATE_NO : TASK_STATE_YES);
        //根据类型获取具体任务枚举类
        KeyGenerator key = null;
        switch (taskEnum) {
            case TASK_NEWS_READ:
                key = NEWS_READ.copy().setKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                userMissionModel.setAddress("tt://readNews");
                break;
            case TASK_NEWS_SHARE:
                key = TASK_SHARE_NEWS_NUM.copy().appendKey(userId.toString()).appendKey(DateUtils.formatDate(new Date()));
                userMissionModel.setAddress("tt://shareWxchat");
                break;
            case TASK_VIDEO_READ:
                key = VIDEO_READ_NUM.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                userMissionModel.setAddress("tt://lookVideo");
                break;
            case TASK_INVITED_FRIEND:
                userMissionModel.setCompleted(false);
                userMissionModel.setAddress("tt://inviteSt");
                break;
            case TASK_ACTIVATION_VIP:
                Boolean isVip = checkVipActivation(userId, locationDetailDTO);
                userMissionModel.setCompleted(isVip);
                userMissionModel.setAddress("tt://callActivateVip");
                break;
            case TASK_POST_INTIVATION:
                userMissionModel.setCompleted(isPost);
                key = POST_FORUM.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                userMissionModel.setAddress("tt://postInvitation");
                userMissionModel.setName(userMissionModel.getName().replace("本地", locationDetailDTO.getName()).replace("市", "").replace("县", ""));
                break;
            case TASK_FIRST_BROWSE:
                userMissionModel.setAddress("tt://shareNews");
                //替换本地为当地名称
                userMissionModel.setName(userMissionModel.getName().replace("本地", locationDetailDTO.getName()).replace("市", "").replace("县", ""));
                break;
            case TASK_COMMENT_NEWS:
                userMissionModel.setAddress("tt://commentNews");
                //替换本地为当地名称
                userMissionModel.setName(userMissionModel.getName().replace("本地", locationDetailDTO.getName()).replace("市", "").replace("县", ""));
                key = NEWS_COMMENT.copy().appendKey(userId + DateUtils.PATTERN_NO_DELIMITER_FORMAT.get().format(new Date()));
                break;
            default:
                break;
        }
        if (key != null) {
            Integer number;
            if (redisStringAdapter.hasKey(key)) {
                number = this.redisStringAdapter.getInt(key);
            } else {
                number = 0;
            }
            userMissionModel.setCompleted(number >= userMissionModel.getNumber());
            userMissionModel.setCompletedNum(number);
            userMissionModel.setTotalNum(userMissionModel.getNumber());
        }

    }

    /**
     * 判断是否完成新人任务
     *
     * @param userId
     * @param taskEnum
     * @return
     */
    private boolean isCompleteNoviceTask(Long userId, TaskEnum taskEnum) {
        NoviceTaskRecord noviceTaskRecord = noviceTaskRecordMapper.findSelectiveByTaskType(taskEnum.getType(), userId);
        if (null != noviceTaskRecord && NoviceTaskRecord.FINISHED == noviceTaskRecord.getState()) {
            logger.debug("查询的新人任务已完成,用户id:{},任务类型:{}", userId, taskEnum.getDesc());
            return true;
        }

        logger.debug("查询新人任务未完成,用户id:{},任务类型:{}", userId, taskEnum.getDesc());
        return false;
    }

    /**
     * 过滤不对外展示的任务(新手)
     *
     * @param model 任务信息类
     * @return 符合过滤(需要过滤的 - 即为不展示的任务)返回true, 否则返回false
     */
    private boolean filterNotDisplayTaskByNoviceTask(UserMissionModel model) {
        return TaskEnum.TASK_FIRST_LOGIN.getType() == model.getTaskType();
    }
}
