package com.bxm.localnews.activity.strategy;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.activity.command.TaskInvoke;
import com.bxm.localnews.activity.command.TransferPushMsgCommand;
import com.bxm.localnews.activity.command.TransferUserEventCommand;
import com.bxm.localnews.activity.common.constant.TaskStateEnum;
import com.bxm.localnews.activity.domain.DailyTaskMapper;
import com.bxm.localnews.activity.dto.NewsMissionRewardDto;
import com.bxm.localnews.activity.param.TaskContext;
import com.bxm.localnews.activity.vo.DailyTask;
import com.bxm.localnews.common.constant.RedisConfig;
import com.bxm.localnews.common.constant.TaskEnum;
import com.bxm.localnews.facade.PushMsgSupplyFeignService;
import com.bxm.localnews.integration.UserAccountIntegrationService;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;

import java.math.BigDecimal;

public abstract class AbstractTaskStrategy extends BaseService implements TaskStrategy {

    protected static ThreadLocal<TaskInvoke> taskInvokeThreadLocal = ThreadLocal.withInitial(TaskInvoke::new);

    @Autowired
    protected UserAccountIntegrationService userAccountIntegrationService;

    @Autowired
    private PushMsgSupplyFeignService pushMsgSupplyFeignService;

    @Autowired
    private DailyTaskMapper dailyTaskMapper;

    @Autowired
    protected SequenceCreater sequenceCreater;

    @Autowired
    private DistributedLock lock;

    private static final String LOCK_KEY = "LOCK_KEY";

    private static final String REQUEST_ID = "REQUEST_ID";

    @Override
    public NewsMissionRewardDto process(TaskContext taskContext) {
        logger.debug("进入抽象策略，起始方法，线程id:[{}],threadLocal地址:[{}]", Thread.currentThread(), taskInvokeThreadLocal.toString());

        //所有任务的通用预处理（预先判断任务是否有效）
        if (!beforeProcessing(taskContext)) {
            return convertTask(taskContext.getTaskEnum(), 0L, taskContext.getUserId());
        }

        //所有任务的通用执行处理任务方法（完成任务）
        NewsMissionRewardDto missionRewardDto = completeTask(taskContext);
        logger.debug("完成任务[{}],获得奖励[{}]",taskContext.getTaskEnum().name(),JSONObject.toJSONString(missionRewardDto));

        //所有任务的通用后置处理（执行处理消息推送或者金币增加命令，并清除ThreadLocal）
        afterProcessing();

        return missionRewardDto;
    }

    /**
     * 数据预处理（预先判断任务是否有效）
     * @param taskContext
     * @return
     */
    private boolean beforeProcessing(TaskContext taskContext) {
        DailyTask dailyTask = dailyTaskMapper.selectByTaskType(taskContext.getTaskEnum().getType());
        logger.debug("获得任务:[{}]", JSONObject.toJSONString(dailyTask));
        //任务是否存在
        if (dailyTask == null) {
            logger.error("[{}]任务不存在", taskContext.getTaskEnum());
            return false;
        }
        //新手任务是否开启状态（10-关/20-开）
        if (TaskStateEnum.ENABLE != TaskStateEnum.isAble(dailyTask.getState())) {
            logger.warn("[{}]任务状态为'关闭'", taskContext.getTaskEnum());
            return false;
        }
        taskContext.setDailyTask(dailyTask);
        return true;
    }

    /**
     * 判断任务是否完成
     * 方法骨架：
     * 预处理
     * 正式处理
     * 后置处理
     */
    private NewsMissionRewardDto completeTask(TaskContext taskContext) {

        //具体到每个任务的预处理（大部分都没有）判断任务是否能完成
        Message msg = preProcessing(taskContext);
        if (!msg.isSuccess()) {
            logger.warn("该任务[{}]已无法完成,具体原因：[{}]", taskContext.getTaskEnum().getDesc(), msg.getLastMessage());
            return convertTask(taskContext.getTaskEnum(), 0L, taskContext.getUserId());
        }

        //具体到每个任务的正式处理
        NewsMissionRewardDto missionRewardDto = processing(taskContext);

        //返回任务的包装xxx

        //具体到每个任务的任务的后置处理（将消息推送或者金币增加加入命令队列）
        postProcessing(taskContext);

        return missionRewardDto;
    }

    /**
     * 具体到每个任务的预处理（大部分都没有）判断任务是否能完成
     * 今日红花是否溢出等
     * @param taskContext
     * @return
     */
    protected abstract Message preProcessing(TaskContext taskContext);

    /**
     * 正式处理每个任务
     * @param taskContext
     * @return
     */
    protected abstract NewsMissionRewardDto processing(TaskContext taskContext);

    /**
     * 任务后置处理，具体为，将红花增加或者消息推送加入命令队列
     * @param taskContext
     */
    protected abstract void postProcessing(TaskContext taskContext);

    /**
     * 数据后置处理
     */
    private void afterProcessing() {
        try {
            taskInvokeThreadLocal.get().execute();
        } finally {
            taskInvokeThreadLocal.remove();
            logger.debug("清除当前线程的threadLocal:[{}]", taskInvokeThreadLocal.get() == null);
        }
    }

    /**
     * 默认转换任务实体
     * @param task   任务类型
     * @param reward 奖励金额
     * @param userId 任务所属用户
     * @return 任务奖励信息
     */
    protected NewsMissionRewardDto convertTask(TaskEnum task, Long reward, Long userId) {
        NewsMissionRewardDto result = new NewsMissionRewardDto();
        result.setTaskName(task.getDesc());
        result.setGoldNum(reward);
        result.setTotalGold(BigDecimal.valueOf(userAccountIntegrationService.getUserUsableGold(userId)));
        return result;
    }

    /**
     * 在任务完成后进行消息推送
     */
    protected void afterCompleteToPushMsg(TaskContext taskContext) {
        //推送消息
        TransferPushMsgCommand command = new TransferPushMsgCommand(taskContext, userAccountIntegrationService, pushMsgSupplyFeignService);
        taskInvokeThreadLocal.get().setCommand(command);
    }

    /**
     * 在任务完成后进行用户事件通知
     */
    protected void afterCompleteToUserEvent(TaskContext taskContext) {
        //推送消息
        TransferUserEventCommand command = new TransferUserEventCommand(taskContext, userAccountIntegrationService, pushMsgSupplyFeignService);
        taskInvokeThreadLocal.get().setCommand(command);
    }

    protected boolean lock(TaskContext t) {

        String lockKey = t.getParam(LOCK_KEY);

        if (StringUtils.isEmpty(lockKey)) {
            lockKey = getLockKey(t);
        }

        // 如果没有设置则表示不加锁
        if (StringUtils.isEmpty(lockKey)) {
            return true;
        }

        String requestId = sequenceCreater.nextLongId().toString();

        t.addParam(LOCK_KEY, lockKey);
        t.addParam(REQUEST_ID, requestId);

        return lock.lock(lockKey, requestId);
    }

    private String getLockKey(TaskContext param) {
        return RedisConfig.ACTIVITY_LOCK
                .copy()
                .appendKey(param.getTaskEnum().getType())
                .appendKey(param.getUserId())
                .gen();
    }

}
