package com.bxm.localnews.activity.record.parser.impl;

import com.baidu.aip.speech.AipSpeech;
import com.bxm.localnews.activity.common.config.ActivityProperties;
import com.bxm.localnews.activity.record.convert.ConvertProcesser;
import com.bxm.localnews.activity.record.download.OSSDownloader;
import com.bxm.localnews.activity.record.download.WeixinMediaDownload;
import com.bxm.localnews.activity.record.parser.RecordParser;
import com.bxm.localnews.activity.vo.RecordInfoBean;
import com.bxm.newidea.component.oss.service.AliyunOSSService;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.json.JSONArray;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;

@Service
@Slf4j
public class RecordParserImpl implements RecordParser {

    private final static Logger logger = LoggerFactory.getLogger(RecordParserImpl.class);
    private AliyunOSSService aliyunOSSService;

    private ActivityProperties activityProperties;

    private WeixinMediaDownload weixinMediaDownload;

    private OSSDownloader ossDownloader;

    private ConvertProcesser convertProcesser;

    @Autowired
    public RecordParserImpl(AliyunOSSService aliyunOSSService,
                            ActivityProperties activityProperties,
                            WeixinMediaDownload weixinMediaDownload,
                            ConvertProcesser convertProcesser,
                            OSSDownloader ossDownloader) {
        this.aliyunOSSService = aliyunOSSService;
        this.activityProperties = activityProperties;
        this.weixinMediaDownload = weixinMediaDownload;
        this.convertProcesser = convertProcesser;
        this.ossDownloader = ossDownloader;
    }

    @Override
    public RecordInfoBean parse(String url, int duration) {
        File ossFile = ossDownloader.download(url);
        RecordInfoBean recordInfo = convertProcesser.convert(ossFile, duration);
        recordInfo.setOssUrl(url);
        parse(recordInfo);

        return recordInfo;
    }

    @Override
    public RecordInfoBean parse(File recordFile, int duration) {
        RecordInfoBean recordInfo = convertProcesser.convert(recordFile, duration);
        parse(recordInfo);

        return recordInfo;
    }

    @Override
    public RecordInfoBean parse(String mediaId) {
        File mediaFile = weixinMediaDownload.download(mediaId);

        RecordInfoBean recordInfo = convertProcesser.convert(mediaFile, -1);

        // 语音oss上传
        String ossUrl = aliyunOSSService.upload(recordInfo.getConvertFile(),null);
        recordInfo.setOssUrl(ossUrl);
        logger.info("wechat vioce ossAddress :[{}]",ossUrl);

        parse(recordInfo);
        return recordInfo;
    }

    /**
     * 调用百度的AIP接口进行录音的识别
     * @param recordInfo 录音文件信息
     */
    private void parse(RecordInfoBean recordInfo) {
        AipSpeech client = new AipSpeech(activityProperties.getBaiduAiAppId(),
                activityProperties.getBaiduAiApiKey(),
                activityProperties.getBaiduAiSecretKey());

        client.setConnectionTimeoutInMillis(10000);
        client.setSocketTimeoutInMillis(100000);

        FileInputStream fis = null;
        try {
            fis = new FileInputStream(recordInfo.getConvertFile());
            byte[] res = IOUtils.toByteArray(fis);
            JSONObject result = client.asr(res, recordInfo.getConvertExtend(), 16000, null);

            int errorNo = result.getInt("err_no");
            if (0 == errorNo) {
                JSONArray resultArray = result.getJSONArray("result");
                String firstMatchResult = (String) resultArray.get(0);
                recordInfo.setContent(firstMatchResult);
            } else {
                recordInfo.setContent(StringUtils.EMPTY);
                String errorMsg = result.getString("err_msg");
                log.error("录音解析错误，errorCode:{},errorMsg:{}", errorNo, errorMsg);
            }
        } catch (Exception e) {
            recordInfo.setContent(StringUtils.EMPTY);
            log.error("录音文件读取错误，文件信息:" + recordInfo, e);
        } finally {
            if (fis != null) {
                try {
                    fis.close();
                } catch (IOException e) {
                    log.error(e.getMessage(), e);
                }
            }
            clear(recordInfo);
        }
    }

    /**
     * 解析完成后的清理方法
     * @param recordInfo 录音信息
     */
    private void clear(RecordInfoBean recordInfo) {
        if (null != recordInfo.getSourceFile()) {
            boolean result = recordInfo.getSourceFile().delete();
            if (!result) {
                log.warn("文件无法删除或不存在,文件路径：" + recordInfo.getSourceFile());
            }
        }
        if (null != recordInfo.getConvertFile()) {
            boolean reuslt = recordInfo.getConvertFile().delete();
            if (!reuslt) {
                log.warn("文件无法删除或不存在,文件路径：" + recordInfo.getConvertFile());
            }
        }
    }
}
