package com.bxm.localnews.activity.strategy.dailyConcreteStrategy;

import com.bxm.localnews.activity.domain.BrowseRecordMapper;
import com.bxm.localnews.activity.domain.SignRecordMapper;
import com.bxm.localnews.activity.dto.NewsMissionRewardDto;
import com.bxm.localnews.activity.dto.SignCompleTaskDTO;
import com.bxm.localnews.activity.param.TaskContext;
import com.bxm.localnews.activity.vo.SignRecord;
import com.bxm.localnews.common.config.NewsProperties;
import com.bxm.localnews.common.constant.TaskEnum;
import com.bxm.localnews.integration.UserIntegrationService;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.vo.Message;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.*;

import static com.bxm.localnews.common.constant.RedisConfig.IS_SIGN_SHARE;

@Service
public class NewsSignShareTaskStrategy extends AbstractDailyTaskStrategy {

    private final SignRecordMapper signRecordMapper;

    @Autowired
    public NewsSignShareTaskStrategy(RedisStringAdapter redisStringAdapter,
                                     RedisHashMapAdapter redisHashMapAdapter,
                                     NewsProperties newsProperties,
                                     UserIntegrationService userIntegrationService,
                                     BrowseRecordMapper browseRecordMapper,
                                     SignRecordMapper signRecordMapper) {
        super(redisStringAdapter, redisHashMapAdapter, newsProperties, userIntegrationService, browseRecordMapper);
        this.signRecordMapper = signRecordMapper;
    }

    @Override
    public boolean support(TaskEnum taskEnum) {
        return taskEnum.equals(TaskEnum.TASK_SIGN_SHARE);
    }

    @Override
    protected Message customJudge(TaskContext taskContext) {
        Message msg = judgeIsOverRewardCount(taskContext);
        if (!msg.isSuccess()) {
            return msg;
        }

        SignRecord signRecord = signRecordMapper.getLastSignRecord(taskContext.getUserId());
        if (!(signRecord != null && signRecord.getSignDate().equals(DateUtils.clearTimePart(new Date())))) {
            return Message.build(false, "该用户[{" + taskContext.getUserId() + "}]今日没有签到，无法获得奖励!");
        }

        long signReward = Math.toIntExact(signRecord.getSignReward());
        taskContext.addParam("signReward", signReward);
        return Message.build(true);
    }

    @Override
    protected boolean isSupportPushMessage() {
        return false;
    }

    @Override
    protected NewsMissionRewardDto processing(TaskContext taskContext) {
        //在完成任务之后需要对返回进行一定的修饰
        NewsMissionRewardDto newsMissionRewardDto = defaultProcess(taskContext);
        return convertTask(taskContext, newsMissionRewardDto);
    }

    private NewsMissionRewardDto convertTask(TaskContext taskContext, NewsMissionRewardDto newsMissionRewardDto) {
        SignCompleTaskDTO result = new SignCompleTaskDTO();
        result.setTaskName(newsMissionRewardDto.getTaskName());
        result.setGoldNum(newsMissionRewardDto.getGoldNum());
        result.setTotalGold(newsMissionRewardDto.getTotalGold());
        if (newsMissionRewardDto.getGoldNum() >= 0) {
            result.setRewardChooses(getRewardChoose(taskContext));
        }
        return result;
    }

    /**
     * 得到三个选择
     * @param taskContext
     * @return
     */
    private List<SignCompleTaskDTO.RewardChoose> getRewardChoose(TaskContext taskContext) {
        //签到奖励
        int signReward = Math.toIntExact(taskContext.getParam("signReward"));
        //实际转发的奖励
        int reward = taskContext.getDailyTask().getReward().intValue();
        int min = reward / 2;

        int other1;
        int other2;
        if (!redisHashMapAdapter.exists(IS_SIGN_SHARE.copy(), String.valueOf(taskContext.getUserId()))) {
            other1 = RandomUtils.nextInt(min, reward);
            other2 = RandomUtils.nextInt(min, other1);
            redisHashMapAdapter.put(IS_SIGN_SHARE.copy(),String.valueOf(taskContext.getUserId()),DateUtils.formatDate(new Date()));
        } else {
            other1 = RandomUtils.nextInt(min, reward);
            other2 = RandomUtils.nextInt(min, signReward * 10);
        }

        SignCompleTaskDTO.RewardChoose rewardChoose1 = getRewardChoose(reward, signReward, true);
        SignCompleTaskDTO.RewardChoose rewardChoose2 = getRewardChoose(other1, signReward, false);
        SignCompleTaskDTO.RewardChoose rewardChoose3 = getRewardChoose(other2, signReward, false);

        List<SignCompleTaskDTO.RewardChoose> rewardChooseSet = new ArrayList<>();
        rewardChooseSet.add(rewardChoose1);
        rewardChooseSet.add(rewardChoose2);
        rewardChooseSet.add(rewardChoose3);

        return rewardChooseSet;
    }

    /**
     * 计算倍数，倍数=奖励数额/当日签到所得红花数，小于10时保留一位小数，不小于10时取整
     * @param reward
     * @param signReward
     * @param isSelected
     * @return
     */
    private SignCompleTaskDTO.RewardChoose getRewardChoose(int reward, int signReward, boolean isSelected) {
        BigDecimal rewardBig = BigDecimal.valueOf(reward);
        BigDecimal signRewardBig = BigDecimal.valueOf(signReward);
        float multiple;
        BigDecimal bigDecimal = rewardBig.divide(signRewardBig, 1, RoundingMode.HALF_UP);
        if (bigDecimal.compareTo(BigDecimal.TEN) >= 0) {
            multiple = bigDecimal.intValue();
        } else {
            multiple = bigDecimal.floatValue();
        }
        return new SignCompleTaskDTO.RewardChoose(reward, multiple, isSelected);
    }

    /**
     * 每次转发奖励控制在30~60朵红花，数值范围由服务端配置（保留热更的可能），且不可大于当日签到所得红花数*10
     * @param taskContext
     * @return
     */
    @Override
    protected long getTaskReward(TaskContext taskContext) {
        int min = taskContext.getDailyTask().getReward().intValue();
        int max = taskContext.getDailyTask().getRewardLimit().intValue();
        long signReward = taskContext.getParam("signReward");
        //如果是第一次转发签到则取最大值30-60，如果签到红花数*10大于60则取60
        if (!redisHashMapAdapter.exists(IS_SIGN_SHARE.copy(), String.valueOf(taskContext.getUserId()))) {
            if (max >= signReward * 10) {
                return signReward * 10;
            } else {
                return max;
            }
        } else {
            //如果是不是第一次转发签到则取30-x区间的数
            if (max >= signReward * 10) {
                return RandomUtils.nextInt(min, Math.toIntExact(signReward * 10));
            } else {
                return RandomUtils.nextInt(min, max);
            }
        }
    }
}
