package com.bxm.localnews.admin.controller;

import com.bxm.localnews.admin.dto.AreaDivisionDTO;
import com.bxm.localnews.admin.dto.AreaInfoDTO;
import com.bxm.localnews.admin.dto.LocationDTO;
import com.bxm.localnews.admin.param.AreaInfoParam;
import com.bxm.localnews.admin.param.AreaInfoQueryParam;
import com.bxm.localnews.admin.param.LocationParam;
import com.bxm.localnews.admin.service.AreaDivisionService;
import com.bxm.localnews.admin.service.AreaInfoService;
import com.bxm.localnews.admin.service.LocationService;
import com.bxm.localnews.admin.vo.AreaInfo;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

import javax.validation.Valid;
import java.util.List;

@Api(tags = "1-03 [管理]地区管理", description = "地区管理的相关操作")
@RestController
@RequestMapping("api/admin/area")
public class AdminAreaController extends BaseController {

    private LocationService locationService;

    private AreaDivisionService areaDivisionService;

    private AreaInfoService areaInfoService;

    @Autowired
    public AdminAreaController(LocationService locationService,
                               AreaDivisionService areaDivisionService,
                               AreaInfoService areaInfoService) {
        this.locationService = locationService;
        this.areaDivisionService = areaDivisionService;
        this.areaInfoService = areaInfoService;
    }

    @GetMapping(value = "/list")
    @ApiOperation(value = "1-03-01 获取所有可开通和已开通城市")
    public Json<List<LocationDTO>> listLocation(@Valid LocationParam locationParam) {
        return ResultUtil.genSuccessResult(locationService.listLocation(locationParam));
    }

    @PostMapping(value = "/{code}/hot")
    @ApiOperation(value = "1-03-02 设置为热门城市", notes = "把城市设为热门城市，使之在客户端以列表形式展现")
    public Json saveHotLocation(@PathVariable String code) {
        locationService.saveHotLocation(code);
        return ResultUtil.genSuccessResult();
    }

    @PostMapping(value = "/{code}/opened")
    @ApiOperation(value = "1-03-03 设置为已开通城市", notes = "把城市设为已开通城市，使之在客户端以列表形式展现")
    public Json saveOpenedLocation(@PathVariable String code) {
        locationService.saveOpenedLocation(code);
        return ResultUtil.genSuccessResult();
    }

    @GetMapping(value = "/{code}/list")
    @ApiOperation(value = "1-03-04 获取省下所有已开通城市", notes = "在地址联动选择时，在下方显示该城市下的已开通地区")
    public Json<List<LocationDTO>> listLocation(@PathVariable String code) {
        return ResultUtil.genSuccessResult(locationService.listOpenedLocationByAreaWithTwoLevels(code, (byte) 1));
    }

    @GetMapping(value = "/province/{code}/list")
    @ApiOperation(value = "1-03-05 仅获取省份所有已开通或可开通城市", notes = "在地址联动选择时，在下方显示该城市下的已开通或可开通地区")
    public Json<List<LocationDTO>> listProvinceLocation(@PathVariable String code) {
        return ResultUtil.genSuccessResult(locationService.listOpenedLocationByAreaWithTwoLevels(code, null));
    }

    @GetMapping(value = "/info/list")
    @ApiOperation(value = "1-03-06 后台区域管理-获取所有地域信息", notes = "包含覆盖率和传播率")
    public Json<PageWarper<AreaInfo>> listAreaInfo(AreaInfoQueryParam areaInfoParam) {
        return ResultUtil.genSuccessResult(areaInfoService.queryPages(areaInfoParam));
    }

    @PostMapping(value = "/info/add")
    @ApiOperation(value = "1-03-07 后台区域管理-新增地域信息")
    public Json saveAreaInfo(@RequestBody AreaInfoParam areaInfoParam) {
        return Json.build(areaInfoService.saveAreaInfo(areaInfoParam));
    }


    @GetMapping(value = "/info/detail/{code}")
    @ApiOperation(value = "1-03-11 后台区域管理-根据code得到地域的详细信息")
    @ApiImplicitParam(name = "code", value = "地区编码", required = true)
    public Json<AreaInfoDTO> getAreaDetail(@PathVariable String code) {
        return ResultUtil.genSuccessResult(areaInfoService.getAreaInfo(code));
    }

    @PostMapping(value = "/info/modify")
    @ApiOperation(value = "1-03-08 后台区域管理-根据code修改地域信息")
    public Json modifyAreaInfo(@RequestBody AreaInfoParam areaInfoParam) {
        int flag = areaInfoService.updateAreaInfo(areaInfoParam);
        if (flag > 0) {
            return ResultUtil.genSuccessMsg("更新成功");
        }
        return ResultUtil.genFailedResult("更新失败");

    }

    @PostMapping(value = "/info/del/{code}")
    @ApiOperation(value = "1-03-09 后台区域管理-根据code删除地域信息")
    @ApiImplicitParam(name = "code", value = "地区编码", required = true)
    public Json delAreaInfo(@PathVariable String code) {
        int flag = areaInfoService.delAreaInfo(code);
        if (flag > 0) {
            return ResultUtil.genSuccessMsg("更新成功");
        }
        return ResultUtil.genFailedResult("更新失败");
    }


    @GetMapping(value = "/division/{parentCode}")
    @ApiOperation(value = "1-03-10 根据上级编码获取省市区县街道", notes = "如果parentCode为0,则返回最上层的省份信息")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "path", name = "parentCode", value = "父级编号"),
    })
    public Json<List<AreaDivisionDTO>> listAreaDivision(@PathVariable String parentCode) {
        return ResultUtil.genSuccessResult(areaDivisionService.listAreaDivisionDTO(parentCode));
    }


}
