package com.bxm.localnews.admin.controller.base;

import com.bxm.localnews.admin.controller.CommonController;
import com.bxm.localnews.admin.dto.*;
import com.bxm.localnews.admin.param.AppVersionParam;
import com.bxm.localnews.admin.param.RoleParam;
import com.bxm.localnews.admin.service.base.AdminAppVersionService;
import com.bxm.localnews.admin.service.security.*;
import com.bxm.localnews.admin.vo.*;
import com.bxm.localnews.admin.vo.security.AdminRole;
import com.bxm.localnews.admin.vo.security.AdminRoleMenu;
import com.bxm.localnews.admin.vo.security.AdminRolePerm;
import com.bxm.localnews.admin.vo.security.AdminUser;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

/**
 * @author zhaoyadong 2019/1/10 11:18
 * @desc
 */
@Api(tags = "1-25 [管理]App版本管理", description = "App版本管理的相关操作")
@RestController
@RequestMapping("api/admin/appversion")
public class AdminAppVersionController extends BaseController {

    private AdminAppVersionService adminAppVersionService;

    @Autowired
    public AdminAppVersionController(AdminAppVersionService adminAppVersionService) {
        this.adminAppVersionService = adminAppVersionService;
    }

    @ApiOperation(value = "1-25-1 获取App版本列表", notes = "根据输入参数获取对应的版本数据")
    @GetMapping("/list")
    public Json<PageWarper<AppVersionDTO>> getAppVersionList(AppVersionParam appVersionParam) {
        return ResultUtil.genSuccessResult(this.adminAppVersionService.getAppVersionList(appVersionParam));
    }

    @ApiOperation(value = "1-25-2 获取App版本详情", notes = "获取对应的版本详情")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "id", required = true)
    })
    @GetMapping("/detail")
    public Json<AppVersionDTO> getAppVersionById(Long id) {
        return ResultUtil.genSuccessResult(this.adminAppVersionService.getAppVersionById(id));
    }

    @ApiOperation(value = "1-25-3 app版本信息批量更新审核状态和版本更新方式", notes = "", hidden = true)
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id字符串，以”,“分隔", required = true),
            @ApiImplicitParam(name = "status", value = "状态", required = false),
            @ApiImplicitParam(name = "force", value = "更新方式", required = false)
    })
    @PostMapping("/batchUpdateStatus")
    public Json updateAppVersionStatusByIds(String ids, Byte status, Byte force) {
        if (StringUtils.isBlank(ids)) {
            return ResultUtil.genFailedResult("版本信息不存在");
        }
        String[] id = ids.split(",");
        if (this.adminAppVersionService.updateAppVersionStatusById(id, status, force) > 0) {
            return ResultUtil.genSuccessMsg("更新成功");
        }
        return ResultUtil.genFailedResult("更新失败");
    }

    @ApiOperation(value = "1-25-4 app版本信息更新审核状态和版本更新方式", notes = "")

    @PostMapping("/updateStatus")
    public Json updateAppVersionStatusById(AppVersion appVersion) {
        if (this.adminAppVersionService.updateAppVersionStatusById(appVersion) > 0) {
            return ResultUtil.genSuccessMsg("更新成功");
        }
        return ResultUtil.genFailedResult("更新失败");
    }

    @ApiOperation(value = "1-25-5 新增或编辑版本信息", notes = "")
    @PostMapping("/saveOrUpdateAppVersion")
    public Json addAdvertDetail(AppVersion appVersion) {
        if (this.adminAppVersionService.addAppVersionDetail(appVersion) > 0) {
            return ResultUtil.genSuccessMsg("更新成功");
        }
        return ResultUtil.genFailedResult("更新失败");
    }

    @Api(tags = "1-05 [管理]菜单管理", description = "菜单栏相关操作")
    @Controller
    @RequestMapping("api/admin/menu")
    public static class AdminMenuController {

        @Resource
        private AdminMenuService adminMenuService;

        @Resource
        private AdminRoleMenuService adminRoleMenuService;

        @ApiOperation(value = "1-5-1 根据用户角色获取拥有的菜单")
        @ApiImplicitParam(name = "roleId", value = "角色id")
        @GetMapping("getRoleMenu")
        @ResponseBody
        public Json<List<AdminMenuDTO>> getRoleMenu(Integer roleId) {
            if (roleId == null) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数为空");
            }
            return ResultUtil.genSuccessResult(this.adminMenuService.getMenusByRoleId(roleId));
        }

        @ApiOperation(value = "1-5-2 根据角色获取菜单列表")
        @ApiImplicitParam(name = "roleId", value = "角色id")
        @GetMapping("getUserMenu")
        @ResponseBody
        public Json<List<AdminMenuDTO>> getMenuByUserId(Long roleId) {
            List<AdminMenuDTO> allMenus = this.adminMenuService.getMenusByRole(roleId);
            return ResultUtil.genSuccessResult(allMenus);
        }

        /**
         * 获取所有菜单列表
         * @return
         */
        @ApiOperation(value = "1-5-3 获取所有菜单列表")
        @GetMapping("getAllMenu")
        @ResponseBody
        public Json<List<AdminMenuDTO>> getAllMenu() {
            List<AdminMenuDTO> allMenus = this.adminMenuService.getAllMenu();
            return ResultUtil.genSuccessResult(allMenus);
        }

        /**
         * 根据id获取菜单信息
         */
        @ApiOperation(value = "1-5-4 获取菜单详情")
        @ApiImplicitParam(name = "menuId", value = "菜单id")
        @ResponseBody
        @GetMapping("getMenuInfo")
        public Json<AdminMenu> getMenuInfo(@RequestParam(value = "menuId") Integer menuId) {
            AdminMenu menu = this.adminMenuService.selectByPrimaryKey(menuId);
            return ResultUtil.genSuccessResult(menu);
        }

        /**
         * 获取父节点菜单
         * @return
         */
        @ApiOperation(value = "1-5-5 获取父节点菜单")
        @GetMapping("getParentMenuList")
        @ResponseBody
        public Json<List<AdminMenu>> getParentMenuList() {
            return ResultUtil.genSuccessResult(this.adminMenuService.getParentMenus());
        }

        @ApiOperation(value = "1-5-6 保存菜单")
        @RequestMapping(value = "save", method = RequestMethod.POST)
        @ResponseBody
        public Json saveMenu(@RequestBody AdminMenu adminMenu) {
            if (StringUtils.isEmpty(adminMenu.getName())
                    || (adminMenu.getLevel() == 1 && StringUtils.isEmpty(adminMenu.getPath()))) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "菜单名称为空或子菜单路径为空");
            }
            int flag = this.adminMenuService.upsert(adminMenu);
            if (flag == 1 || flag > 1) {
                return ResultUtil.genSuccessMsg();
            }
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "数据库异常");
        }

        @ApiOperation(value = "1-5-7 删除菜单")
        @ApiImplicitParam(name = "menuId", value = "菜单id", required = true)
        @PostMapping(value = "delete")
        @ResponseBody
        public Json delete(Integer menuId) {
            Integer userNum = this.adminRoleMenuService.countByMenuId(menuId);
            if (userNum > 0) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "该菜单关联其他角色，不能删除");
            }
            this.adminMenuService.deleteByPrimaryKey(menuId);
            return ResultUtil.genSuccessMsg();
        }

    }

    @Api(tags = "1-13 [管理]角色管理", description = "角色操作接口")
    @Controller
    @RequestMapping("api/admin/role")
    public static class AdminRoleController extends CommonController {

        @Resource
        private AdminRoleService adminRoleService;

        @Resource
        private AdminUserRoleService adminUserRoleService;

        @Resource
        private AdminMenuService adminMenuService;

        @Resource
        private AdminRoleMenuService adminRoleMenuService;

        @Resource
        private AdminRolePermService adminRolePermService;

        @ApiOperation(value = "1-13-1 获取全部角色", notes = "")
        @GetMapping("list")
        @ResponseBody
        public Json<PageWarper<AdminRoleDTO>> getAllRoleList(RoleParam param) {

            return ResultUtil.genSuccessResult(this.adminRoleService.selectAll(param));
        }

        @ApiOperation(value = "1-13-2 保存角色信息", notes = "")
        @RequestMapping(value = "save", method = RequestMethod.POST)
        @ResponseBody
        public Json saveRole(@RequestBody AdminRole adminRole) {
            AdminUser loginUser = this.getLoginUser();

            if (loginUser == null) {
                ResultUtil.genFailedResult("请先登录");
            }
            //判断唯一标示
            String roleCode = adminRole.getRoleCode();
            if (StringUtils.isEmpty(roleCode)) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "角色唯一标示为空");
            }
            AdminRole adminRole1 = this.adminRoleService.selectByRoleCode(roleCode);

            if (adminRole.getId() == null) {
                if (adminRole1 != null) {
                    return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "角色唯一标示重复");
                }
                adminRole.setCreateTime(new Date());
                adminRole.setCreateUser(loginUser.getId());
                this.adminRoleService.insertSelective(adminRole);
            }

            if (null != adminRole1 && adminRole.getId() != null && null != adminRole.getRoleCode()) {
                if (adminRole1.getRoleCode().equals(adminRole.getRoleCode()) && !adminRole1.getId().equals(adminRole.getId())) {
                    return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "角色唯一标示重复");
                }

                adminRole.setUpdateUser(loginUser.getId());
                adminRole.setUpdateTime(new Date());
                this.adminRoleService.updateByPrimaryKeySelective(adminRole);
            }

            return ResultUtil.genSuccessMsg();
        }

        @ApiOperation(value = "1-13-3 删除角色", notes = "")
        @ApiImplicitParam(name = "roleId", value = "角色id")
        @PostMapping(value = "delete")
        @ResponseBody
        public Json delete(Integer roleId) {
            Integer userNum = this.adminUserRoleService.countByRoleId(roleId);
            if (userNum > 0) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "该角色关联其他用户，不能删除");
            }
            this.adminRoleService.deleteByPrimaryKey(roleId);
            return ResultUtil.genSuccessMsg();
        }

        @ApiOperation(value = "1-13-4 根据角色获取菜单和权限列表", notes = "")
        @ApiImplicitParam(name = "roleId", value = "角色id")
        @GetMapping("getMenuAndPerm")
        @ResponseBody
        public Json<List<MenuDTO>> getMenuAndPerm(Integer roleId) {
            List<MenuDTO> menusAndPermsByRoleId = this.adminMenuService.getMenusAndPermsByRoleId(roleId);

            for (MenuDTO menuDTO : menusAndPermsByRoleId) {
                int i = 0;
                List<MenuDTO> children = menuDTO.getChildren();
                if (children != null && !children.isEmpty()) {
                    for (MenuDTO child : children) {
                        if (child.getChecked() == 1) {
                            i++;
                        }
                    }
                    if (children.size() != i) {
                        menuDTO.setChecked(0);
                    }
                }
            }
            return ResultUtil.genSuccessResult(menusAndPermsByRoleId);
        }

        /**
         * 保存角色拥有菜单和权限
         * @param menuAndPermDTO
         * @return
         */
        @ApiOperation(value = "1-13-5 保存角色拥有菜单和权限", notes = "")
        @RequestMapping(value = "saveMenuAndPerm", method = RequestMethod.POST)
        @ResponseBody
        public Json saveMenuAndPerm(@RequestBody MenuAndPermDTO menuAndPermDTO) {

            Integer roleId = menuAndPermDTO.getRoleId();
            if (roleId == null) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "角色id为空");
            }

            AdminRole adminRole = this.adminRoleService.selectByPrimaryKey(roleId);

            if (adminRole != null) {

                //清空角色对应的菜单和权限
                this.adminRoleMenuService.deleteByRoleId(roleId);
                //重新保存
                for (Integer menuId : menuAndPermDTO.getMenuIds()) {
                    AdminRoleMenu adminRoleMenu = new AdminRoleMenu();
                    adminRoleMenu.setMenuId(menuId);
                    adminRoleMenu.setRoleId(roleId);
                    this.adminRoleMenuService.insertSelective(adminRoleMenu);
                }

                this.adminRolePermService.deleteByRoleId(roleId);
                for (Integer permId : menuAndPermDTO.getPermIds()) {
                    AdminRolePerm adminRolePerm = new AdminRolePerm();
                    adminRolePerm.setPermId(permId);
                    adminRolePerm.setRoleId(roleId);
                    this.adminRolePermService.insertSelective(adminRolePerm);
                }
                return ResultUtil.genSuccessMsg();
            }
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "角色不存在");
        }

    }
}
