package com.bxm.localnews.admin.controller.news;

import com.bxm.localnews.admin.dto.AccountQueryDto;
import com.bxm.localnews.admin.dto.ContentDTO;
import com.bxm.localnews.admin.enums.ShowLevelEnum;
import com.bxm.localnews.admin.param.*;
import com.bxm.localnews.admin.service.news.AdminNewsService;
import com.bxm.localnews.admin.service.recommend.MixRecommendPoolService;
import com.bxm.localnews.admin.vo.News;
import com.bxm.localnews.admin.vo.NewsKind;
import com.bxm.localnews.admin.vo.NewsOverView;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.controller.BaseController;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import java.util.List;

@Api(tags = "1-07 [管理]新闻管理", description = "新闻上线管理，防止受管制的内容出现在线上")
@RestController
@RequestMapping("api/admin/news")
public class AdminNewsController extends BaseController {

    @Resource
    private AdminNewsService adminNewsService;

    @Resource
    private MixRecommendPoolService mixRecommendPoolService;

    @ApiOperation(value = "1-7-1 获取新闻列表", notes = "支持id，标题，内容，状态，标签，时间的等查询")
    @PostMapping("/list")
    public Json<PageWarper<NewsOverView>> getNews(NewsParam newsParam) {
        if (newsParam.getPageNum() != null && newsParam.getPageNum() * newsParam.getPageSize() > 10000) {
            return ResultUtil.genFailedResult("亲,暂不支持数据量10000以上的分页哟，请使用搜索功能查找新闻，如有硬性要求请联系沈涛");
        }
        return ResultUtil.genSuccessResult(this.adminNewsService.queryNews(newsParam));
    }

    @ApiOperation("1-7-4 获取新闻详情")
    @GetMapping("/getNewsDetail")
    public Json<News> getNewsDetail(Long id) {
        return ResultUtil.genSuccessResult(this.adminNewsService.selectByPrimaryKey(id));
    }

    @ApiOperation("1-7-5 新闻置顶")
    @Deprecated
    @PostMapping("/topping")
    public Json toppingNews(Long id) {
        return adminNewsService.settingLevel(ShowLevelEnum.TOP, id);
    }

    @ApiOperation("1-7-6 将新闻置为热门")
    @Deprecated
    @PostMapping("/popularize")
    public Json popularizeNews(Long id) {
        return adminNewsService.settingLevel(ShowLevelEnum.HOT, id);
    }


    @ApiOperation("1-7-7 新增新闻")
    @PostMapping("/add")
    public Json add(NewsAddParam param) {
        return adminNewsService.addNews(param);
    }

    @ApiOperation("1-7-8 修改新闻")
    @PostMapping("/update")
    public Json update(NewsUpdateParam param) {
        return adminNewsService.updateNews(param);
    }

    @ApiOperation("1-7-9 获取所有新闻频道[包括显示与隐藏]")
    @GetMapping("/getAllKind")
    public Json<List<NewsKind>> getAllKind() {
        List<NewsKind> list = adminNewsService.getAllKind();
        return ResultUtil.genSuccessResult(list);
    }

    @ApiOperation("1-7-10 根据文章链接抓取文章内容")
    @PostMapping("/getSpiderContent")
    public Json<ContentDTO> getSpiderContent(@RequestBody CrawlingParam crawlingParam) {
        return ResultUtil.genSuccessResult(adminNewsService.getSpiderContent(crawlingParam));
    }

    @ApiOperation("1-7-11 添加待爬取的公众号 区域为全国时：区域编码传0、区域名称传全国、分类必传。区域为指定区域时，分类不传")
    @PostMapping("/addAccount")
    public Json addAccount(@RequestBody AccountAddParam accountAddParam) {
        Message message = adminNewsService.addAccount(accountAddParam);
        if (message.isSuccess()) {
            return ResultUtil.genSuccessResult(message.getLastMessage());
        }

        return ResultUtil.genFailedResult(message.getLastMessage());
    }

    @ApiOperation("1-7-12 新闻通过审核")
    @PostMapping("/review")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "id", required = true),
            @ApiImplicitParam(name = "review", value = "是否通过审核 0:待审核  1:已拒绝  2:已通过  传2代表审核通过", required = true)
    })
    public Json review(Long id, Byte review) {
        adminNewsService.review(id, review);
        return ResultUtil.genSuccessMsg("更新成功");
    }

    @ApiOperation("1-7-13 新闻批量通过审核")
    @PostMapping("/batchReview")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "ids", value = "id字符串，以”,“分隔", required = true),
            @ApiImplicitParam(name = "review", value = "是否通过审核 0:待审核  1:已拒绝  2:已通过", required = true)
    })
    public Json batchReview(String ids, Byte review) {
        String[] idArray = ids.split(",");
        if (idArray.length > 0) {
            adminNewsService.batchReview(ids, review);
            return ResultUtil.genSuccessResult("更新成功");
        } else {
            return ResultUtil.genFailedResult("请选择新闻");

        }
    }

    @ApiOperation(value = "1-7-15 获取新增新闻id")
    @GetMapping("/getCreateId")
    public Json<Long> getCreateId() {
        return ResultUtil.genSuccessResult(adminNewsService.getCreateId());
    }

    @ApiOperation(value = "1-7-16 新闻添加到推荐库")
    @PostMapping("/addNewsToMixRecommendPool")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "新闻id", required = true),
            @ApiImplicitParam(name = "origin", value = "类型 1：社区帖子2：本地新闻 3：全国新闻", required = true),
    })
    public Json addNewsToMixRecommendPool(MixPoolParam mixPoolParam) {
        Message i = mixRecommendPoolService.addMixRecommendPool(mixPoolParam);
        if (!i.isSuccess()) {
            return ResultUtil.genFailedResult(i.getLastMessage());
        } else {
            return ResultUtil.genSuccessMsg("添加成功");
        }
    }

    @ApiOperation(value = "1-7-17 将新闻从新闻库中删除")
    @PostMapping("/deleteNewsById")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "id", value = "新闻id", required = true)
    })
    public Json deleteNewsById(Long id) {
        adminNewsService.deleteNewsById(id);
        return ResultUtil.genSuccessMsg("删除成功!");
    }

    @ApiOperation(value = "1-7-18 分页查询待抓取公众号列表")
    @GetMapping("/listAccount")
    public Json<PageWarper<AccountQueryDto>> queryAccountByName(AccountQueryParam accountQueryParam) {
        return ResultUtil.genSuccessResult(adminNewsService.listWechatAccount(accountQueryParam));
    }

    @ApiOperation(value = "1-7-19 删除待抓取公众号")
    @PostMapping("/deleteAccount")
    @ApiImplicitParam(name = "id", value = "公众号主键id", required = true)
    public Json deleteById(Long id) {
        adminNewsService.deleteWechatAccountById(id);
        return ResultUtil.genSuccessMsg("删除成功!");
    }

}

