package com.bxm.localnews.admin.controller.security;

import com.bxm.localnews.admin.controller.CommonController;
import com.bxm.localnews.admin.dto.AdminUserDTO;
import com.bxm.localnews.admin.dto.CurrentUserVo;
import com.bxm.localnews.admin.dto.LoginSuccessWarper;
import com.bxm.localnews.admin.param.AdminUserParam;
import com.bxm.localnews.admin.service.security.AdminRoleService;
import com.bxm.localnews.admin.service.security.AdminUserRoleService;
import com.bxm.localnews.admin.service.security.AdminUserService;
import com.bxm.localnews.admin.service.security.JwtTokenService;
import com.bxm.localnews.admin.vo.security.AdminRole;
import com.bxm.localnews.admin.vo.security.AdminUser;
import com.bxm.localnews.admin.vo.security.AdminUserRole;
import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.Json;
import com.bxm.newidea.component.tools.MD5Util;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler;
import org.springframework.web.bind.annotation.*;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.util.Date;
import java.util.List;
import java.util.stream.Collectors;

/**
 * Created by hanshuaiqi 13:33 2017/9/22
 */
@Api(tags = "1-16 [管理]后台用户管理接口", description = "包含用户登录、登出等功能")
@RestController
@RequestMapping("api/admin")
public class AdminUserController extends CommonController {

    @Resource
    private AdminUserService adminUserService;

    @Resource
    private AdminRoleService adminRoleService;

    @Resource
    private AdminUserRoleService adminUserRoleService;

    @Resource
    private JwtTokenService jwtTokenService;

    @ApiOperation(value = "1-16-1 登出", notes = "")
    @ApiImplicitParam(name = "userName", value = "登出的用户账号")
    @PostMapping(value = "/logout")
    public Json logout(@RequestParam String userName, HttpServletRequest request, HttpServletResponse response) {
        Authentication auth = SecurityContextHolder.getContext().getAuthentication();
        if (auth != null) {
            new SecurityContextLogoutHandler().logout(request, response, auth);
        }
        //清除缓存中的用户信息
        adminUserService.clearCacheUser(userName);
        return ResultUtil.genSuccessResult();
    }

    @ApiOperation(value = "1-16-2 用户登录", notes = "后续此token必须在请求中携带")
    @PostMapping(value = "toLoginNew")
    public Json<LoginSuccessWarper> loginAjax(AdminUserDTO sysUser) {

        if (StringUtils.isEmpty(sysUser.getUsername()) ||
                StringUtils.isEmpty(sysUser.getPassword())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数错误,用户名、密码必须输入");
        }

        AdminUser cacheUser = this.adminUserService.getCacheUser(sysUser.getUsername());
        if (cacheUser.getStatus() != 0) {
            return ResultUtil.genFailedResult("该用户已被禁用");
        }

        //判断用户名、密码
        String encodePwd = MD5Util.hgmd5(sysUser.getPassword());
        if (null == cacheUser || StringUtils.notEquals(encodePwd, cacheUser.getPassword())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "用户名、密码错误");
        }

        //创建token
        String token = this.jwtTokenService.generateToken(cacheUser);

        LoginSuccessWarper warper = new LoginSuccessWarper();
        warper.setToken(token);
        warper.setUserName(cacheUser.getUsername());
        warper.setRealName(cacheUser.getName());
        List<String> roleCodes = adminRoleService.selectByUserId(cacheUser.getId()).stream().map(AdminRole::getRoleCode)
                .collect(Collectors.toList());
        warper.setRoleList(roleCodes);

        return ResultUtil.genSuccessResult(warper).setErrorMsg("登陆成功");
    }

    @ApiOperation(value = "1-16-3 修改密码", notes = "")
    @RequestMapping(value = "modifyPassword", method = RequestMethod.POST)
    @ResponseBody
    public Json modifyPassword(AdminUserDTO sysUser) {
        return this.adminUserService.modifyPassword(sysUser);
    }

    @ApiOperation(value = "1-16-4 管理后台用户列表", notes = "")
    @GetMapping("list")
    @ResponseBody
    public Json<PageWarper<AdminUserDTO>> getAdminUserList(AdminUserParam adminUserParam) {

        return ResultUtil.genSuccessResult(this.adminUserService.queryAdminUsers(adminUserParam));
    }

    @ApiOperation(value = "1-16-5 保存用户", notes = "")
    @RequestMapping(value = "save", method = RequestMethod.POST)
    @ResponseBody
    public Json saveAdminUser(AdminUserDTO adminUserDTO) {
        if (StringUtils.isEmpty(adminUserDTO.getName())
                || StringUtils.isEmpty(adminUserDTO.getUsername())
                || StringUtils.isEmpty(adminUserDTO.getPhone())
                || StringUtils.isEmpty(adminUserDTO.getRoleId())) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数为空");
        }

        //判断用户名是否存在
        AdminUser adminUser = this.adminUserService.selectByUsername(adminUserDTO.getUsername().trim());
        if (adminUserDTO.getId() != null) {
            if (adminUser != null) {
                if (!adminUser.getId().equals(adminUserDTO.getId())) {
                    return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "用户名重复");
                }
            }
            adminUserDTO.setUpdateTime(new Date());
            adminUserDTO.setUpdateUser("system");
            this.adminUserService.updateByPrimaryKeySelective(adminUserDTO);
        } else {
            if (adminUser != null) {
                return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "用户名重复");
            }
            this.adminUserService.upsert(adminUserDTO);
        }

        //增加角色
        String[] roleIds = adminUserDTO.getRoleId().split(",");
        this.adminUserRoleService.deleteByUserId(adminUserDTO.getId());

        for (String roleId : roleIds) {
            Integer intRoleId = Integer.parseInt(roleId);
            AdminRole adminRole = this.adminRoleService.selectByPrimaryKey(intRoleId);
            if (adminRole != null) {
                AdminUserRole adminUserRole = new AdminUserRole();
                adminUserRole.setRoleId(intRoleId);
                adminUserRole.setUserId(adminUserDTO.getId());
                this.adminUserRoleService.insertSelective(adminUserRole);
            }
        }
        return ResultUtil.genSuccessMsg();
    }

    @ApiOperation(value = "1-16-6 锁定或解锁用户", notes = "")
    @RequestMapping(value = "updateUserStatus", method = RequestMethod.POST)
    @ResponseBody
    public Json updateUserStatus(AdminUser user) {

        if (user.getId() == null || user.getStatus() == null) {
            return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "参数为空");
        }
        AdminUser adminUser = new AdminUser();
        adminUser.setId(user.getId());
        adminUser.setStatus(user.getStatus());
        this.adminUserService.updateByPrimaryKeySelective(adminUser);

        return ResultUtil.genSuccessMsg();
    }

    @ApiOperation(value = "1-16-7 获取当前用户信息和权限", notes = "")
    @GetMapping(value = "getCurrentUserInfo")
    @ResponseBody
    public Json<CurrentUserVo> getCurrentUserInfo() {

        AdminUser adminUser = this.getLoginUser();
        if (adminUser != null) {
            CurrentUserVo currentUserVo = new CurrentUserVo(this.adminRoleService.selectByUserId(adminUser.getId()),
                    adminUser.getId(), adminUser.getUsername(),
                    adminUser.getPhone());
            return ResultUtil.genSuccessResult(currentUserVo);
        }
        return ResultUtil.genFailedResult(RespCode.BAD_REQUEST, "请重新登录");
    }

    @ApiOperation(value = "1-16-8 修改用户密码（管理员专用）", notes = "")
    @PostMapping(value = "/password")
    @ResponseBody
    public Json modifyPasswordByAdmin(AdminUserDTO sysUser) {
        return this.adminUserService.modifyPasswordByAdmin(sysUser);
    }


    @ApiOperation(value = "1-16-9 修改用户角色", notes = "")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "用户id", value = "userId"),
            @ApiImplicitParam(name = "角色id", value = "roleId")
    })
    @PostMapping(value = "/updateUserRole")
    @ResponseBody
    public Json updateUserRole(Long userId, Integer roleId) {
        this.adminUserService.updateUserRole(userId, roleId);
        return ResultUtil.genSuccessResult();
    }


    @ApiOperation(value = "1-16-10 获取管理后台所有可用账号列表", notes = "")
    @GetMapping("user/list")
    public Json<List<AdminUserDTO>> getUserList() {
        return ResultUtil.genSuccessResult(adminUserService.getAvailableAdminUser());
    }

}








