package com.bxm.localnews.admin.service.impl;

import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.AdminAdvertAreaMapper;
import com.bxm.localnews.admin.domain.AdminAdvertMapper;
import com.bxm.localnews.admin.dto.AdvertDTO;
import com.bxm.localnews.admin.param.AdvertParam;
import com.bxm.localnews.admin.service.AdminAdvertService;
import com.bxm.localnews.admin.vo.Advert;
import com.bxm.localnews.admin.vo.AdvertArea;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2018/10/19 14:43
 * @desc
 */
@Service
public class AdminAdvertServiceImpl implements AdminAdvertService {

    private AdminAdvertMapper adminAdvertMapper;

    private AdminAdvertAreaMapper adminAdvertAreaMapper;

    private SequenceCreater sequenceCreater;

    private RedisStringAdapter redisStringAdapter;

    @Autowired
    public AdminAdvertServiceImpl(AdminAdvertMapper adminAdvertMapper, AdminAdvertAreaMapper adminAdvertAreaMapper,
                                  SequenceCreater sequenceCreater, RedisStringAdapter redisStringAdapter) {
        this.adminAdvertMapper = adminAdvertMapper;
        this.adminAdvertAreaMapper = adminAdvertAreaMapper;
        this.sequenceCreater = sequenceCreater;
        this.redisStringAdapter = redisStringAdapter;
    }

    @Override
    public PageWarper<AdvertDTO> getAdvertList(AdvertParam advertParam) {

        PageWarper<AdvertDTO> pageWarper = new PageWarper<>(this.adminAdvertMapper.queryAdverts(advertParam));

        pageWarper.getList().forEach(advertDTO -> {
            if (advertDTO.getGlobalFlag() != null && advertDTO.getGlobalFlag() == 0) {
                advertDTO.setAreaList(this.adminAdvertAreaMapper.getAdvertAreaDTOByadvertId(advertDTO.getId()));
            }
        });
        return pageWarper;
    }

    @Override
    public AdvertDTO getAdvertById(Long id) {
        Advert advert = this.adminAdvertMapper.selectByPrimaryKey(id);

        return convertAdvert(advert);
    }

    @Override
    public int updateAdvertStatusById(String[] ids, Byte status) {
        return this.adminAdvertMapper.updateAdvertStatusById(ids, status);
    }

    @Override
    public int updateAdvertStatusById(Long id, Byte status) {
        Advert advert = this.adminAdvertMapper.selectByPrimaryKey(id);
        if (advert != null) {
            advert.setStatus(status);
            if (StringUtils.isNotBlank(advert.getType())) {
                redisStringAdapter.remove(getAdvertListKey(Byte.valueOf(advert.getType())));
            }
        }

        return this.adminAdvertMapper.updateByPrimaryKeySelective(advert);
    }

    @Override
    public int addAdvertDetail(Advert advert, String areaCodes) {

        Long advertId = advert.getId();
        if (advertId == null) {
            advertId = sequenceCreater.nextLongId();
        }

        updateAdvertArea(areaCodes, advertId);

        int update;
        if (advert.getId() == null) {
            advert.setId(advertId);
            advert.setAddTime(new Date());
            update = this.adminAdvertMapper.insertSelective(advert);
        } else {
            update = this.adminAdvertMapper.updateByPrimaryKeySelective(advert);
        }

        if (StringUtils.isNotBlank(advert.getType())) {
            redisStringAdapter.remove(this.getAdvertListKey(Byte.valueOf(advert.getType())));
        }

        return update;
    }

    @Override
    public int deleteAdvert(Long id) {
        Advert advert = this.adminAdvertMapper.selectByPrimaryKey(id);

        if (StringUtils.isNotBlank(advert.getType())) {
            redisStringAdapter.remove(this.getAdvertListKey(Byte.valueOf(advert.getType())));
        }

        return this.adminAdvertMapper.deleteByPrimaryKey(advert.getId());
    }

    /**
     * 更新对应地区信息
     *
     * @param areaCodes
     * @param advertId
     */
    private void updateAdvertArea(String areaCodes, Long advertId) {
        if (StringUtils.isBlank(areaCodes)) {
            return;
        }
        this.adminAdvertAreaMapper.deleteByAdvertId(advertId);

        insertAdvertArea(areaCodes, advertId);

    }

    /**
     * 添加地区信息
     *
     * @param areaCodes
     * @param advertId
     */
    private void insertAdvertArea(String areaCodes, Long advertId) {
        String[] areaCode = StringUtils.split(areaCodes, ",");
        List<AdvertArea> advertAreaList = new ArrayList<>(areaCode.length);
        for (String s : areaCode) {
            AdvertArea advertArea = new AdvertArea();
            advertArea.setAdvertId(advertId);
            advertArea.setCode(s);
            advertArea.setId(sequenceCreater.nextLongId());
            advertAreaList.add(advertArea);
        }
        this.adminAdvertAreaMapper.insertAdvertAreas(advertAreaList);
    }

    /**
     * 获取广告类列表的key
     *
     * @param type
     * @return
     */
    private KeyGenerator getAdvertListKey(Byte type) {
        return RedisConfig.THIRDPARTY_TYPE_ADVERT.copy().appendKey(type).appendKey("list");
    }

    private KeyGenerator getAllAdvertListKey() {
        return RedisConfig.THIRDPARTY_TYPE_ADVERT.copy().appendKey("*").appendKey("list");
    }

    private AdvertDTO convertAdvert(Advert advert) {
        AdvertDTO advertDTO = new AdvertDTO();
        BeanUtils.copyProperties(advert, advertDTO);
        if (advert.getGlobalFlag() == 0) {
            advertDTO.setAreaList(this.adminAdvertAreaMapper.getAdvertAreaDTOByadvertId(advert.getId()));
        }
        return advertDTO;
    }
}
