package com.bxm.localnews.admin.service.impl;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.AreaDivisionMapper;
import com.bxm.localnews.admin.domain.LocationMapper;
import com.bxm.localnews.admin.dto.LocationDTO;
import com.bxm.localnews.admin.param.LocationParam;
import com.bxm.localnews.admin.service.LocationService;
import com.bxm.localnews.admin.vo.AreaDivision;
import com.bxm.localnews.admin.vo.Location;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.PinyinUtils;
import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 地区相关接口实现类
 */
@Service
public class LocationServiceImpl implements LocationService {


    private final LocationMapper locationMapper;

    private final AreaDivisionMapper areaDivisionMapper;

    private final RedisStringAdapter redisStringAdapter;

    @Autowired
    public LocationServiceImpl(LocationMapper locationMapper, AreaDivisionMapper areaDivisionMapper, RedisStringAdapter redisStringAdapter) {
        this.locationMapper = locationMapper;
        this.areaDivisionMapper = areaDivisionMapper;
        this.redisStringAdapter = redisStringAdapter;
    }

    @Override
    public List<LocationDTO> listLocation(LocationParam locationParam) {
        Location location = new Location();
        BeanUtils.copyProperties(locationParam, location);
        List<Location> locationList = locationMapper.selectByModel(location);
        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        return locationDTOList;
    }

    @Override
    public void saveHotLocation(String code) {
        Location location = locationMapper.selectByCode(code);
        if (location.getHot().equals((byte) 0)) {
            location.setHot((byte) 1);
            location.setShow((byte) 1);
            int count = locationMapper.updateByCodeSelective(location);
            if (count == 1) {
                List<LocationDTO> locationDTOList = listHotLocation();
                redisStringAdapter.set(RedisConfig.HOT_LOCATION, locationDTOList);
            }
        }
    }

    private List<LocationDTO> listHotLocation() {
        Location locationParam = new Location();
        locationParam.setHot((byte) 1);
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);
        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        return locationDTOList;
    }

    private List<LocationDTO> listOpenedLocation() {
        Location locationParam = new Location();
        locationParam.setShow((byte) 1);
        List<Location> locationList = locationMapper.selectByModel(locationParam);


        List<LocationDTO> locationDTOList = locationList.stream().map(this::convertLocation).collect(Collectors.toList());
        //填充首字母
        locationDTOList.forEach(x -> x.setPinyin(PinyinUtils.getPinYinHeadChar(x.getName()).substring(0, 1).toUpperCase()));

        return locationDTOList;
    }

    @Override
    public void saveOpenedLocation(String code) {
        Location location = locationMapper.selectByCode(code);
        if (location.getShow().equals((byte) 0)) {
            location.setShow((byte) 1);
            int count = locationMapper.updateByCodeSelective(location);
            if (count == 1) {
                List<LocationDTO> locationDTOList = listOpenedLocation();
                redisStringAdapter.set(RedisConfig.OPENED_LOCATION, locationDTOList);
            }
        }
    }

    @Override
    public List<LocationDTO> listOpenedLocationByArea(String code, Byte show) {
        List<Location> locationList = new ArrayList<>();
        Location location = locationMapper.selectByCode(code);
        List<Location> locationListByParent = locationMapper.selectByParentCode(code, show);
        if (null != location) {
            locationList.add(location);
        }
        if (CollectionUtils.isNotEmpty(locationListByParent)) {
            locationList.addAll(locationListByParent);
        }
        return locationList.stream().map(this::convertLocation).collect(Collectors.toList());
    }

    @Override
    public List<LocationDTO> listOpenedLocationByAreaWithTwoLevels(String code, Byte show) {
        List<LocationDTO> locationDTOList = new ArrayList<>();
//        根据省份信息得到所有已开通城市
        AreaDivision areaDivisionParam = new AreaDivision();
        areaDivisionParam.setParentCode(code);
        List<AreaDivision> areaDivisionList = areaDivisionMapper.selectByModel(areaDivisionParam);
        for (AreaDivision areaDivision : areaDivisionList) {
            List<LocationDTO> temLocationDTOList = listOpenedLocationByArea(areaDivision.getCode(), show);
            locationDTOList.addAll(temLocationDTOList);
        }

        return locationDTOList;
    }

    private LocationDTO convertLocation(Location location) {
        LocationDTO locationDTO = new LocationDTO();
        locationDTO.setCode(location.getCode());
        locationDTO.setHot(location.getHot());
        locationDTO.setShow(location.getShow());
        locationDTO.setLevel(location.getLevel());
        locationDTO.setName(location.getName());
        locationDTO.setPinyin(location.getPinyin());
        return locationDTO;
    }

    private LocationDTO convertLocationByDivision(AreaDivision areaDivision) {
        LocationDTO locationDTO = new LocationDTO();
        locationDTO.setCode(areaDivision.getCode());
        locationDTO.setLevel(areaDivision.getLevel());
        locationDTO.setName(areaDivision.getName());
        return locationDTO;
    }

}
