package com.bxm.localnews.admin.service.strategy;

import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.stream.Collectors;

import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.localnews.admin.config.AdminBizConfigProperties;
import com.bxm.localnews.admin.config.BizConfigProperties;
import com.bxm.localnews.admin.constant.PrivilegeStatusEnum;
import com.bxm.localnews.admin.domain.CommodityCodeMapper;
import com.bxm.localnews.admin.domain.PrivilegeMapper;
import com.bxm.localnews.admin.domain.WinnerMapper;
import com.bxm.localnews.admin.service.CommodityService;
import com.bxm.localnews.admin.service.PushMsgIntegService;
import com.bxm.localnews.admin.service.ShortLinkService;
import com.bxm.localnews.admin.service.SmsIntegService;
import com.bxm.localnews.admin.vo.CommodityCodeBean;
import com.bxm.localnews.admin.vo.ParticipantBean;
import com.bxm.localnews.admin.vo.PrivilegeBean;
import com.bxm.localnews.admin.vo.WinnerBean;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.model.dto.PushGroupMessage;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.SmsSupplyDTO;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.google.common.collect.Lists;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;

/**
 * 抽象的抽奖策略，处理抽奖处理中的公共逻辑
 */
public abstract class AbstractDrawStrategyService extends BaseService implements DrawStrategyService {

    @Autowired
    private PrivilegeMapper privilegeMapper;

    @Autowired
    private CommodityCodeMapper commodityCodeMapper;

    @Autowired
    private WinnerMapper winnerMapper;

    @Autowired
    private AdminBizConfigProperties adminBizConfigProperties;

    @Autowired
    private BizConfigProperties bizConfigProperties;

    @Autowired
    private ShortLinkService shortLinkService;

    @Autowired
    private CommodityService commodityService;

    @Autowired
    private PushMsgIntegService pushMsgIntegService;

    @Autowired
    private SmsIntegService smsIntegService;

    @Override
    public Message execDraw(PrivilegeBean privilege) {
        //判断活动状态是否为进行中
        if (privilege.getStatus() == PrivilegeStatusEnum.BEGIN.getCode()) {
            //执行抽奖逻辑
            Message message = draw(privilege);

            if (message.isSuccess()) {
                //抽奖成功，变更活动状态为已完成
                privilegeMapper.changeStatusAndActualNum(privilege.getId(), PrivilegeStatusEnum.FINISH.getCode(), privilege.getActualNum());
            }

            return message;
        }
        return Message.build(false).setMessage("活动[" + privilege.getId() + "]状态不是进行中，不进行处理");
    }

    /**
     * 执行具体的抽奖逻辑
     *
     * @param privilege 活动详情
     * @return 抽奖执行结果
     */
    abstract Message draw(PrivilegeBean privilege);

    /**
     * 保存获奖人员信息
     *
     * @param privilege       活动信息
     * @param winnerSet       获奖人员信息
     * @param relationWinners 关联中奖人员信息
     * @param participants    未中奖人员名单
     */
    void saveWinner(PrivilegeBean privilege, Set<Long> winnerSet, List<ParticipantBean> relationWinners, List<ParticipantBean> participants) {
        if (winnerSet == null || winnerSet.size() == 0) {
            commodityService.execStoreReturned(privilege.getCommodityId(), privilege.getPrizeNum(), privilege.getId());
            return;
        }

        List<WinnerBean> winners = Lists.newArrayList();

        //根据活动对应的商品ID获取商品编码列表
        List<CommodityCodeBean> prizeList = commodityCodeMapper.getUnusedCommodities(privilege.getCommodityId(), winnerSet.size());

        int prizeCount = prizeList.size();
        if (prizeCount > 0 && winnerSet.size() != prizeCount) {
            logger.error("奖品数量不足以发放奖品，存在业务逻辑错误，仅发放剩余的奖品，活动ID[{}],商品ID[{}]",
                    privilege.getId(),
                    privilege.getCommodityId());
        }

        //是否虚拟商品，虚拟商品将使用用户的手机号码作为中奖码
        boolean virtualPrize = prizeCount == 0;

        int index = 0;
        WinnerBean winner;
        for (Long winnerId : winnerSet) {
            if (!virtualPrize && prizeCount == 0) {
                break;
            }

            winner = new WinnerBean();

            Optional<ParticipantBean> relation = relationWinners.stream()
                    .filter(entity -> Objects.equals(entity.getInviteUserId(), winnerId))
                    .findFirst();

            if (relation.isPresent()) {
                ParticipantBean relationBean = relation.get();
                winner.setInviteFlag(1);
                winner.setInviteUserId(relationBean.getInviteUserId());
            } else {
                winner.setInviteFlag(0);
            }

            winner.setId(nextId());
            winner.setUserId(winnerId);
            winner.setPrivilegeId(privilege.getId());
            winner.setRedeemFlag(0);
            winner.setAddTime(new Date());

            if (!virtualPrize) {
                winner.setCommodityCode(prizeList.get(index).getCode());
            }

            winners.add(winner);
            index++;
            prizeCount--;
        }

        new BatchHelper<WinnerMapper, WinnerBean>(WinnerMapper.class, winners) {
            @Override
            protected int invoke(WinnerBean element) {
                return mapper.insert(element);
            }
        };

        //如果是虚拟商品则使用用户的手机号码作为中奖码
        if (virtualPrize) {
            winnerMapper.updateWinnerPrizeCode(privilege.getId());
        } else {
            //退还差额库存
            if (winners.size() < privilege.getPrizeNum()) {
                int num = privilege.getPrizeNum() - winners.size();
                commodityService.execStoreReturned(privilege.getCommodityId(), num, privilege.getId());
            }
        }

        //发送消息通知
        sendMessage(privilege, winners, participants);
    }

    /**
     * 用户中奖后给用户发送中奖消息（短消息和个推）
     * 给未中奖用户发送开奖信息
     *
     * @param privilege    活动详情
     * @param winners      中奖人员名单
     * @param participants 未中奖人员名单
     */
    private void sendMessage(PrivilegeBean privilege, List<WinnerBean> winners, List<ParticipantBean> participants) {
        if (adminBizConfigProperties.isPrivilegePushMsg()) {
            //发送中奖短信
            sendWinnerPush(privilege, winners);
            //给所有参与者发送开奖信息
            sendParticipantPush(privilege, participants);
        } else {
            logger.info("活动[{}]已发奖", privilege);
            logger.info("获奖人员：[{}]", winners);
            logger.info("未获奖的参与人人员：[{}]", participants);
        }
    }

    private void sendWinnerPush(PrivilegeBean privilege, List<WinnerBean> winners) {
        List<String> winnerIds = winners.stream().map(winner -> winner.getUserId().toString()).collect(Collectors.toList());

        //发送中奖通知(端内推送)
        String content = "恭喜你抽中【" + privilege.getTitle() + "】，今天最幸运的就是你！";

        PushMessage message = PushMessage.build("中奖提醒", content);
        PushPayloadInfo payloadInfo = PushPayloadInfo.build(PushMessageEnum.PRIVILEGE_WINNING);
        payloadInfo.addExtend("privilegeId", privilege.getId() + "");

        message.setPayloadInfo(payloadInfo);
        PushGroupMessage pushGroupMessage = new PushGroupMessage();
        pushGroupMessage.setUserIds(winnerIds);
        pushGroupMessage.setPushMessage(message);
        pushMsgIntegService.pushGroupMsg(pushGroupMessage);

        //获取中奖人员名单对应的手机号码，发送中奖短信
        List<WinnerBean> winnerWithPhone = winnerMapper.getWinner(privilege.getId());
        if (CollectionUtils.isNotEmpty(winnerWithPhone)) {
            for (WinnerBean winner : winnerWithPhone) {
                String smsContent = "恭喜你抽中[" + privilege.getTitle() + "]啦，点击 " + getH5PrivilegeDetailUrl(privilege, winner) + " 领取";
                SmsSupplyDTO smsSupplyDTO = new SmsSupplyDTO();
                smsSupplyDTO.setPhoneNo(winner.getPhone());
                smsSupplyDTO.setContent(smsContent);
                smsIntegService.sendSmsByCustomize(smsSupplyDTO);
            }
        }
    }

    private String getH5PrivilegeDetailUrl(PrivilegeBean privilege, ParticipantBean winner) {
        try {
            String afterEncodeUserName = winner.getPhone();
            if (StringUtils.isNotBlank(winner.getUserName())) {
                afterEncodeUserName = URLEncoder.encode(winner.getUserName(), StandardCharsets.UTF_8.name());
            }
            String url = bizConfigProperties.getH5ServerHost() +
                    "/shareDrawActivity.html?id=" + privilege.getId() +
                    "&userId=" + winner.getUserId() +
                    "&isLogin=1" +
                    "&userName" + afterEncodeUserName +
                    "&userUrl=" + winner.getUserImg();

            return shortLinkService.getShortUrl(url, true);
        } catch (UnsupportedEncodingException e) {
            logger.error(e.getMessage(), e);
        }
        return null;
    }

    private void sendParticipantPush(PrivilegeBean privilege, List<ParticipantBean> participants) {
        List<String> participantIds = participants.stream().map(participant -> participant.getUserId().toString()).collect(Collectors.toList());
        String content = "【" + privilege.getTitle() + "】开奖了，快来看看幸运儿是谁！";

        PushMessage message = PushMessage.build("开奖提醒", content);
        PushPayloadInfo payloadInfo = PushPayloadInfo.build(PushMessageEnum.PRIVILEGE_OPEN_PRIZE);
        payloadInfo.addExtend("privilegeId", privilege.getId() + "");

        message.setPayloadInfo(payloadInfo);
        PushGroupMessage pushGroupMessage = new PushGroupMessage();
        pushGroupMessage.setUserIds(participantIds);
        pushGroupMessage.setPushMessage(message);
        pushMsgIntegService.pushGroupMsg(pushGroupMessage);
    }

}
