package com.bxm.localnews.admin.service.impl;

import java.util.ArrayList;
import java.util.List;

import com.bxm.localnews.admin.constant.RedisConfig;
import com.bxm.localnews.admin.domain.AdminToolsLocationMapper;
import com.bxm.localnews.admin.domain.AdminToolsMapper;
import com.bxm.localnews.admin.dto.ToolsDTO;
import com.bxm.localnews.admin.param.ToolsParam;
import com.bxm.localnews.admin.service.AdminToolsService;
import com.bxm.localnews.admin.vo.Tools;
import com.bxm.localnews.admin.vo.ToolsLocation;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.PageWarper;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

/**
 * @author zhaoyadong 2018/10/23 13:58
 * @desc
 */
@Service
public class AdminToolsServiceImpl implements AdminToolsService {

    private AdminToolsMapper adminToolsMapper;

    private AdminToolsLocationMapper adminToolsLocationMapper;

    private RedisHashMapAdapter redisHashMapAdapter;

    private SequenceCreater sequenceCreater;

    @Autowired
    public AdminToolsServiceImpl(AdminToolsMapper adminToolsMapper, AdminToolsLocationMapper adminToolsLocationMapper,
                                 RedisHashMapAdapter redisHashMapAdapter, SequenceCreater sequenceCreater) {
        this.adminToolsMapper = adminToolsMapper;
        this.adminToolsLocationMapper = adminToolsLocationMapper;
        this.redisHashMapAdapter = redisHashMapAdapter;
        this.sequenceCreater = sequenceCreater;
    }

    @Override
    public PageWarper<ToolsDTO> getToolsList(ToolsParam toolsParam) {

        PageWarper<ToolsDTO> pageWarper = new PageWarper<>(this.adminToolsMapper.queryToolsList(toolsParam));
        pageWarper.getList().forEach(toolsDTO -> {
            if (toolsDTO.getGlobalFlag() == 0) {
                toolsDTO.setLocationList(this.adminToolsLocationMapper.getToolsLocationByToolId(toolsDTO.getId()));
            }
        });
        return pageWarper;
    }

    @Override
    public ToolsDTO getToolsById(Long id) {
        Tools tools = this.adminToolsMapper.selectByPrimaryKey(id);
        return convertTools(tools);
    }

    @Override
    public int updateToolsStatusById(String[] ids, Byte status) {
        redisHashMapAdapter.remove(this.getToolsListKey());
        return this.adminToolsMapper.updateToolsStatusById(ids, status);
    }

    @Override
    public int updateToolsStatusById(Long id, Byte status) {
        Tools tools = new Tools();
        tools.setId(id);
        tools.setStatus(status);
        redisHashMapAdapter.remove(this.getToolsListKey());
        return this.adminToolsMapper.updateByPrimaryKeySelective(tools);
    }

    @Override
    public int addToolsDetail(Tools tools, String areaCodes) {
        Long toolsId = tools.getId();
        if (toolsId == null) {
            toolsId = sequenceCreater.nextLongId();
        }

        updateToolsLocation(toolsId, areaCodes);
        int update;
        if (tools.getId() == null) {
            tools.setId(toolsId);
            update = this.adminToolsMapper.insertSelective(tools);
        } else {
            update = this.adminToolsMapper.updateByPrimaryKeySelective(tools);
        }

        redisHashMapAdapter.remove(this.getToolsListKey());

        return update;
    }

    @Override
    public int deleteTools(Long id) {
        Tools tools = this.adminToolsMapper.selectByPrimaryKey(id);

        redisHashMapAdapter.remove(this.getToolsListKey());

        return this.adminToolsMapper.deleteByPrimaryKey(tools.getId());
    }

    /**
     * 更新对应地区的信息
     *
     * @param toolId
     * @param areaCodes
     */
    private void updateToolsLocation(Long toolId, String areaCodes) {
        if (StringUtils.isBlank(areaCodes)) {
            return;
        }

        this.adminToolsLocationMapper.deleteByToolId(toolId);
        insertToolsLocation(areaCodes, toolId);
    }

    /**
     * 添加地区信息
     *
     * @param areaCodes
     * @param toolId
     */
    private void insertToolsLocation(String areaCodes, Long toolId) {
        String[] areaCode = StringUtils.split(areaCodes, ",");

        List<ToolsLocation> toolsLocations = new ArrayList<>(areaCode.length);
        for (String s : areaCode) {
            ToolsLocation toolsLocation = new ToolsLocation();
            toolsLocation.setId(sequenceCreater.nextLongId());
            toolsLocation.setAreaCode(s);
            toolsLocation.setToolId(toolId);
            toolsLocations.add(toolsLocation);
        }
        this.adminToolsLocationMapper.insertToolsLocations(toolsLocations);
    }

    /**
     * 获取工具类列表的key
     *
     * @return
     */
    private KeyGenerator getToolsListKey() {
        return RedisConfig.THIRDPARTY_AREACODE_TOOLS.copy().appendKey("list");
    }

    /**
     * 移除对应的redis缓存
     *
     * @param areaCodes
     */
    private void removeRedisInfo(String areaCodes) {
        if (StringUtils.isBlank(areaCodes)) {
            return;
        }
        String[] areaCode = StringUtils.split(areaCodes, ",");
        redisHashMapAdapter.remove(this.getToolsListKey(), areaCode);
    }

    private ToolsDTO convertTools(Tools tools) {
        ToolsDTO toolsDTO = new ToolsDTO();
        BeanUtils.copyProperties(tools, toolsDTO);
        if (tools.getGlobalFlag() == 0) {
            toolsDTO.setLocationList(adminToolsLocationMapper.getToolsLocationByToolId(tools.getId()));
        }

        return toolsDTO;
    }
}
