package com.bxm.localnews.admin.service.impl;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.*;
import java.util.stream.Collectors;

import com.alibaba.fastjson.JSONObject;
import com.bxm.component.mybatis.utils.BatchHelper;
import com.bxm.component.office.excel.format.config.CellTypeEnum;
import com.bxm.component.office.excel.parse.ExcelParser;
import com.bxm.component.office.excel.parse.setting.ParseHeader;
import com.bxm.component.office.excel.parse.setting.ParseSetting;
import com.bxm.localnews.admin.constant.CommodityCategoryEnum;
import com.bxm.localnews.admin.domain.CommodityCodeMapper;
import com.bxm.localnews.admin.domain.CommodityMapper;
import com.bxm.localnews.admin.domain.PrivilegeMapper;
import com.bxm.localnews.admin.param.CommodityParam;
import com.bxm.localnews.admin.param.CommodityQueryParam;
import com.bxm.localnews.admin.service.CommodityService;
import com.bxm.localnews.admin.vo.CommodityBean;
import com.bxm.localnews.admin.vo.CommodityCodeBean;
import com.bxm.localnews.admin.vo.CommodityCodeSet;
import com.bxm.newidea.component.service.BaseService;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.base.Preconditions;
import com.google.common.collect.Lists;

import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

@Service
public class CommodityServiceImpl extends BaseService implements CommodityService {

    private CommodityMapper commodityMapper;

    private CommodityCodeMapper commodityCodeMapper;

    private PrivilegeMapper privilegeMapper;

    @Autowired
    public CommodityServiceImpl(CommodityMapper commodityMapper, CommodityCodeMapper commodityCodeMapper, PrivilegeMapper privilegeMapper) {
        this.commodityMapper = commodityMapper;
        this.commodityCodeMapper = commodityCodeMapper;
        this.privilegeMapper = privilegeMapper;
    }

    @Override
    public PageWarper<CommodityBean> getCommodities(CommodityQueryParam param) {
        Preconditions.checkArgument(param != null);
        return new PageWarper<>(commodityMapper.queryByPageSize(param));
    }

    @Override
    public Message save(CommodityParam param) {
        CommodityBean entity = convert(param);
        entity.setId(nextId());
        entity.setAddTime(new Date());

        Message message = Message.build();

        List<CommodityCodeBean> commodityCodes = parseExcel(param);

        logger.debug("excel's info"+JSONObject.toJSONString(commodityCodes));

        if (commodityCodes.size() > 0 && entity.getNum() != commodityCodes.size() && !param.getCategory() .equals( CommodityCategoryEnum.MOVIE.getType())) {
            message.setSuccess(false).setMessage("上传的商品兑奖码文件中数量[" + commodityCodes.size() + "]" +
                    "与商品库存数量[" + entity.getNum() + "]不符，请修改至两者数量一致");
            return message;
        }

        message = Message.build(commodityMapper.insert(entity));

        if (message.isSuccess()) {
            saveCommodityCodes(commodityCodes, entity.getId());
        }

        return message;
    }

    @Override
    public Message modify(CommodityParam param) {
        CommodityBean entity = convert(param);

        Message message = Message.build();
        //获取已经存在的编码
        List<CommodityCodeBean> codes = commodityCodeMapper.getUsedCodes(entity.getId());
        List<CommodityCodeSet> usedCodesSetList = convert(codes);
        List<CommodityCodeBean> usedCodes = codes.stream().filter(code->code.getUsedFlag()==1).collect(Collectors.toList());
        logger.debug("db's info:{}, db's used code:{}",JSONObject.toJSON(usedCodesSetList),JSONObject.toJSON(usedCodes));
        //删除对应的商品编码
        commodityCodeMapper.deleteByCommodityId(entity.getId());

        //获取excel的解析结果
        List<CommodityCodeBean> commodityCodes = parseExcel(param);
        List<CommodityCodeSet> commodityCodesSetList = convert(commodityCodes);
        logger.debug("excel's info"+JSONObject.toJSONString(commodityCodesSetList));

        Set<CommodityCodeSet> finalCodeSet = new HashSet<>();
        finalCodeSet.addAll(usedCodesSetList);
        finalCodeSet.addAll(commodityCodesSetList);
        logger.debug("merge's info"+JSONObject.toJSONString(commodityCodesSetList));

        if (commodityCodes.size() > 0 && finalCodeSet.size() != commodityCodes.size()&& !param.getCategory() .equals( CommodityCategoryEnum.MOVIE.getType())) {
            message.setSuccess(false).setMessage("上传的商品兑奖码文件中数量[" + commodityCodes.size() + "]" +
                    "与商品库存数量[" + finalCodeSet.size() + "]不符，请修改至两者数量一致");
            return message;
        }

        commodityCodes = Lists.newArrayList();
        CommodityCodeBean commodityCode;

        for (CommodityCodeSet commodityCodeSet : finalCodeSet) {
            commodityCode = new CommodityCodeBean();
            commodityCode.setId(nextId());
            commodityCode.setCode(commodityCodeSet.getCode());
            commodityCode.setExpireTime(commodityCodeSet.getExpireTime());
            boolean used = usedCodes.stream().anyMatch(usedCode -> StringUtils.equals(usedCode.getCode(), commodityCodeSet.getCode()));
            commodityCode.setUsedFlag(used ? 1 : 0);
            commodityCodes.add(commodityCode);
        }

        message = Message.build(commodityMapper.updateByPrimaryKey(entity));
        if (message.isSuccess()) {

            saveCommodityCodes(commodityCodes, entity.getId());
        }

        return Message.build(true);
    }

    private List<CommodityCodeSet> convert(List<CommodityCodeBean> commodityCodeBeanList){
        return commodityCodeBeanList.stream().map(commodityCodeBean->{
            CommodityCodeSet commodityCodeSet = new CommodityCodeSet();
            commodityCodeSet.setCode(commodityCodeBean.getCode());
            commodityCodeSet.setExpireTime(commodityCodeBean.getExpireTime());
            return commodityCodeSet;
        }).collect(Collectors.toList());
    }

    private void saveCommodityCodes(List<CommodityCodeBean> commodityCodes, Long commodityId) {
        if (commodityCodes.size() > 0) {
            new BatchHelper<CommodityCodeMapper, CommodityCodeBean>(CommodityCodeMapper.class, commodityCodes) {
                @Override
                protected int invoke(CommodityCodeBean element) {
                    element.setId(nextId());
                    element.setCommodityId(commodityId);

                    return mapper.insert(element);
                }
            };
        }
    }

    private CommodityBean convert(CommodityParam param) {
        Preconditions.checkArgument(param.getMechantId() > 0, "商户ID必须有值");

        CommodityBean entity = new CommodityBean();
        BeanUtils.copyProperties(param, entity);

        if (null == entity.getCurrentPrice()) {
            entity.setCurrentPrice(BigDecimal.ZERO);
        }
        if (null == entity.getOriginalCost()) {
            entity.setOriginalCost(BigDecimal.ZERO);
        }

        return entity;
    }

    private List<CommodityCodeBean> parseExcel(CommodityParam param) {
        logger.debug("收到商品编码excel[{}]", param.getCodeExcel());
        List<CommodityCodeBean> result = Lists.newArrayList();
        if (null != param.getCodeExcel()) {
            ExcelParser<CommodityCodeBean> excelParser = SpringContextHolder.getBean(ExcelParser.class);

            ParseSetting setting = new ParseSetting();
            setting.setEntityClasz(CommodityCodeBean.class);
            setting.addHead(new ParseHeader("商品编码", "code"));
            setting.addHead(new ParseHeader("过期时间", "expireTime", CellTypeEnum.DATE));
            try {
                result = excelParser.parse(param.getCodeExcel().getInputStream(), setting);
            } catch (IOException e) {
                logger.error("解析excel异常，请求参数为：[{}]", param);
                logger.error(e.getMessage(), e);
            }
        }

        return result;
    }

    @Override
    public CommodityBean get(long id) {
        return commodityMapper.selectByPrimaryKey(id);
    }

    @Override
    public Message change(long id, int status) {
        return Message.build(commodityMapper.updateStatus(id, status));
    }

    @Override
    public Message remove(long id) {
        //判断商品是否有关联的活动
        int relation = privilegeMapper.getRelationPrivilegeByCommodity(id);
        if (relation > 0) {
            return Message.build(false).setMessage("商品有" + relation + "个关联的活动，请先关闭对应的活动");
        }
        return Message.build(commodityMapper.deleteByPrimaryKey(id));
    }

    @Override
    public List<CommodityBean> getTopMatchCommodities(String keyword) {
        return commodityMapper.queryTopMatch(keyword);
    }

    @Override
    public Message execStoreDeduction(long commodityId, int num, long privilegeId) {
        Preconditions.checkArgument(commodityId > 0, "请输入有效的商品ID");
        Preconditions.checkArgument(num >= 0, "扣除库存数量必须大于等于0");

        Message message = Message.build();
        //判断库存是否足够
        int lastNum = commodityMapper.getStoreNum(commodityId, num);
        if (lastNum >= 0) {
            //库存足够，进行库存扣除
            message = Message.build(commodityMapper.updateStoreNum(commodityId, -num));
        } else {
            message.setSuccess(false).setMessage("商品库存不足，无法进行预出库操作");
        }

        if (message.isSuccess()) {
            //设置商品编码为已使用
            List<CommodityCodeBean> commodityCodeBeans = commodityCodeMapper.getUnusedCommodities(commodityId, num);
            if (commodityCodeBeans.size() > 0) {
                BatchHelper<CommodityCodeMapper, CommodityCodeBean> batchHelper = new BatchHelper<CommodityCodeMapper, CommodityCodeBean>(CommodityCodeMapper.class, commodityCodeBeans) {
                    @Override
                    protected int invoke(CommodityCodeBean element) {
                        element.setUsedFlag(1);
                        element.setPrivilegeId(privilegeId);
                        return mapper.updateByPrimaryKey(element);
                    }
                };
            }
        }
        return message;
    }

    @Override
    public Message execStoreReturned(long commodityId, int num, long privilegeId) {
        Preconditions.checkArgument(commodityId > 0);
        Preconditions.checkArgument(num >= 0, "退回的库存数量必须大于等于0");
        Message message = Message.build(commodityMapper.updateStoreNum(commodityId, num));

        if (message.isSuccess()) {
            //退回商品
            List<CommodityCodeBean> commodityCodeBeans = commodityCodeMapper.getUseCommoditiesAndPrivilege(commodityId, num, privilegeId);
            if (commodityCodeBeans.size() > 0) {
                new BatchHelper<CommodityCodeMapper, CommodityCodeBean>(CommodityCodeMapper.class, commodityCodeBeans) {
                    @Override
                    protected int invoke(CommodityCodeBean element) {
                        element.setUsedFlag(0);
                        element.setPrivilegeId(null);
                        return mapper.updateByPrimaryKey(element);
                    }
                };
            }
        }

        return message;
    }

}
