package com.bxm.localnews.base.controller;


import java.util.List;

import com.bxm.localnews.base.service.AreaInfoService;
import com.bxm.localnews.base.service.LocationRelationService;
import com.bxm.localnews.base.service.LocationService;
import com.bxm.localnews.common.dto.AreaDivisionDTO;
import com.bxm.localnews.common.dto.CustomerServiceDTO;
import com.bxm.localnews.common.dto.LocationMeta;
import com.bxm.localnews.common.dto.LocationRelationDTO;
import com.bxm.localnews.common.util.ResultUtil;
import com.bxm.localnews.common.vo.AreaInfo;
import com.bxm.localnews.common.vo.BasicParam;
import com.bxm.localnews.common.vo.Json;

import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

@Api(tags = "1-04 地址基础服务接口", description = "地址基础服务接口")
@RequestMapping("/api/area")
@RestController
public class LocationController {

    /**
     * 日志对象
     */
    private static final Logger LOG = LoggerFactory.getLogger(LocationController.class);

    private LocationService locationService;

    private AreaInfoService areaInfoService;

    private LocationRelationService locationRelationService;

    @Autowired
    public LocationController(LocationService locationService,
                              AreaInfoService areaInfoService,
                              LocationRelationService locationRelationService) {
        this.locationService = locationService;
        this.areaInfoService = areaInfoService;
        this.locationRelationService = locationRelationService;
    }

    @GetMapping(value = "/division")
    @ApiOperation(value = "1-04-1 获得省份信息", notes = "获得全国省份信息")
    public Json<List<AreaDivisionDTO>> listAreaDivisionAdditional() {
        return ResultUtil.genSuccessResult(locationService.listAreaDivisionDTO(null));
    }

    @GetMapping(value = "/division/{parentCode}")
    @ApiOperation(value = "1-04-2 根据上级编码获取省市区县街道", notes = "如果pcode为空,则返回最上层的省份信息")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "path", name = "parentCode", value = "父级编号", required = true),
    })
    public Json<List<AreaDivisionDTO>> listAreaDivision(@PathVariable String parentCode) {
        return ResultUtil.genSuccessResult(locationService.listAreaDivisionDTO(parentCode));
    }

    @GetMapping(value = "/location")
    @ApiOperation(value = "1-04-3 获取定位列表", notes = "包含开放的城市列表和热门城市列表,并根据首字母进行分组")
    public Json<LocationMeta> listAreaLocation() {
        return ResultUtil.genSuccessResult(locationService.listLocation());
    }

    @GetMapping(value = "/info/{code}")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "path", name = "code", value = "区域编号 一般为12位", required = true),
    })
    @ApiOperation(value = "1-04-4 根据code获取区域信息", notes = "")
    public Json<AreaInfo> getAreaInfo(@PathVariable String code, BasicParam basicParam) {
        if ("undefined".equals(code)) {
            return ResultUtil.genFailedResult("code非法，无法定位:"+code);
        }
        return ResultUtil.genSuccessResult(areaInfoService.getAreaInfo(code,basicParam));
    }

    @GetMapping(value = "/speech")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "code", value = "区域编号 一般为12位"),
            @ApiImplicitParam(name = "type", value = "话术类型 1：微信 2：朋友圈 5:小程序微信 6：小程序朋友圈")
    })
    @ApiOperation(value = "1-04-5 根据code获取区域的话术", notes = "在分享至微信的时候,需要显示的文字：包括微信、朋友圈的话术，其中包含内容和ID")
    public Json<LocationRelationDTO> getAreaSpeech(@RequestParam String code, @RequestParam Byte type) {

        LocationRelationDTO locationRelation = locationRelationService.getLocationRelation(code, type);
        return ResultUtil.genSuccessResult(locationRelation);
    }


    @GetMapping(value = "/location/{code}/customer")
    @ApiImplicitParams({
            @ApiImplicitParam(paramType = "path", name = "code", value = "区域编号 一般为12位", required = true),
    })
    @ApiOperation(value = "1-04-7 根据地区code获取客服信息", notes = "获取客服信息-二维码和微信号")
    public Json<CustomerServiceDTO> getCustomerServiceByCode(@PathVariable String code) {

        try {
            CustomerServiceDTO customerServiceDTO = locationService.getCustomerServiceInfo(code);
            if (null == customerServiceDTO) {
                return ResultUtil.genFailedResult("获取客服信息失败");
            }

            return ResultUtil.genSuccessResult(customerServiceDTO);
        } catch (Exception e) {
            LOG.error("获取客服信息异常", e);
            return ResultUtil.genFailedResult("获取客服信息异常");
        }
    }


}
