package com.bxm.localnews.im.controller;

import com.bxm.localnews.common.constant.RespCode;
import com.bxm.localnews.common.vo.Json;
import com.bxm.localnews.im.dto.SessionInfoDto;
import com.bxm.localnews.im.service.IMService;
import com.bxm.localnews.im.service.UserBlockService;
import com.bxm.localnews.im.service.UserService;
import com.bxm.newidea.component.vo.Message;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiImplicitParam;
import io.swagger.annotations.ApiImplicitParams;
import io.swagger.annotations.ApiOperation;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.web.bind.annotation.*;

@Api(tags = "7-01 用户即时通讯相关操作", description = "即时通讯相关的用户服务接口")
@RestController
@RequestMapping("user")
public class UserController {

    private final IMService imService;

    private final UserBlockService userBlockService;

    private final UserService userService;

    @Autowired
    public UserController(IMService imService,
                          UserBlockService userBlockService,
                          UserService userService) {
        this.imService = imService;
        this.userBlockService = userBlockService;
        this.userService = userService;
    }

    @ApiOperation(value = "7-01-1 屏蔽、解除屏蔽用户消息", notes = "选择解除屏蔽、屏蔽特定的用户时，上报服务端进行信息同步")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "userId", value = "当前用户ID"),
            @ApiImplicitParam(name = "blockUserId", value = "屏蔽用户的ID"),
            @ApiImplicitParam(name = "status", value = "1:屏蔽，0：解除屏蔽")
    })
    @PostMapping("block")
    public Json<Boolean> block(@RequestParam("userId") Long userId,
                               @RequestParam("blockUserId") Long blockUserId,
                               @RequestParam("status") Byte status) {
        Message message;
        if (status == 1) {
            message = userBlockService.add(userId, blockUserId);
        } else {
            message = userBlockService.cancel(userId, blockUserId);
        }
        return Json.build(message);
    }

    @ApiOperation(value = "7-01-2 获取即时通讯SDK需要的token（返回值的result即是）", notes = "客户端发起聊天前获取，服务端进行存储")
    @GetMapping("token")
    public Json<String> token(Long userId) {
        String token = imService.getToken(userId);
        if (StringUtils.isBlank(token)) {
            return Json.build(RespCode.BAD_REQUEST, "用户信息无效，可能是临时用户，请确认");
        }
        return Json.build(token);
    }

    @ApiOperation(value = "7-01-3 用户发起聊天时，获取通讯对象状态", notes = "用户发起聊天时，获取会话状态，包括：是否可以发送、是否关注对方")
    @ApiImplicitParams({
            @ApiImplicitParam(name = "fromUid", value = "发起会话的用户（当前用户）"),
            @ApiImplicitParam(name = "toUid", value = "会话目标用户")
    })
    @GetMapping("session")
    public Json<SessionInfoDto> session(@RequestParam("fromUid") Long fromUid,
                                        @RequestParam("toUid") Long toUid) {
        return Json.build(userService.get(fromUid, toUid));
    }
}
