package com.bxm.localnews.market.integration;

import com.bxm.localnews.market.dto.UserInfoDTO;
import com.bxm.localnews.market.dto.UserInviteHistoryDTO;
import com.bxm.localnews.market.dto.UserPhoneDTO;
import com.bxm.localnews.market.dto.UserTbkInfo;
import com.bxm.localnews.market.facade.MerchantGoodsFeignService;
import com.bxm.localnews.market.facade.UserFeignService;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class UserIntegrationService {

    @Resource
    private UserFeignService userFeignService;
    @Autowired
    private MerchantGoodsFeignService merchantGoodsFeignService;

    public UserInfoDTO selectUserFromCache(Long userId) {
        try {
            ResponseEntity<UserInfoDTO> userInfoDTOResponseEntity = this.userFeignService.selectUserFromCache(userId);
            if (Objects.nonNull(userInfoDTOResponseEntity)) {
                return userInfoDTOResponseEntity.getBody();
            }
        } catch (Exception e) {
            log.error("获取用户信息失败userId: {}", userId, e);
        }
        return null;
    }

    /**
     * 根据渠道id获取用户信息
     *
     * @param relationId 渠道id
     * @return 用户信息
     */
    public UserTbkInfo selectUserFromCacheByRelationId(String relationId) {

        try {
            ResponseEntity<UserTbkInfo> userTbkInfoResponseEntity = userFeignService.selectUserFromCacheByRelationId(relationId);
            if (Objects.nonNull(userTbkInfoResponseEntity)) {
                return userTbkInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("获取用户信息失败relationId :{}", relationId, e);
        }
        return null;
    }

    /**
     * 根据 userId
     *
     * @param userId userId
     * @return 用户信息
     */
    public UserTbkInfo selectUserFromCacheByUserId(Long userId) {

        try {
            ResponseEntity<UserTbkInfo> userTbkInfoResponseEntity = userFeignService.selectUserFromCacheByUserId(userId);
            if (Objects.nonNull(userTbkInfoResponseEntity)) {
                return userTbkInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("获取用户信息失败 userId :{}", userId, e);
        }
        return null;
    }

    /**
     * 更新标志位
     *
     * @param userId
     * @param judgeMarker
     * @return
     */
    public Boolean updateUserJudgeMarker(Long userId, LocalNewsUserJudgeMarkerEnum judgeMarker, boolean add) {

        if (log.isDebugEnabled()) {
            log.debug("userId: {} 更新标志位judgeMarker: {} ", userId, judgeMarker);
        }

        try {
            ResponseEntity<Boolean> responseEntity = userFeignService.updateUserJudgeMarker(userId, judgeMarker, add);

            if (Objects.nonNull(responseEntity)) {
                return responseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("更新标志位失败 userId: {} judgeMarker: {}", userId, judgeMarker, e);
        }

        return false;
    }


    /**
     * 根据脱敏手机号码模糊匹配用户信息
     *
     * @param sensitivePhone 脱敏手机号码
     * @return 用户手机号码信息
     */
    public List<UserPhoneDTO> getUserInfoBySensitivePhone(String sensitivePhone) {

        try {
            ResponseEntity<List<UserPhoneDTO>> userPhoneInfoResponseEntity = userFeignService.getUserInfoBySensitivePhone(sensitivePhone);
            if (Objects.nonNull(userPhoneInfoResponseEntity)) {
                return userPhoneInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("根据用户脱敏手机号码，获取用户手机号码信息失败 sensitivePhone :{}", sensitivePhone, e);
        }
        return Collections.emptyList();
    }

    /**
     * 根据用户手机号码获取用户id
     *
     * @param phone 手机号码
     * @return id信息
     */
    public UserPhoneDTO getUserInfoByPhone(String phone) {
        try {
            ResponseEntity<UserPhoneDTO> userPhoneInfoResponseEntity = userFeignService.getUserInfoByPhone(phone);
            if (Objects.nonNull(userPhoneInfoResponseEntity)) {
                return userPhoneInfoResponseEntity.getBody();
            }

        } catch (Exception e) {
            log.error("根据用户手机号码，获取用户id信息失败 phone :{}", phone, e);
        }
        return null;
    }
    /***
     * 获取邀请好友的信息
     * @param userId ： 用户id
     * @return ： 返回邀请记录
     */
    public UserInviteHistoryDTO getInviteInfo(Long userId){
        ResponseEntity<UserInviteHistoryDTO> userPhoneInfoResponseEntity = userFeignService.getInviteInfo(userId);
        if (Objects.nonNull(userPhoneInfoResponseEntity)) {
            UserInviteHistoryDTO inviteHistoryDTO = userPhoneInfoResponseEntity.getBody();
            if(null == inviteHistoryDTO || Objects.equals(inviteHistoryDTO.getState(), AppConst.USER_STATE.NORMAL)){
                return inviteHistoryDTO;
            }
        }
        return null;
    }

    /***
     * 获取邀请好友的信息
     * @param goodsId  : 商品id
     * @param num : 数量
     * @return ： 返回邀请记录
     */
    public Boolean subOrderNum(Long goodsId,  Integer num){
        return merchantGoodsFeignService.subOrderNum(goodsId,num).getBody();
    }
}
