package com.bxm.localnews.market.model.enums;

import java.util.Arrays;
import java.util.Objects;
import java.util.Optional;

import static java.util.Objects.isNull;

/**
 * 团购订单状态
 *
 * 单订单流转状态

 *                              |-------------------------------------|
 *                              |                                     |
 *                              V                                     |
 *      确认订单  ---支付--->  支付成功  ---核销--->  消费成功             |
 *         |                    |                                     |
 *         取                   发                                     |
 *         消                   起                                     |
 *         支                   退                                     |
 *         付 （手动 or 自动）    款                                     |
 *         |                    |                                      |
 *         V                    V                                      |
 *    取消支付的已失效          退款中  ----------拒绝退款-----------------|
 *                              |
 *                              同
 *                              意
 *                              退
 *                              款（手动 or 自动）
 *                              |
 *                              V
 *                        退款成功的已失效
 *
 *
 *
 *
 * @author gonzo
 * @date 2020-12-10 21:49
 **/
public enum GroupOrderStates {
    /**
     * 初始状态
     * 伪状态
     */
    INIT(0, true),

    /**
     * 待确认订单
     * 伪状态
     * 该状态未生成订单
     */
    WAIT_CONFIRM(0, true),

    /**
     * 已确认订单，待支付
     * 该状态已生成订单信息
     */
    WAIT_PAY(3, false),

    /**
     * 支付成功
     */
    PAY_SUCCESS(6, false),

    /**
     * 部分消费成功
     */
    CONSUME_PART_SUCCESS(0, true),

    /**
     * 消费成功
     */
    CONSUME_SUCCESS(7, false),

    /**
     * 退款中
     */
    REFUNDING(8, false),

    /**
     * 退款伪状态
     */
    REFUNDING_CHOSE(0, true),

    /**
     * 已失效
     */
    DISABLE(9, false),

    // 以下状态已废弃，挪到了子状态中，但客户端还在使用，所以依然保留定义
    /**
     * ：超时未支付
     */
    @Deprecated
    TIME_OUT(4, false),

    /**
     * 5：取消支付
     */
    @Deprecated
    CANCEL(5, false),

    /**
     * 10:拒绝退单待核销
     */
    @Deprecated
    REFUSE_REFUND_WAIT_CONSUME(10, false),

    ;

    private int status;

    private boolean pseudo;

    GroupOrderStates(int status, boolean pseudo) {
        this.status = status;
        this.pseudo = pseudo;
    }

    public static GroupOrderStates getByCode(Integer orderStatus) {
        if (isNull(orderStatus)) {
            return DISABLE;
        }

        Optional<GroupOrderStates> any = Arrays.stream(GroupOrderStates.values()).filter(p -> p.getStatus() == orderStatus).findAny();

        return any.orElse(null);
    }

    public boolean isPseudo() {
        return pseudo;
    }

    public int getStatus() {
        return status;
    }

    /**
     * 根据新版本的状态和子状态，来适配老版本的状态
     *
     * @param orderStates 一级状态
     * @param subStates   二级状态
     * @return 兼容的一级状态
     */
    public static GroupOrderStates adapt(GroupOrderStates orderStates, GroupOrderSubStates subStates) {
        if (isNull(orderStates)) {
            return DISABLE;
        }

        if (Objects.equals(orderStates, DISABLE) && Objects.nonNull(subStates)) {
            switch (subStates) {
                case DISABLE_BY_TIME_OUT_CANCEL:
                    return TIME_OUT;
                case DISABLE_BY_MANUAL_CANCEL:
                    return CANCEL;
                case DISABLE_BY_MANUAL_REFUND:
                case DISABLE_BY_AUTO_REFUND:
                    return DISABLE;
                default:
                    break;
            }
        }

        return orderStates;
    }

    /**
     * 根据新版本的状态和子状态，来适配老版本的状态
     *
     * @param orderStatesInt 一级状态
     * @param subStatesInt   二级状态
     * @return 兼容的一级状态
     */
    public static GroupOrderStates adapt(Integer orderStatesInt, Integer subStatesInt) {
        return adapt(GroupOrderStates.getByCode(orderStatesInt), GroupOrderSubStates.getByStatus(subStatesInt));
    }
}
