/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.market.service;

import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.market.integration.MessageSmsIntegrationService;
import com.bxm.localnews.market.model.dto.GoodsOrderEventDTO;
import com.bxm.localnews.market.model.dto.PushMessageByOrderDTO;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.OrderTabItem;
import com.bxm.localnews.market.model.enums.OrderTabType;
import com.bxm.localnews.mq.common.constant.PushMessageEnum;
import com.bxm.localnews.mq.common.constant.SmsTemplateEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.model.dto.SmsSupplyDTO;
import com.bxm.localnews.msg.facade.SmsFacadeService;
import com.bxm.localnews.msg.integration.SmsIntegrationService;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.newidea.component.tools.StringUtils;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import org.springframework.web.util.UriUtils;

import java.math.BigDecimal;
import java.nio.charset.Charset;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;

/**
 * @author jieliGG
 * @date 2020/8/31 19:32
 * 订单推送消息工厂
 **/
@Component
@Slf4j
public class OrderPushServiceFactory {
    @Autowired
    private MessageSender messageSender;

    @Autowired
    private DomainIntegrationService domainIntegrationService;
    @Autowired
    private MessageSmsIntegrationService messageSmsIntegrationService;

    /**
     * 给商家发送推送
     * 根据下单信息推送消息
     */
    public void pushMessageToMerchantByOrder(PushMessageByOrderDTO messageByOrderDTO) {
        try {
            String info = null;
            String title = null;
            switch (messageByOrderDTO.getOrderStatusEnum()) {
                case UNPAY:
                    title = "用户下单通知";
                    info = order(messageByOrderDTO);
                    break;
                case SUCCESS_PAY:
                    title = "用户付费订单";
                    info = pay(messageByOrderDTO);
                    break;
                case VERIFICATION_PAY:
                    title = "用户核销成功";
                    info = verification(messageByOrderDTO);
                    break;
                case REFUNDING:
                    title = "用户申请退款";
                    info = refund(messageByOrderDTO);
                    break;
                default:
                    log.info("传错类型了小宝贝~~~~");
            }
            if (null == title) {
                return;
            }
            //构建推送消息体
            PushPayloadInfo pushPayloadInfo = PushPayloadInfo.build(PushMessageEnum.JUMP_TYPE);
            pushPayloadInfo.setProtocol(buildMerchantProtocol(messageByOrderDTO.getMerchantId()));

            PushMessage message = PushMessage.build();
            message.setTitle(title);
            message.setContent(info);
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPayloadInfo(pushPayloadInfo);
            message.assign(messageByOrderDTO.getPushUserId());

            doSend(message);
        } catch (Exception e) {
            log.error("给商家发送推送bug messageByOrderDTO: {}", toJSONString(messageByOrderDTO), e);
        }

    }

    /**
     * 给用户发送
     * 根据下单信息推送消息
     */
    public void pushMessageToUserByOrder(PushMessageByOrderDTO messageByOrderDTO, GoodsOrderEventDTO orderEventDTO) {

        try {
            String info;
            String title;
            BigDecimal price;

            switch (messageByOrderDTO.getOrderStatusEnum()) {
                case SUCCESS_PAY:
                    title = "购买商品成功通知";
                    info = "恭喜你购买成功（%s），分享商品给好友可获得分佣哦~";
                    info = String.format(info, messageByOrderDTO.getGoodsName());
                    break;
                case REFUND_SUCCESS:
                    title = "退款提醒";
                    info = "您申请的退款（%s），已到账，请注意查收~";

                    if (Objects.nonNull(orderEventDTO.getOrderInfo())
                            && Objects.nonNull(orderEventDTO.getOrderInfo().getPayPrice())) {
                        price = orderEventDTO.getOrderInfo().getPayPrice();
                    } else {
                        price = BigDecimal.ZERO;
                    }
                    info = String.format(info, StringUtils.substring(orderEventDTO.getOrderInfo().getGoodsName(),0,5));
                    break;

                case REFUND_REFUSE:
                    title = "商家拒绝退款";
                    info = "商家拒绝你申请的退款（%s），如有疑问请联系商家或万事通客服花花";
                    if (Objects.nonNull(orderEventDTO.getOrderInfo())
                            && Objects.nonNull(orderEventDTO.getOrderInfo().getPayPrice())) {
                        price = orderEventDTO.getOrderInfo().getPayPrice();
                    } else {
                        price = BigDecimal.ZERO;
                    }
                    info = String.format(info,  StringUtils.substring(orderEventDTO.getOrderInfo().getGoodsName(),0,5));
                    break;
                case VERIFICATION_PAY:
                    title = "评价提醒";
                    info = "您有一笔订单待评价，评价可以获得小红花奖励哦~";
                    break;
                default:
                    //因为没有定义【长时间未使用类型】，所以就放到default里面了，使用的小伙伴注意看一下！！！！！
                    title = "消费提醒";
                    info = "您有一笔订单待消费，赶快去查看哦~";
                    break;
            }
            //构建推送消息体
            PushPayloadInfo pushPayloadInfo = PushPayloadInfo.build(PushMessageEnum.JUMP_TYPE);
            //根据订单tab和订单状态跳转对应订单tab
            pushPayloadInfo.setProtocol(buildUserProtocol(messageByOrderDTO.getPushUserId(),
                    OrderTabType.MY_ORDER_TAB, messageByOrderDTO.getOrderStatusEnum()));

            PushMessage message = PushMessage.build();
            message.setTitle(title);
            message.setContent(info);
            message.setType(TemplateTypeEnum.NOTIFCTION);
            message.setPayloadInfo(pushPayloadInfo);
            message.assign(messageByOrderDTO.getPushUserId());

            doSend(message);
        } catch (Exception e) {
            log.error("发送推送失败messageByOrderDTO: {} orderEventDTO: {}", toJSONString(messageByOrderDTO),
                    toJSONString(orderEventDTO), e);
        }

    }

    private void doSend(PushMessage message) {
        try {
            if (log.isDebugEnabled()) {
                log.debug("推送消息：{}", toJSONString(message));
            }
            // 推送
            messageSender.sendPushMessage(message);
        } catch (Exception e) {
            log.error("调用推送，发送推送失败 message: {}", toJSONString(message), e);
        }
    }

    /**
     * 购买商品下单
     */
    public void buyGoodsSendSms(String phoneNo,String goodsName,Long verificationCode){
        SmsSupplyDTO param = new SmsSupplyDTO();
        String info = "核销码%s，您已成功购买商品%s，戳 https://prod.wstong.com/r/order  查看我的订单";
        info = String.format(info,verificationCode,goodsName);
        param.setPhoneNo(phoneNo);
        param.setContent(info);
        messageSmsIntegrationService.sendSmsByCustomize(param);
    }


    /**
     * 退款申请通过
     */
    public void refundSuccessSendSms(String phoneNo,BigDecimal price){
        SmsSupplyDTO param = new SmsSupplyDTO();
        String info = "您申请的%s元退款已通过，将在7个工作日内退款，戳https://prod.wstong.com/r/order  查看详情";
        info = String.format(info,String.valueOf(price));
        param.setPhoneNo(phoneNo);
        param.setContent(info);
        messageSmsIntegrationService.sendSmsByCustomize(param);

    }


    /**
     * 退款申请拒绝
     */
    public void refundFailSendSms(String phoneNo,BigDecimal price){
        SmsSupplyDTO param = new SmsSupplyDTO();
        String info = "您申请的%s元退款被商家拒绝，戳https://prod.wstong.com/r/order  查看详情";
        info = String.format(info,String.valueOf(price));
        param.setPhoneNo(phoneNo);
        param.setContent(info);
        messageSmsIntegrationService.sendSmsByCustomize(param);
    }


    /**
     * 构建商户通知消息的跳转协议地址
     *
     * @return 组装的客户端协议，用于跳转到商家中心
     */
    private String buildMerchantProtocol(Long merchantId) {
        StringBuilder urlBuilder = new StringBuilder();
        urlBuilder.append(domainIntegrationService.getInnerH5BaseUrl())
                .append("/tk/merchant.html#/shopCenter?merchantId=").append(merchantId)
                .append("&userId={userId}")
                .append("&areaCode={areaCode}")
                .append("&areaName={areaName}");

        return "wst://web/webDetail?url=" + UriUtils.encode(urlBuilder.toString(), Charset.defaultCharset());
    }

    /**
     * 构建用户通知消息的跳转协议地址
     *
     * @param userId      用户id
     * @param order       0 我的订单，1 团队订单
     * @param orderStatus 订单状态
     * @return * @return
     */
    private String buildUserProtocol(Long userId, OrderTabType order, OrderStatusEnum orderStatus) {
        String jumpUrl = "wst://mine/billingDetail?order=%s&orderType=%s";

        // 转换后的类型：0 全部，1待结算，2已结算，3失效，4待付款，5待使用，6退款/售后
        int orderType = convert2OrderType(orderStatus);
        //拼接url
        jumpUrl = String.format(jumpUrl, order.getCode(), orderType);
        return jumpUrl;
    }

    /**
     * 根据订单状态转换为对应的订单tab值
     * 转换前：0：待结算|1：已结算|2：已失效 3:待支付 4：超时未支付 5：取消支付 6:成功支付 7:核销成功 8:退单中 9:退单成功 10:拒绝退单待核销
     * 转换后的类型：0 全部，1待结算，2已结算，3失效，4待付款，5待使用，6退款/售后
     *
     * @param orderStatus 订单状态
     * @return 订单tab的索引
     */
    private int convert2OrderType(OrderStatusEnum orderStatus) {
        if (orderStatus == null) {
            return OrderTabItem.ALL.getCode();
        }
        OrderTabItem result;
        switch (orderStatus) {
            case UNSETTLED:
                result = OrderTabItem.UNSETTLED;
                break;
            case HAVE_SETTLED:
                result = OrderTabItem.HAVE_SETTLED;
                break;
            case INVALID:
                result = OrderTabItem.INVALID;
                break;
            case UNPAY:
                result = OrderTabItem.WAIT_PAY;
                break;
            case TIMEOUT_PAY:
            case CANCLE_PAY:
            case VERIFICATION_PAY:
                result = OrderTabItem.ALL;
                break;
            case SUCCESS_PAY:
            case REFUND_REFUSE:
                result = OrderTabItem.WAIT_USE;
                break;
            case REFUNDING:
            case REFUND_SUCCESS:
                result = OrderTabItem.REFUND;
                break;
            default:
                result = OrderTabItem.ALL;
        }
        return result.getCode();
    }

    /**
     * 用户下单通知
     *
     * @param messageByOrderDTO： 类型信息
     * @return info
     */
    private String order(PushMessageByOrderDTO messageByOrderDTO) {
        String info = "用户%s刚刚下单了商品%s";
        info = String.format(info, messageByOrderDTO.getNickname(), messageByOrderDTO.getGoodsName());
        return info;
    }

    /**
     * 用户支付
     *
     * @param messageByOrderDTO： 类型信息
     * @return info
     */
    private String pay(PushMessageByOrderDTO messageByOrderDTO) {
        String info = "用户%s刚刚付费购买了商品%s";
        info = String.format(info, messageByOrderDTO.getNickname(), messageByOrderDTO.getGoodsName());
        return info;
    }

    /**
     * 用户核销成功
     *
     * @param messageByOrderDTO： 类型信息
     * @return info
     */
    private String verification(PushMessageByOrderDTO messageByOrderDTO) {
        String info = "用户%s已核销成功商品(%s)";
        info = String.format(info, messageByOrderDTO.getNickname(), messageByOrderDTO.getGoodsName());
        return info;
    }

    /**
     * 用户申请退款
     *
     * @param messageByOrderDTO ： 类型信息
     * @return info
     */
    private String refund(PushMessageByOrderDTO messageByOrderDTO) {
        String info = "用户%s申请退款（商品%s），请及时处理，超过5天未处理，系统自动退款";
        info = String.format(info, messageByOrderDTO.getNickname(), messageByOrderDTO.getGoodsName());

        return info;
    }
}
