package com.bxm.localnews.market.service.impl;

import com.bxm.localnews.constants.RedisConfig;
import com.bxm.localnews.market.config.MarketProperties;
import com.bxm.localnews.market.domain.MarketCommodityMapper;
import com.bxm.localnews.market.domain.MarketOrderAddressMapper;
import com.bxm.localnews.market.domain.MarketOrderMapper;
import com.bxm.localnews.market.model.dto.ExchangeRecordDTO;
import com.bxm.localnews.market.model.dto.ExchangeRoundInfoDTO;
import com.bxm.localnews.market.model.dto.MarketCommodityDTO;
import com.bxm.localnews.market.model.vo.MarketCommodity;
import com.bxm.localnews.market.model.vo.MarketOrder;
import com.bxm.localnews.market.dto.UserInfoDTO;
import com.bxm.localnews.market.integration.UserAccountIntegrationService;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.integration.WithdrawIntegrationService;
import com.bxm.localnews.market.param.AccountGoldParam;
import com.bxm.localnews.market.model.param.ExchangeCommodityParam;
import com.bxm.localnews.market.model.param.ExchangeRecordParam;
import com.bxm.localnews.market.service.MarketService;
import com.bxm.localnews.market.model.vo.MarketOrderAddress;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageParam;
import com.bxm.newidea.component.vo.PageWarper;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.Date;
import java.util.List;

/**
 * 商城服务
 *
 * @author donghy
 */
@Service
public class MarketServiceImpl implements MarketService {

    /**
     * 日志对象
     */
    private static final Logger LOG = LoggerFactory.getLogger(MarketServiceImpl.class);

    /**
     * 商城商品数据接口层
     */
    @Resource
    private MarketCommodityMapper marketCommodityMapper;

    /**
     * 商城商品订单数据接口层
     */
    @Resource
    private MarketOrderMapper marketOrderMapper;

    /**
     * 用户收货地址数据接口层
     */
    @Resource
    private MarketOrderAddressMapper marketOrderAddressMapper;

    /**
     * 用户账户第三方服务
     */
    @Resource
    private UserAccountIntegrationService userAccountIntegrationService;

    /**
     * 用户服务
     */
    @Resource
    private UserIntegrationService userIntegrationService;

    /**
     * 提现服务
     */
    @Resource
    private WithdrawIntegrationService withdrawIntegrationService;

    /**
     * redis
     */
    @Resource
    private RedisStringAdapter redisStringAdapter;

    /**
     * 分布式锁
     */
    @Resource
    private DistributedLock distributedLock;

    /**
     * id生成
     */
    @Resource
    private SequenceCreater sequenceCreater;

    /**
     * 配置
     */
    @Resource
    private MarketProperties marketProperties;

    @Override
    public PageWarper<MarketCommodityDTO> getAllCommodityList(PageParam pageParam) {
        return new PageWarper<>(marketCommodityMapper.queryAllCommodity(pageParam));
    }

    @Override
    public PageWarper<ExchangeRecordDTO> getExchangeRecordInfo(ExchangeRecordParam exchangeRecordParam) {
        PageWarper<ExchangeRecordDTO> resultPage = new PageWarper<>(marketOrderMapper.queryExchangeRecordInfo(exchangeRecordParam));

        //手机号加星号
        if (!CollectionUtils.isEmpty(resultPage.getList())) {
            for (ExchangeRecordDTO result : resultPage.getList()) {
                if (StringUtils.isNotBlank(result.getPhone())) {
                    result.setPhone(StringUtils.hideMobile(result.getPhone()));
                }
            }
        }

        return resultPage;
    }

    @Override
    public Message exchangeCommodity(ExchangeCommodityParam param, String devcId, String ip) {
        Message message = Message.build();
        message.setSuccess(false);
        //查询兑换的商品信息
        MarketCommodity marketCommodity = marketCommodityMapper.selectByPrimaryKey(param.getCommodityId());
        if (null == marketCommodity) {
            LOG.error("未查询到指定商品,商品id为:{}", param.getCommodityId());
            message.setMessage("未查询到指定商品");
            return message;
        }
        String date = DateUtils.formatAtWill(new Date(), "yyyyMMdd");
        //插入订单
        MarketOrderAddress orderAddress = new MarketOrderAddress();
        if ("GOODS".equals(marketCommodity.getType())) {
            orderAddress = generateOrderAddress(param);
            marketOrderAddressMapper.insertSelective(orderAddress);
        }
        MarketOrder order = generateOrder(param, marketCommodity, date, orderAddress);
        marketOrderMapper.insertSelective(order);

        //是否超过兑换次数
        if (null != marketCommodity.getMaxNum()) {
            Long times = redisStringAdapter.increment(RedisConfig.MARKET_EXCHANGE.copy().appendKey(param.getCommodityId())
                    .appendKey(param.getUserId()).appendKey(date));

            if (null != times && times > marketCommodity.getMaxNum()) {
                order.setStatus(1);
                marketOrderMapper.updateByPrimaryKeySelective(order);
                message.setMessage("今天的商品已经兑换完，请明日再兑");
                return message;
            }
        }

        //扣除用户金币
        AccountGoldParam accountGoldParam = new AccountGoldParam(param.getUserId(), "CONSUME_GOLD", false,
                marketCommodity.getPrice().intValue(), param.getCommodityId(), "EXCHANGE_COMMODITY");
        Boolean addGold = userAccountIntegrationService.addGold(accountGoldParam);

        if (null == addGold || !addGold) {
            order.setStatus(1);
            marketOrderMapper.updateByPrimaryKeySelective(order);
            message.setMessage("余额不足");
            return message;
        }

        //虚拟商品直接发放
        if ("VIRTUAL".equals(marketCommodity.getType())) {
            Message withdrawMessage = withdrawIntegrationService.appletWithdraw(param.getUserId(), param.getOpenId(),
                    marketCommodity.getPrice().divide(marketProperties.getGoldRate(), 2, 1), devcId, ip);

            if (null != withdrawMessage && withdrawMessage.isSuccess()) {
                order.setStatus(0);
                marketOrderMapper.updateByPrimaryKeySelective(order);
            } else {
                //提现失败返还金币
                accountGoldParam.setGold(-marketCommodity.getPrice().intValue());
                accountGoldParam.setGoldFlowType("WITHDRAW_FAIL_RETURN");
                userAccountIntegrationService.addGold(accountGoldParam);

                order.setStatus(1);
                marketOrderMapper.updateByPrimaryKeySelective(order);

                if (withdrawMessage != null) {
                    return withdrawMessage;
                } else {
                    message.setMessage("提现失败，请稍后再试");
                    return message;
                }
            }
        }

        message.setSuccess(true);
        message.setMessage("兑换成功");
        return message;
    }

    @Override
    public List<ExchangeRoundInfoDTO> queryExchangeRound() {
        return marketOrderMapper.queryExchangeRound();
    }

    private MarketOrderAddress generateOrderAddress(ExchangeCommodityParam param) {
        MarketOrderAddress orderAddress = new MarketOrderAddress();

        orderAddress.setId(sequenceCreater.nextLongId());
        orderAddress.setIsWechatAddress(param.getIsWechatAddress());
        orderAddress.setRecipient(param.getRecipient());
        orderAddress.setPhone(param.getPhone());
        orderAddress.setProvince(param.getProvince());
        orderAddress.setProvinceCode(param.getProvinceCode());
        orderAddress.setCity(param.getCity());
        orderAddress.setCityCode(param.getCityCode());
        orderAddress.setCounty(param.getCounty());
        orderAddress.setCountyCode(param.getCountyCode());
        orderAddress.setStreet(param.getStreet());
        orderAddress.setStreetCode(param.getStreetCode());
        orderAddress.setDetailAddress(param.getDetailAddress());
        orderAddress.setUserId(param.getUserId());
        orderAddress.setCreateTime(new Date());

        orderAddress.setFullAddress(param.getProvince() + param.getCity() +
                param.getCounty() + param.getStreet() + param.getDetailAddress());

        return orderAddress;
    }

    private MarketOrder generateOrder(ExchangeCommodityParam param, MarketCommodity marketCommodity, String date, MarketOrderAddress orderAddress) {
        MarketOrder order = new MarketOrder();

        UserInfoDTO userInfoDTO = userIntegrationService.selectUserFromCache(param.getUserId());
        if (null != userInfoDTO) {
            order.setUserName(userInfoDTO.getNickname());
        }

        order.setId(sequenceCreater.nextLongId());
        order.setOrderNo(date + redisStringAdapter.increment(RedisConfig.MARKET_EXCHANGE.appendKey(date).copy()));
        order.setUserId(param.getUserId());
        order.setCommodityId(param.getCommodityId());
        order.setCommodityTitle(marketCommodity.getTitle());
        order.setCommodityImages(marketCommodity.getImages());
        order.setCommodityPrice(marketCommodity.getPrice());
        order.setStatus(2);
        order.setAddressId(orderAddress.getId());
        order.setPhone(param.getPhone());
        order.setFinalAddress(orderAddress.getFullAddress());
        order.setCreateTime(new Date());

        return order;
    }


}
