package com.bxm.localnews.market.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.component.httpclient.utils.OkHttpUtils;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.component.tbk.order.model.enums.TakeOutOrderSource;
import com.bxm.component.tbk.order.model.enums.TbkOrderStatusEnum;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.market.config.*;
import com.bxm.localnews.market.domain.OrderInfoExtendMapper;
import com.bxm.localnews.market.domain.OrderInfoMapper;
import com.bxm.localnews.market.domain.OrderProfitExtendMapper;
import com.bxm.localnews.market.dto.UserInfoDTO;
import com.bxm.localnews.market.integration.MerchantGoodsIntegrationService;
import com.bxm.localnews.market.integration.PaymentInfoIntegrationService;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.integration.UserVipIntegrationService;
import com.bxm.localnews.market.model.constant.CommonConstant;
import com.bxm.localnews.market.model.constant.ProfitTypeConstant;
import com.bxm.localnews.market.model.dto.*;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.entity.OrderProfit;
import com.bxm.localnews.market.model.entity.UserOrderInfoBean;
import com.bxm.localnews.market.model.enums.*;
import com.bxm.localnews.market.model.param.*;
import com.bxm.localnews.market.model.param.order.UserBuyGoodsOrderParam;
import com.bxm.localnews.market.model.vo.*;
import com.bxm.localnews.market.service.UserOrderService;
import com.bxm.localnews.market.service.order.usergoods.event.GoodsOrderManage;
import com.bxm.localnews.market.util.Md5Utils;
import com.bxm.localnews.market.util.SignUtils;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import com.bxm.newidea.component.tools.BitOperatorUtil;
import com.bxm.newidea.component.vo.Message;
import com.github.pagehelper.Page;
import com.github.pagehelper.PageHelper;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.*;
import java.util.stream.Collectors;

import static java.nio.charset.StandardCharsets.UTF_8;
import static java.util.Objects.isNull;
import static org.apache.commons.lang3.StringUtils.*;

/**
 * 用户订单服务实现类
 *
 * @author wzy
 * @date 2020-09-04 10:09:48
 */
@Slf4j
@Service
public class UserOrderServiceImpl implements UserOrderService {

    @Autowired
    private OrderProfitExtendMapper orderProfitExtendMapper;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private MerchantGoodsIntegrationService merchantGoodsIntegrationService;

    @Autowired
    private OrderInfoExtendMapper orderInfoExtendMapper;
    @Autowired
    private OrderInfoMapper orderInfoMapper;


    @Autowired
    private OrderProperties orderProperties;

    @Autowired
    private KfcProperties kfcProperties;

    @Autowired
    private Taoquan365Properties taoquan365Properties;

    @Autowired
    private TbProperties tbProperties;

    @Autowired
    private GoodsOrderManage goodsOrderManage;

    @Autowired
    private PaymentInfoIntegrationService paymentInfoIntegrationService;

    @Autowired
    private OrderIconProperties orderIconProperties;

    @Autowired
    private DomainIntegrationService domainIntegrationService;
    @Autowired
    private UserVipIntegrationService userVipIntegrationService;


    @Override
    public List<UserOrderInfoVO> queryCommodityList(UserOrderParam param) {

        if (checkUserOrderParam(param)) {
            return Collections.emptyList();
        }

        Page<UserOrderInfoBean> page = PageHelper.startPage(param).doSelectPage(() ->
                orderProfitExtendMapper.selectByUserIdPageInfo(param));

        return page.getResult()
                .stream()
                .map(this::build)
                .collect(Collectors.toList());
    }

    @Override
    public List<MyOrderInfoVO> queryMyOrderList(MyOrderParam param) {
        if (checkMyOrderParam(param)) {
            return Collections.emptyList();
        }

        // 调用查询我的订单，用t_order_info表匹配t_order_profit
        Page<UserOrderInfoBean> page = PageHelper.startPage(param).doSelectPage(() ->
                orderProfitExtendMapper.selectMyOrderByUserIdPageInfo(param));

        // 获取t_order_info订单
        return page.getResult()
                .stream()
                .map(this::convert2MyOrderInfo)
                .collect(Collectors.toList());
    }

    @Override
    public List<GroupOrderInfoVO> queryGroupOrderList(GroupOrderParam param) {
        if (checkGroupOrderParam(param)) {
            return Collections.emptyList();
        }
        Page<UserOrderInfoBean> page = PageHelper.startPage(param).doSelectPage(() ->
                orderProfitExtendMapper.selectGroupOrderByUserIdPageInfo(param));

        return page.getResult()
                .stream()
                .map(this::convert2GroupOrderInfo)
                .collect(Collectors.toList());
    }

    @Override
    public OilGroupOrderDetailVO queryOilGroupOrderDetail(String orderSn, Long userId) {
        OrderInfo oilOrderInfo = orderInfoMapper.selectByOrderSnAndTypeAndUserId(orderSn,
                OrderTypeEnum.OIL_GROUP.getCode().toString(), userId);
        return convert2OilGroupOrderDetailVO(oilOrderInfo);
    }


    @Override
    public List<OrderStatusCountVO> queryOrderStatusCountList(Long userId) {
        // 目前只统计待使用的订单数量
        int orderNum = orderInfoMapper.countOrderNumByUserId(userId, 5);
        OrderStatusCountVO vo = new OrderStatusCountVO();
        vo.setOrderStatus(5);
        vo.setOrderCount(orderNum);
        List<OrderStatusCountVO> result = new ArrayList<>();
        result.add(vo);
        return result;
    }

    @Override
    public int getGoodsOrdersByUid(GetGoodsOrdersParam param) {
        return orderInfoMapper.countUserGoodsOrder(param.getUserId(), param.getGoodsId());
    }

    @Override
    public List<UserGoodsOrderDTO> getGoodsOrdersList(GetGoodsOrdersListParam param) {
        // 查询总数
        int orderCount = orderInfoMapper.countUserGoodsOrder(param.getUserId(), param.getGoodsId());
        // 查询出全部的订单信息，对信息进行订单信息填充
        List<OrderInfo> orderInfos = orderInfoMapper.queryUserGoodOrderInfo(param.getUserId(), param.getGoodsId());
        return orderInfos.stream().map(o -> convert2UserGoodsInfoDTO(o, orderCount)).collect(Collectors.toList());
    }

    @Override
    public BuyMerchantOrderDTO getOrderInfoByVerification(Long verification, Long merchantId) {
        BuyMerchantOrderDTO buyMerchantOrderDTO = new BuyMerchantOrderDTO();
        OrderInfo orderInfo = orderInfoMapper.getByVerification(verification);
        if (null == orderInfo) {
            buyMerchantOrderDTO.setVerificationState(2);
            return buyMerchantOrderDTO;
        }
        //商家是否一致
        MerchantGoodsInfoDTO goodsInfoDTO = merchantGoodsIntegrationService.getMerchantGoodsById(Long.parseLong(orderInfo.getGoodsId()));
        if (isNull(goodsInfoDTO) || isNull(merchantId) || isNull(goodsInfoDTO.getMerchantId())
                || !Objects.equals(merchantId, goodsInfoDTO.getMerchantId())) {
            buyMerchantOrderDTO.setVerificationState(2);
            return buyMerchantOrderDTO;
        }

        buyMerchantOrderDTO.setOrderNo(orderInfo.getOrderSn());
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy-MM-dd HH:mm:ss");
        buyMerchantOrderDTO.setCreateTime(sdf.format(orderInfo.getCreateTime()));
        buyMerchantOrderDTO.setGoodsName(orderInfo.getGoodsName());
        buyMerchantOrderDTO.setPayPrice(orderInfo.getPayPrice());
        buyMerchantOrderDTO.setTotalPrice(orderInfo.getPayPrice().multiply(new BigDecimal(orderInfo.getGoodsNum())));
        //暂时没有支付时间，先用创建时间
        buyMerchantOrderDTO.setPayTime(sdf.format(orderInfo.getCreateTime()));
        buyMerchantOrderDTO.setNum(orderInfo.getGoodsNum());
        buyMerchantOrderDTO.setGoodsImg(orderInfo.getImgUrl());
        buyMerchantOrderDTO.setPayPrice(orderInfo.getPayPrice());
        buyMerchantOrderDTO.setName(orderInfo.getOwnerUserName());
        buyMerchantOrderDTO.setPhone(orderInfo.getOwnerUserPhone());
        buyMerchantOrderDTO.setVerificationCode(String.valueOf(orderInfo.getVerificationCode()));
        if (Objects.equals(orderInfo.getOrderStatus(), OrderStatusEnum.SUCCESS_PAY.getStatus())) {
            buyMerchantOrderDTO.setVerificationState(1);
        } else if (Objects.equals(orderInfo.getOrderStatus(), OrderStatusEnum.VERIFICATION_PAY.getStatus())) {
            buyMerchantOrderDTO.setVerificationState(0);
        } else {
            buyMerchantOrderDTO.setVerificationState(2);
        }
        //设置vip优惠价格
        buyMerchantOrderDTO.setVipPrice(orderInfo.getGoodsPrice().subtract(orderInfo.getVipPurchaseCommission()));
        //是否是VIP
        if (Objects.nonNull(orderInfo.getOwnerUserId())) {
            buyMerchantOrderDTO.setIsVip(userVipIntegrationService.isVip(orderInfo.getOwnerUserId()) ? 1 : 0);
        } else {
            buyMerchantOrderDTO.setIsVip(Objects.equals(orderInfo.getVipPurchaseCommission().compareTo(BigDecimal.ZERO), 0) ? 0 : 1);
        }
        return buyMerchantOrderDTO;
    }

    /**
     * 转换为用户商品订单信息对象
     *
     * @param orderInfo  订单信息对象
     * @param orderCount 该用户购买该商品的订单数量
     * @return 用户商品订单信息对象
     */
    private UserGoodsOrderDTO convert2UserGoodsInfoDTO(OrderInfo orderInfo, int orderCount) {
        UserGoodsOrderDTO userGoodsOrderDTO = new UserGoodsOrderDTO();
        userGoodsOrderDTO.setGoodsName(orderInfo.getGoodsName());
        userGoodsOrderDTO.setVerificationCode(orderInfo.getVerificationCode());
        userGoodsOrderDTO.setState(orderInfo.getOrderStatus());
        userGoodsOrderDTO.setOrderNums(orderCount);
        userGoodsOrderDTO.setOrderSn(orderInfo.getOrderSn());

        MerchantInfoSimpleVO merchantInfoSimpleInfo = merchantGoodsIntegrationService.getMerchantInfoSimpleInfo(Long.valueOf(orderInfo.getGoodsId()));
        if (merchantInfoSimpleInfo != null) {
            userGoodsOrderDTO.setAddress(merchantInfoSimpleInfo.getAddress());
            userGoodsOrderDTO.setPhone(merchantInfoSimpleInfo.getMobile());
        }
        return userGoodsOrderDTO;
    }


    /**
     * 处理我的订单对象
     *
     * @param userOrderInfoBean 订单对象
     * @return 返回我的订单对象
     */
    private MyOrderInfoVO convert2MyOrderInfo(UserOrderInfoBean userOrderInfoBean) {
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm");
        MyOrderInfoVO info = MyOrderInfoVO.builder()
                .commission(userOrderInfoBean.getCommission())
                .goodsId(userOrderInfoBean.getGoodsId())
                .goodsName(userOrderInfoBean.getGoodsName())
                .orderSn(userOrderInfoBean.getOrderSn())
                .goodsFirstImg(userOrderInfoBean.getImgUrl())
                .orderStatus(userOrderInfoBean.getOrderStatus())
                .orderTypeName(profitTypeToOrderType(userOrderInfoBean.getOrderProfitType()))
                .payPrice(userOrderInfoBean.getPayPrice())
                .source(userOrderInfoBean.getSource())
                .sourceOrderCreateTime(userOrderInfoBean.getSourceOrderCreateTime() != null ? sdf.format(userOrderInfoBean.getSourceOrderCreateTime()) : null)
                .orderType(OrderTypeEnum.getDescEnumName(userOrderInfoBean.getTbOrderType()))
                .orderSubType(userOrderInfoBean.getOrderSubType())
                .sourceOwnerOrderStatus(userOrderInfoBean.getSourceOwnerOrderStatus())
                .build();

        DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss");
        String urlPrefix = "wst://function/openApp?url=";
        // 如果订单类型是淘宝订单
        if (OrderTypeEnum.TAOBAO.name().equals(info.getOrderType())
                || OrderTypeEnum.JUHUASUAN.name().equals(info.getOrderType())
                || OrderTypeEnum.TMALL.name().equals(info.getOrderType())) {
            // 判断订单状态,如果订单未收货
            if (TbkOrderStatusEnum.HAVA_PAID.getStatus().equals(info.getSourceOwnerOrderStatus())) {
                info.setConfirmReceived(false);
                info.setThirdPartyConfirmUrl(String.format(tbProperties.getConfirmJumpUrl(), info.getOrderSn()));
                //如果为待结算
                if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus())) {
                    info.setExpectSettlementTime("说明：确认收货后次月25号可结算");
                }
            } else {
                info.setConfirmReceived(true);
            }
            // 如果订单状态为待结算，且第三方订单状态不为已付款
            if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus()) &&
                    !TbkOrderStatusEnum.HAVA_PAID.getStatus().equals(info.getSourceOwnerOrderStatus())) {
                LocalDateTime now = LocalDateTime.now();
                // 下个月25日结算
                LocalDateTime nextMonth = now.plusMonths(1);
                // 得到下个月的日期
                LocalDateTime expectSettlementTime = LocalDateTime.of(nextMonth.getYear(), nextMonth.getMonth(), 25, 0, 0, 0);
                // 设置预估结算时间
                info.setExpectSettlementTime("预计结算时间：" + expectSettlementTime.format(df));
            }
            info.setOrderTitle("淘宝返现订单");
            info.setLogoUrl(orderIconProperties.getTbIconUrl());
        }
        // 如果订单类型是饿了么
        if (OrderTypeEnum.ELE.name().equals(info.getOrderType())) {
            // 如果为待结算
            if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus())) {
                // 预计结算时间
                Date sourceOrderCreateTime = userOrderInfoBean.getSourceOrderCreateTime();
                // 【饿了么结算时间】：订单生成次日11点
                LocalDateTime sourceLocalTime = convertToLocalDateTime(sourceOrderCreateTime);
                LocalDateTime nextDate = sourceLocalTime.plusDays(1);
                // 得到预计结算日期
                LocalDateTime expectSettlementTime = LocalDateTime.of(nextDate.getYear(), nextDate.getMonth(), nextDate.getDayOfMonth(), 11, 0, 0);
                // 设置预估结算时间
                info.setExpectSettlementTime("预计结算时间：" + expectSettlementTime.format(df));
            }
            info.setOrderTitle("外卖返现订单");
            info.setLogoUrl(orderIconProperties.getEleIconUrl());
            info.setGoodsFirstImg(orderIconProperties.getEleCoverImgUrl());

        }
        // 如果订单类型是美团
        if (OrderTypeEnum.MEITUAN.name().equals(info.getOrderType())) {
            //如果订单为待结算
            if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus())) {
                Date sourceOrderCreateTime = userOrderInfoBean.getSourceOrderCreateTime();
                // 【美团结算时间】订单生成次月25号0点
                LocalDateTime sourceLocalTime = convertToLocalDateTime(sourceOrderCreateTime);
                // 下个月25日结算
                LocalDateTime nextMonth = sourceLocalTime.plusMonths(1);
                // 得到下个月的日期
                LocalDateTime expectSettlementTime = LocalDateTime.of(nextMonth.getYear(), nextMonth.getMonth(), 25, 0, 0, 0);
                // 设置预估结算时间
                info.setExpectSettlementTime("预计结算时间：" + expectSettlementTime.format(df));
            }
            info.setOrderTitle("外卖返现订单");
            info.setLogoUrl(orderIconProperties.getMeituanIconUrl());
            info.setGoodsFirstImg(orderIconProperties.getMeiTuanCoverImgUrl());
        }
        // 如果是淘券365
        if (OrderTypeEnum.TAOQUAN365.name().equals(info.getOrderType())) {
            //返回淘券订单url
            info.setThirdPartyFindUrl(urlPrefix + getTaoQuanDetailUrl(info));
            info.setOrderTitle("卡券订单");
            info.setLogoUrl(orderIconProperties.getCardIconUrl());
            info.setGoodsFirstImg(orderIconProperties.getCardCoverImgUrl());
        }

        // 如果是kfc订单
        if (OrderTypeEnum.QIANZHU_KFC.name().equals(info.getOrderType())) {
            //设置查看KFC订单url
            info.setThirdPartyFindUrl(urlPrefix + getKfcDetailUrl(info));
            info.setOrderTitle("KFC在线点餐");
            info.setLogoUrl(orderIconProperties.getKfcIconUrl());
            info.setGoodsFirstImg(orderIconProperties.getKfcCoverImgUrl());
        }

        // 如果是团油订单
        if (OrderTypeEnum.OIL_GROUP.name().equals(info.getOrderType())) {
            info.setOrderTitle("加油订单");
            // 跳转h5详情页
            info.setThirdPartyFindUrl(getOilDetailUrl(info.getOrderSn()));
            info.setLogoUrl(orderIconProperties.getOilIconUrl());
            info.setGoodsFirstImg(orderIconProperties.getOilCoverImgUrl());
        }

        // 如果是团购订单
        if (OrderTypeEnum.WANSHITONG_ONE.name().equals(info.getOrderType())) {
            info.setOrderTitle("团购订单");
            info.setLogoUrl(orderIconProperties.getGroupIconUrl());
            // 所有状态都会下发h5
            String goodsDetailUrl = "tk/goods.html#/details?userId={userId}&goodsId=%s&areaCode={areaCode}&areaName={areaName}&lng={lng}&lat={lat}&wstClose=true";
            info.setThirdPartyFindUrl(domainIntegrationService.getInnerH5BaseUrl() + String.format(goodsDetailUrl, info.getGoodsId()));

            if (OrderStatusEnum.UNPAY.getStatus().equals(info.getOrderStatus())) {
                info.setPayH5Url(goodsDetailUrl);
            }
            if (log.isDebugEnabled()) {
                log.debug("订单跳转支付url: {}", info.getPayH5Url());
            }
        }

        // 如果是团购分销订单
        if (OrderTypeEnum.WST_ONE_COMMI.name().equals(info.getOrderType())) {
            //如果为待结算
            if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus())) {
                info.setExpectSettlementTime("说明：订单到店核销后可结算");
            }
            info.setOrderTitle("团购分销订单");
            info.setLogoUrl(orderIconProperties.getGroupIconUrl());

            String goodsDetailUrl = "tk/goods.html#/details?userId={userId}&goodsId=%s&areaCode={areaCode}&areaName={areaName}&lng={lng}&lat={lat}&wstClose=true";
            info.setThirdPartyFindUrl(domainIntegrationService.getInnerH5BaseUrl() + String.format(goodsDetailUrl, info.getGoodsId()));
        }

        return info;
    }

    /**
     * 获取团油详情页跳转地址（前端h5）
     *
     * @param orderSn 订单编号
     * @return 团油详情页跳转地址
     */
    private String getOilDetailUrl(String orderSn) {
        String oilDetailUrl = "tk/index.html#/orderDetails?userId={userId}&orderSn=%s";
        return domainIntegrationService.getInnerH5BaseUrl() + String.format(oilDetailUrl, orderSn);
    }

    /**
     * 将订单对象转换为团队订单对象
     *
     * @param userOrderInfoBean 用户
     * @return 团队订单对象
     */
    private GroupOrderInfoVO convert2GroupOrderInfo(UserOrderInfoBean userOrderInfoBean) {
        DateTimeFormatter df = DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm");
        SimpleDateFormat sdf = new SimpleDateFormat("yyyy/MM/dd HH:mm");
        GroupOrderInfoVO info = GroupOrderInfoVO.builder()
                .commission(userOrderInfoBean.getCommission())
                .goodsId(userOrderInfoBean.getGoodsId())
                .goodsName(userOrderInfoBean.getGoodsName())
                .orderSn(userOrderInfoBean.getOrderSn())
                .goodsFirstImg(userOrderInfoBean.getImgUrl())
                .orderStatus(userOrderInfoBean.getOrderStatus())
                .orderTypeName(profitTypeToOrderType(userOrderInfoBean.getType()))
                .payPrice(userOrderInfoBean.getPayPrice())
                .source(userOrderInfoBean.getSource())
                .sourceOrderCreateTime(userOrderInfoBean.getSourceOrderCreateTime() != null ? sdf.format(userOrderInfoBean.getSourceOrderCreateTime()) : null)
                .orderType(OrderTypeEnum.getDescEnumName(userOrderInfoBean.getTbOrderType()))
                .orderSubType(userOrderInfoBean.getOrderSubType())
                .build();

        // 校验是否为好友订单
        FriendOrderVO friendOrderVO = friendsOrder(userOrderInfoBean);

        if (Objects.nonNull(friendOrderVO)) {
            info.setFriendNick(friendOrderVO.getFriendNick());
            info.setImgUrl(friendOrderVO.getImgUrl());
            info.setCoupon(friendOrderVO.getCoupon());
            info.setType(friendOrderVO.getType());
            info.setFriendCommission(friendOrderVO.getCommission());
            info.setFriendOrder(getFriendOrder(userOrderInfoBean.getType()));
        }

        if (Objects.equals(userOrderInfoBean.getTbOrderType(), TakeOutOrderSource.ELE.getDesc())) {
            info.setGoodsFirstImg(orderProperties.getEleOrderImgUrl());
        } else if (Objects.equals(userOrderInfoBean.getTbOrderType(), TakeOutOrderSource.MEITUAN.getDesc())) {
            info.setGoodsFirstImg(orderProperties.getMeiTuanOrderImgUrl());
        }


        // 如果为上级返利
        if (ProfitTypeEnum.PARENT.name().equals(userOrderInfoBean.getType())) {
            // 设置为一级徒弟
            info.setOrderLevel(1);
            info.setOrderTitle("一级徒弟订单");
        }
        // 如果为上级返利
        if (ProfitTypeEnum.GRANDPARENT.name().equals(userOrderInfoBean.getType())) {
            // 设置为二级徒弟
            info.setOrderLevel(2);
            info.setOrderTitle("二级徒弟订单");
        }

        // 如果订单类型是淘宝订单类型
        if (OrderTypeEnum.TAOBAO.name().equals(info.getOrderType())
                || OrderTypeEnum.JUHUASUAN.name().equals(info.getOrderType())
                || OrderTypeEnum.TMALL.name().equals(info.getOrderType())) {
            // 判断订单状态,如果订单未收货
            if (TbkOrderStatusEnum.HAVA_PAID.getStatus().equals(userOrderInfoBean.getSourceOwnerOrderStatus())) {
                //如果为待结算
                if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus())) {
                    info.setExpectSettlementTime("说明：好友确认收货后次月25号可结算");
                }
                info.setConfirmReceived(false);
            } else {
                info.setConfirmReceived(true);
            }
            // 如果订单状态为待结算，且第三方订单状态不为已付款
            if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus()) &&
                    !TbkOrderStatusEnum.HAVA_PAID.getStatus().equals(userOrderInfoBean.getSourceOwnerOrderStatus())) {
                LocalDateTime now = LocalDateTime.now();
                // 下个月25日结算
                LocalDateTime nextMonth = now.plusMonths(1);
                // 得到下个月的日期
                LocalDateTime expectSettlementTime = LocalDateTime.of(nextMonth.getYear(), nextMonth.getMonth(), 25, 0, 0, 0);
                // 设置预估结算时间
                info.setExpectSettlementTime("预计结算时间：" + expectSettlementTime.format(df));
            }
        }
        // 如果是团购分销订单
        if (OrderTypeEnum.WST_ONE_COMMI.name().equals(info.getOrderType())) {
            //如果为待结算
            if (OrderStatusEnum.UNSETTLED.getStatus().equals(info.getOrderStatus())) {
                info.setExpectSettlementTime("说明：订单到店核销后可结算");
            }
            String goodsDetailUrl = "tk/goods.html#/details?userId={userId}&goodsId=%s&areaCode={areaCode}&areaName={areaName}&lng={lng}&lat={lat}&wstClose=true";
            info.setThirdPartyFindUrl(domainIntegrationService.getInnerH5BaseUrl() + String.format(goodsDetailUrl, info.getGoodsId()));
        }
        return info;
    }

    /**
     * 对象转换
     */
    private UserOrderInfoVO build(UserOrderInfoBean userOrderInfoBean) {
        UserOrderInfoVO info = UserOrderInfoVO.builder()
                .commission(userOrderInfoBean.getCommission())
                .goodsId(userOrderInfoBean.getGoodsId())
                .goodsName(userOrderInfoBean.getGoodsName())
                .orderSn(userOrderInfoBean.getOrderSn())
                .goodsFirstImg(userOrderInfoBean.getImgUrl())
                .orderStatus(userOrderInfoBean.getOrderStatus())
                .orderTypeName(profitTypeToOrderType(userOrderInfoBean.getType()))
                .payPrice(userOrderInfoBean.getPayPrice())
                .source(userOrderInfoBean.getSource())
                .sourceOrderCreateTime(userOrderInfoBean.getSourceOrderCreateTime())
                .orderType(OrderTypeEnum.getDescEnumName(userOrderInfoBean.getTbOrderType()))
                .orderSubType(userOrderInfoBean.getOrderSubType())
                .sourceOwnerOrderStatus(userOrderInfoBean.getSourceOwnerOrderStatus())
                .build();

        // 校验是否为好友订单
        FriendOrderVO friendOrderVO = friendsOrder(userOrderInfoBean);

        if (Objects.nonNull(friendOrderVO)) {
            info.setFriendNick(friendOrderVO.getFriendNick());
            info.setImgUrl(friendOrderVO.getImgUrl());
            info.setCoupon(friendOrderVO.getCoupon());
            info.setType(friendOrderVO.getType());
            info.setFriendCommission(friendOrderVO.getCommission());
            info.setFriendOrder(getFriendOrder(userOrderInfoBean.getType()));
        }

        if (Objects.equals(userOrderInfoBean.getTbOrderType(), TakeOutOrderSource.ELE.getDesc())) {
            info.setGoodsFirstImg(orderProperties.getEleOrderImgUrl());
        } else if (Objects.equals(userOrderInfoBean.getTbOrderType(), TakeOutOrderSource.MEITUAN.getDesc())) {
            info.setGoodsFirstImg(orderProperties.getMeiTuanOrderImgUrl());
        }
        return info;
    }

    /**
     * 创建用户订单订单
     *
     * @param param
     * @return ： 订单信息
     */
    @Override
    public UserBuyGoodsOrderDTO createGoodsOrder(UserBuyGoodsOrderParam param) {
        UserBuyGoodsOrderDTO goodsOrderDTO = new UserBuyGoodsOrderDTO();
        //构建参数并创建订单
        GoodsOrderEventDTO orderEventDTO = new GoodsOrderEventDTO();
        orderEventDTO.setGoodsOrderEventEnum(GoodsOrderEventEnum.TO_ORDER);
        orderEventDTO.setGoodsId(param.getGoodsId());
        orderEventDTO.setOpenVip(Objects.equals(param.getOpenVip(), AppConst.ENABLE));
        orderEventDTO.setPhone(param.getUserPhone());
        orderEventDTO.setUserId(param.getUserId());
        orderEventDTO.setUserName(param.getUserName());
        if (StringUtils.isNotEmpty(param.getInviterId())) {

            orderEventDTO.setShareUserId(Long.parseLong(param.getInviterId()));
        }

        Message message = goodsOrderManage.orderProcess(orderEventDTO);
        if (message.isSuccess()) {
            //订单信息
            OrderInfo orderInfo = message.getParam(CommonConstant.CREATE_ORDER_INFO_KEY);
            com.bxm.localnews.market.model.param.order.UserOrderParam userOrderParam = new com.bxm.localnews.market.model.param.order.UserOrderParam();
            BeanUtils.copyProperties(param, userOrderParam);
            userOrderParam.setPaymentTitle(orderInfo.getGoodsName());
            userOrderParam.setAmount(orderInfo.getPayPrice());
            userOrderParam.setOrderSn(orderInfo.getOrderSn());
            // 调用支付模块下单
            PaymentOrderDTO paymentOrderDTO = paymentInfoIntegrationService.orderCreate(userOrderParam);
            // 异常情况的商品数量回滚等等操作都是缺失的
            BeanUtils.copyProperties(paymentOrderDTO, goodsOrderDTO);
            // 开卡成功
            goodsOrderDTO.setCode(0);
            goodsOrderDTO.setOpenVip(param.getOpenVip());
            goodsOrderDTO.setOrderNo(orderInfo.getOrderSn());
        } else {
            goodsOrderDTO.setCode(1);
            goodsOrderDTO.setErrorMsg(message.getLastMessage());
        }

        return goodsOrderDTO;
    }

    /**
     * 发起退款
     *
     * @param orderNo ： 订单号
     * @return ： 退款结果
     */
    @Override
    public Boolean refundOrder(String orderNo) {
        GoodsOrderEventDTO orderEventDTO = new GoodsOrderEventDTO();
        orderEventDTO.setOrderNo(orderNo);
        orderEventDTO.setGoodsOrderEventEnum(GoodsOrderEventEnum.REFUND);
        return goodsOrderManage.orderProcess(orderEventDTO).isSuccess();
    }

    /**
     * 将OrderInfo类型团油对象，转换为OilGroupOrderDetailVO
     *
     * @param oilOrderInfo 团油信息
     * @return 团油订单详情对象
     */
    private OilGroupOrderDetailVO convert2OilGroupOrderDetailVO(OrderInfo oilOrderInfo) {
        if (oilOrderInfo == null) {
            return null;
        }
        OilGroupOrderDetailVO vo = new OilGroupOrderDetailVO();
        // 获取订单编号
        vo.setOrderId(oilOrderInfo.getOrderSn());
        // 获取订单状态描述
        vo.setOrderStatus(OilGroupOrderStatusEnum.of(oilOrderInfo.getSourceOwnerOrderStatus()).getDescription());
        // 获取订单的原始数据
        String sourceStr = oilOrderInfo.getSourceStr();
        if (isNotEmpty(sourceStr)) {
            OilGroupOrderDTO.Order thirdPartyOrderInfo = JSON.parseObject(sourceStr, OilGroupOrderDTO.Order.class);
            vo.setGasName(thirdPartyOrderInfo.getGasName());
            vo.setAddress(getOilAddr(thirdPartyOrderInfo));
            vo.setOilNo(thirdPartyOrderInfo.getOilNo());
            vo.setGunNo(thirdPartyOrderInfo.getGunNo());
            vo.setLitre(thirdPartyOrderInfo.getLitre());
            vo.setAmountGun(thirdPartyOrderInfo.getAmountGun());
            vo.setAmountDiscounts(thirdPartyOrderInfo.getAmountDiscounts());
            vo.setCouponMoney(thirdPartyOrderInfo.getCouponMoney());
            vo.setPayType(thirdPartyOrderInfo.getPayType());
            vo.setAmountPay(thirdPartyOrderInfo.getAmountPay());
            vo.setOrderTime(thirdPartyOrderInfo.getOrderTime());
        }
        return vo;
    }

    /**
     * 返回加油站地址
     *
     * @param thirdPartyOrderInfo 第三方订单信息
     * @return 加油站地址信息
     */
    private String getOilAddr(OilGroupOrderDTO.Order thirdPartyOrderInfo) {
        return thirdPartyOrderInfo.getProvince() +
                thirdPartyOrderInfo.getCity() +
                thirdPartyOrderInfo.getCounty() +
                thirdPartyOrderInfo.getGasName();
    }


    /**
     * 获取kfc跳转详情页地址
     *
     * @param orderInfoVO 订单信息vo对象
     * @return 详情页跳转地址
     */
    private String getKfcDetailUrl(MyOrderInfoVO orderInfoVO) {
        OrderInfo kfcOrderInfo = orderInfoMapper.selectByOrderSnAndType(orderInfoVO.getOrderSn(), OrderTypeEnum.QIANZHU_KFC.getCode());
        // 如果对象为空则返回默认地址
        if (kfcOrderInfo == null) {
            return kfcProperties.getDetailUrl();
        }
        String sourceOrderStr = kfcOrderInfo.getSourceStr();
        if (StringUtils.isBlank(sourceOrderStr)) {
            return kfcProperties.getDetailUrl();
        }
        // 原始订单数据
        KfcOrderDTO.Order sourceOrder = JSON.parseObject(sourceOrderStr, KfcOrderDTO.Order.class);
        //获取用户昵称
        String userName = sourceOrder.getUserName();
        String result = OkHttpUtils.get(kfcProperties.getGetTokenUrl(),
                buildKfcParamMap(sourceOrder.getPlatformUniqueId(), userName), Maps.newHashMap());
        if (StringUtils.isBlank(result)) {
            return kfcProperties.getDetailUrl();
        }
        KfcUserTokenDTO kfcUserTokenDTO = JSON.parseObject(result, KfcUserTokenDTO.class);
        if (kfcUserTokenDTO == null) {
            return kfcProperties.getDetailUrl();
        }
        // 如果请求结果失败
        if (kfcUserTokenDTO.getSuccess().equals(false)) {
            log.warn("获取用户token失败，result：{}", result);
            return kfcProperties.getDetailUrl();
        }
        // 如果结果为空
        if (kfcUserTokenDTO.getData() == null) {
            return kfcProperties.getDetailUrl();
        }
        String accessToken = kfcUserTokenDTO.getData().getAccessToken();
        if (StringUtils.isBlank(accessToken)) {
            return kfcProperties.getDetailUrl();
        }
        // 拼接详情页url
        String url = kfcProperties.getDetailUrl() + "?token=%s&orderNo=%s&platformId=%s&backIndex=true&backIndexType=0";
        url = String.format(url, accessToken, orderInfoVO.getOrderSn(), kfcProperties.getPlatformId());
        return url;
    }

    /**
     * 构建请求参数
     */
    private Map<String, Object> buildKfcParamMap(String userId, String nickname) {
        Map<String, Object> signMap = Maps.newHashMap();

        signMap.put("platformId", kfcProperties.getPlatformId());
        signMap.put("platformUniqueId", userId);
        signMap.put("nickname", nickname);
        signMap.put("timestamp", System.currentTimeMillis());

        String sign = SignUtils.generateSign(signMap, kfcProperties.getSecret());
        signMap.put("sign", sign.toLowerCase());

        return signMap;
    }


    /**
     * 获取淘券365订单详情页地址
     *
     * @param orderInfoVO 订单信息vo对象
     * @return 详情页跳转地址
     */
    private String getTaoQuanDetailUrl(MyOrderInfoVO orderInfoVO) {
        String detailUrl = taoquan365Properties.getDetailUrl();
        String urlParamPrefix = "?";
        if (isBlank(detailUrl)) {
            return detailUrl;
        }
        //拼接参数
        if (detailUrl.contains(urlParamPrefix)) {
            detailUrl += "&";
        } else {
            detailUrl += "?";
        }

        //登录账户
        detailUrl += "agentId=" + taoquan365Properties.getAgentId();

        //订单编号
        detailUrl += "&orderNo=" + orderInfoVO.getOrderSn();

        //时间戳
        detailUrl += "&timestamp=" + System.currentTimeMillis();

        //解析URL获取参数信息，计算签名
        List<NameValuePair> paramList = URLEncodedUtils.parse(substring(detailUrl, detailUrl.indexOf("?") + 1), UTF_8);

        paramList = paramList.stream().sorted(Comparator.comparing(NameValuePair::getName)).collect(Collectors.toList());

        StringBuilder signSource = new StringBuilder();

        for (NameValuePair nameValuePair : paramList) {
            if (equalsAny(nameValuePair.getName(), taoquan365Properties.getExcludeParam())) {
                continue;
            }

            signSource.append(nameValuePair.getName()).append("=").append(nameValuePair.getValue());
            signSource.append("&");
        }

        signSource.append("secretKey").append("=").append(taoquan365Properties.getSecretKey());

        //附加签名
        detailUrl += "&sign=" + Md5Utils.md5(signSource.toString()).toUpperCase();

        return detailUrl;
    }

    /**
     * 将Date类型转换为LocalDateTime类型
     *
     * @param sourceOrderCreateTime Date类型
     * @return LocalDateTime类型
     */
    private LocalDateTime convertToLocalDateTime(Date sourceOrderCreateTime) {
        //获取时间实例
        Instant instant = sourceOrderCreateTime.toInstant();
        //获取时间地区ID
        ZoneId zoneId = ZoneId.systemDefault();
        //转换为LocalDate
        return instant.atZone(zoneId).toLocalDateTime();
    }


    /**
     * 校验UserOrderParam对象是否合法
     *
     * @param param 用户订单入参
     * @return 是否合法
     */
    private boolean checkUserOrderParam(UserOrderParam param) {
        if (param.getUserId() == null || param.getUserId() == 0) {
            log.warn("user id is null");
            return true;
        }
        return false;
    }

    /**
     * 校验查询我的订单入参
     *
     * @param param 查询我的订单入参对象
     * @return 是否通过校验
     */
    private boolean checkMyOrderParam(MyOrderParam param) {
        if (param.getUserId() == null || param.getUserId() == 0) {
            log.warn("user id is null");
            return true;
        }
        return false;
    }

    /**
     * 校验GroupOrderParam对象是否合法
     *
     * @param param 团队订单入参
     * @return 是否合法
     */
    private boolean checkGroupOrderParam(GroupOrderParam param) {
        if (param.getUserId() == null || param.getUserId() == 0) {
            log.warn("user id is null");
            return true;
        }
        return false;
    }


    /**
     * 处理好友订单
     */
    private FriendOrderVO friendsOrder(UserOrderInfoBean userOrderInfoBean) {

        if (userOrderInfoBean.getUserId() == null || userOrderInfoBean.getUserId().equals(userOrderInfoBean.getOrderOwnerUserId())) {
            return null;
        }
        UserInfoDTO userInfo = userIntegrationService.selectUserFromCache(userOrderInfoBean.getOrderOwnerUserId());
        if (null == userInfo) {
            log.warn("获取好友信息为空，好友ID为[{}]", userOrderInfoBean.getOrderOwnerUserId());
            return null;
        }

        FriendOrderVO friendOrderVO = new FriendOrderVO();

        friendOrderVO.setFriendNick(userInfo.getNickname());
        friendOrderVO.setImgUrl(userInfo.getHeadImg());

        // 获取用户订单隐私设置状态
        // 默认启用隐私
        if (isNull(userInfo.getJudgeMarker())
                // 或者未设置过关闭隐私
                || !BitOperatorUtil.getBitAsBoolean(userInfo.getJudgeMarker(),
                LocalNewsUserJudgeMarkerEnum.ORDER_PRIVACY.getIndex())) {
            friendOrderVO.setType(PrivacyTypeEnum.ENABLE.getCode());
        } else {
            // 设置了开启隐私
            friendOrderVO.setType(PrivacyTypeEnum.DISABLE.getCode());
        }

        // 优惠券
        OrderInfo orderInfoByOrderId = orderInfoExtendMapper.getOrderInfoByOrderId(userOrderInfoBean.getOrderSn());
        if (orderInfoByOrderId != null) {
            friendOrderVO.setCoupon(orderInfoByOrderId.getCoupon());
        }

        // 购买人获得的佣金
        OrderProfit profitBean = orderProfitExtendMapper.getProfitInfoByUserIdAndOrderSn(userOrderInfoBean.getOrderSn(),
                userOrderInfoBean.getOrderOwnerUserId());
        if (profitBean != null) {
            friendOrderVO.setCommission(profitBean.getProfitAmount());
        }
        return friendOrderVO;
    }

    /**
     * profitType 转换为订单的类型
     */
    private String profitTypeToOrderType(String profitType) {

        if (StringUtils.isBlank(profitType)) {
            // 暂时设置，字段为空则返回自购
            return "自购";
        }
        String orderTypeName;
        switch (profitType.toUpperCase()) {

            case ProfitTypeConstant.PURCHASE:
                orderTypeName = "自购";
                break;
            case ProfitTypeConstant.PARENT:
            case ProfitTypeConstant.GRANDPARENT:
            case ProfitTypeConstant.CONTENT_PARENT:
            case ProfitTypeConstant.CONTENT_GRANDPARENT:
                orderTypeName = "好友";
                break;
            case ProfitTypeConstant.SHARE:
                orderTypeName = "分享";
                break;
            default:
                orderTypeName = "未定义";
        }

        return orderTypeName;

    }

    private Integer getFriendOrder(String orderUser) {
        return ProfitTypeConstant.PARENT.equals(orderUser)
                || ProfitTypeConstant.GRANDPARENT.equals(orderUser)
                || ProfitTypeConstant.CONTENT_PARENT.equals(orderUser)
                || ProfitTypeConstant.CONTENT_GRANDPARENT.equals(orderUser) ? 1 : 0;
    }

}
