package com.bxm.localnews.market.service.order;

import com.bxm.component.tbk.order.model.dto.TbkOrderInfo;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.component.tbk.order.service.impl.AbstractEleTakeOutOrderService;
import com.bxm.localnews.market.config.TakeOutOrderCommissionProperties;
import com.bxm.localnews.market.domain.OrderInfoExtendMapper;
import com.bxm.localnews.market.dto.UserTbkInfo;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.model.constant.ProfitTypeConstant;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.OrderTypeEnum;
import com.bxm.localnews.market.service.OrderCommissionService;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.Objects;

/**
 * 饿了么订单处理
 *
 * @author Gonzo
 * @date 2020-05-06 10:34
 * @since 2.0.0
 */
@Slf4j
@Service
public class EleTakeOutOrderServiceImpl extends AbstractEleTakeOutOrderService {

    @Autowired
    private OrderInfoExtendMapper orderInfoExtendMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private TakeOutOrderCommissionProperties takeOutOrderCommissionProperties;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private OrderCommissionService orderCommissionService;

    @Override
    protected boolean doSyncOrder(TbkOrderInfo orderInfo) {
        log.info("处理饿了么信息: {}", orderInfo);
        OrderInfo takeOutOrderInfo = orderInfoExtendMapper.getOrderInfoByOrderId(orderInfo.getOrderSn());

        if (Objects.nonNull(takeOutOrderInfo)) {
            updateEleTakeOutOrder(orderInfo, takeOutOrderInfo);
        } else {
            createEleTakeOutOrder(orderInfo);
        }

        return true;
    }

    /**
     * 创建饿了么订单入库
     *
     * @param orderInfo 订单信息
     */
    private void createEleTakeOutOrder(TbkOrderInfo orderInfo) {
        // 构建订单信息
        OrderInfo takeOutOrderInfo = build(orderInfo, true);

        // 查询用户信息
        UserTbkInfo userTbkInfo = userIntegrationService.selectUserFromCacheByRelationId(orderInfo.getRelationId());
        boolean isPurchaseBySelf = false;
        Long userId = null;
        if (Objects.nonNull(userTbkInfo)) {
            takeOutOrderInfo.setOwnerUserId(userTbkInfo.getId());
            userId = userTbkInfo.getId();
            // 是否自购
            isPurchaseBySelf = Objects.equals(userTbkInfo.getTbkSpecialId(), orderInfo.getSpecialId());
            // 判断是否为自购订单
            takeOutOrderInfo.setOrderProfitType(isPurchaseBySelf ? ProfitTypeConstant.PURCHASE : ProfitTypeConstant.SHARE);
        }

        // 添加订单记录
        orderInfoExtendMapper.insertSelective(takeOutOrderInfo);

        // 添加佣金记录
        orderCommissionService.addTakeOutOrderCommission(takeOutOrderInfo,
                takeOutOrderCommissionProperties.getEle().getCommonCommissionPurchaseRateForPayPrice(),
                userId, isPurchaseBySelf);

    }

    private OrderInfo build(TbkOrderInfo orderInfo, boolean create) {

        OrderInfo takeOutOrderInfo = new OrderInfo();
        Date now = new Date();
        if (create) {
            // 购物可得佣金
            BigDecimal purchaseCommission = orderInfo.getPayPrice()
                    .multiply(takeOutOrderCommissionProperties.getEle()
                            .getCommonCommissionPurchaseRateForPayPrice())
                    .setScale(2, RoundingMode.DOWN);

            // 上级佣金
            BigDecimal parent = orderInfo.getPayPrice()
                    .multiply(takeOutOrderCommissionProperties.getEle()
                            .getCommonCommissionRateParentForPayPrice())
                    .setScale(2, RoundingMode.DOWN);

            // 订单佣金 外卖订单不计算
            takeOutOrderInfo.setCommission(null);

            // 自购佣金
            takeOutOrderInfo.setPurchaseCommission(purchaseCommission);
            // vip自购佣金
            takeOutOrderInfo.setVipPurchaseCommission(purchaseCommission);
            // 上级
            takeOutOrderInfo.setParentCommission(parent);

            takeOutOrderInfo.setSource(orderInfo.getSource());

            takeOutOrderInfo.setId(sequenceCreater.nextLongId());
            takeOutOrderInfo.setTbOrderType(orderInfo.getTbOrderType());
            takeOutOrderInfo.setPid(orderInfo.getPid());
            takeOutOrderInfo.setType(orderInfo.getType());
            // 外卖订单
            takeOutOrderInfo.setOrderType(OrderTypeEnum.ELE.getCode());

            takeOutOrderInfo.setOrderParentSn(orderInfo.getOrderParentSn());
            takeOutOrderInfo.setGoodsPrice(orderInfo.getGoodsPrice());
            takeOutOrderInfo.setGoodsNum(orderInfo.getGoodsNum());
            takeOutOrderInfo.setPayPrice(orderInfo.getPayPrice());
            takeOutOrderInfo.setSourceOrderCreateTime(orderInfo.getSourceOrderCreateTime());
            takeOutOrderInfo.setSourceOrderEarningTime(orderInfo.getSourceOrderEarningTime());

            takeOutOrderInfo.setGoodsId("");
            takeOutOrderInfo.setGoodsName(orderInfo.getGoodsName());
            takeOutOrderInfo.setImgUrl(orderInfo.getImgUrl());
            takeOutOrderInfo.setRelationId(orderInfo.getRelationId());
            takeOutOrderInfo.setSpecialId(orderInfo.getSpecialId());
            takeOutOrderInfo.setOrderSn(orderInfo.getOrderSn());

            takeOutOrderInfo.setCreateTime(now);
            takeOutOrderInfo.setSourceStr(orderInfo.getSourceStr());
        }

        takeOutOrderInfo.setModifyTime(now);
        takeOutOrderInfo.setOrderStatus(OrderStatusEnum.getOrderStatus(orderInfo.getSourceOwnerOrderStatus()));
        takeOutOrderInfo.setSourceOwnerOrderStatus(orderInfo.getSourceOwnerOrderStatus());


        return takeOutOrderInfo;

    }

    /**
     * 更新饿了么订单
     *
     * @param orderInfo
     * @param takeOutOrder
     */
    private void updateEleTakeOutOrder(TbkOrderInfo orderInfo, OrderInfo takeOutOrder) {
        log.info("饿了么订单: {} 已存在 更新信息", orderInfo.getOrderSn());

        if (Objects.equals(orderInfo.getSourceOwnerOrderStatus(), takeOutOrder.getSourceOwnerOrderStatus())) {
            log.info("订单: {} 已存在库内，且库内源状态: {} 与处理状态相同，跳过处理", takeOutOrder.getSourceOwnerOrderStatus(),
                    orderInfo.getSourceOwnerOrderStatus());
            return;
        }

        if (orderInfo.getSourceOwnerOrderStatus() != 13
                && Objects.equals(takeOutOrder.getOrderStatus(), OrderStatusEnum.HAVE_SETTLED.getStatus())) {
            log.info("订单: {} 库内平台自身状态: {} 为已结算，跳过处理", orderInfo.getOrderSn(), takeOutOrder.getOrderStatus());
            return;
        }

        // 存在 且状态不相同 更新订单数据
        OrderInfo takeOutOrderForUpdate = build(orderInfo, false);
        takeOutOrderForUpdate.setId(takeOutOrder.getId());
        orderInfoExtendMapper.updateByPrimaryKeySelective(takeOutOrderForUpdate);

        // 如果更新的状态为失效
        if (orderInfo.getSourceOwnerOrderStatus() == 13) {

            // 佣金是未结算的 移除待结算金额
            if (Objects.equals(takeOutOrder.getOrderStatus(), OrderStatusEnum.UNSETTLED.getStatus())) {

                // 更新佣金流水和账户信息
                orderCommissionService.updateInvalidOrderCommission(orderInfo.getOrderSn());

            } else if (Objects.equals(takeOutOrder.getOrderStatus(), OrderStatusEnum.HAVE_SETTLED.getStatus())) {
                log.info("饿了么订单: {} 原本库内状态: {} 此次更新库内状态: {} 为已结算但后期进行取消",
                        takeOutOrder.getOrderSn(), takeOutOrder.getOrderStatus(), takeOutOrderForUpdate.getOrderStatus());
            }
        }
    }
}
