package com.bxm.localnews.market.service.order;

import com.alibaba.fastjson.JSON;
import com.bxm.component.httpclient.utils.OkHttpUtils;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.localnews.market.config.KfcProperties;
import com.bxm.localnews.market.domain.OrderInfoMapper;
import com.bxm.localnews.market.integration.PushIntegrationService;
import com.bxm.localnews.market.integration.UserAccountIntegrationService;
import com.bxm.localnews.market.model.dto.KfcOrderDTO;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.KfcOrderStatusEnum;
import com.bxm.localnews.market.model.enums.OrderTypeEnum;
import com.bxm.localnews.market.param.AccountCashParam;
import com.bxm.localnews.market.service.KfcOrderAcquireService;
import com.bxm.localnews.market.util.SignUtils;
import com.bxm.localnews.user.enums.CashEnum;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 千猪KFC订单拉取服务实现类
 *
 * @author wzy
 * @date 2020/8/13 14:03
 **/
@Slf4j
@Service
public class KfcOrderAcquireServiceImpl implements KfcOrderAcquireService {

    private static final String DATE_FORMAT_STR = "yyyy-MM-dd HH:mm:ss";

    private final KfcProperties kfcProperties;

    private final OrderInfoMapper orderInfoMapper;

    private final SequenceCreater sequenceCreater;

    private final PushIntegrationService pushIntegrationService;

    private final UserAccountIntegrationService userAccountIntegrationService;

    @Autowired
    public KfcOrderAcquireServiceImpl(KfcProperties kfcProperties, OrderInfoMapper orderInfoMapper,
                                      SequenceCreater sequenceCreater, PushIntegrationService pushIntegrationService,
                                      UserAccountIntegrationService userAccountIntegrationService) {
        this.kfcProperties = kfcProperties;
        this.orderInfoMapper = orderInfoMapper;
        this.sequenceCreater = sequenceCreater;
        this.pushIntegrationService = pushIntegrationService;
        this.userAccountIntegrationService = userAccountIntegrationService;
    }

    @Override
    public List<KfcOrderDTO.Order> getThirdPartyOrder(LocalDateTime startTime, LocalDateTime endTime) {
        List<KfcOrderDTO.Order> thirdPartyOrderList = new ArrayList<>();
        int pageSize = 100;
        //因为是分页所以要循环去拿
        for (int i = 1; i < Integer.MAX_VALUE; i++) {
            Map<String, Object> signMap = buildParamMap(startTime, endTime, i, pageSize);
            String result = OkHttpUtils.postJsonBody(kfcProperties.getPagedOrderUrl(),
                    JSON.toJSONString(signMap), Maps.newHashMap());

            KfcOrderDTO kfcOrderDTO = JSON.parseObject(result, KfcOrderDTO.class);
            //如果返回结果不成功，则直接结束循环，直接返回，防止死循环
            if (!kfcOrderDTO.getSuccess()) {
                log.error("获取千猪KFC第三方订单失败,result: {}", result);
                return thirdPartyOrderList;
            }
            List<KfcOrderDTO.Order> currentData = kfcOrderDTO.getData();
            if (currentData == null || currentData.isEmpty()) {
                break;
            }
            thirdPartyOrderList.addAll(currentData);
        }
        return thirdPartyOrderList;
    }

    @Override
    public void handleThirdPartyOrder(KfcOrderDTO.Order thirdPartyOrder,
                                      boolean push, boolean increase) {
        try {
            //根据类型和订单号查询是否有这笔订单
            OrderInfo dbOrderInfo = orderInfoMapper.selectByOrderSnAndType(thirdPartyOrder.getOrderNo(),
                    OrderTypeEnum.QIANZHU_KFC.getCode());
            // 第三方订单状态
            if (dbOrderInfo != null) {
                //如果第三方订单的状态和数据库中的订单状态相同，则不需要去更新订单
                if (dbOrderInfo.getSourceOwnerOrderStatus().equals(thirdPartyOrder.getStatus())) {
                    return;
                }
                // 如果订单已经是最终状态则不需要更新
                if (dbOrderInfo.getSourceOwnerOrderStatus().equals(KfcOrderStatusEnum.SUCCESS.getCode()) ||
                        dbOrderInfo.getSourceOwnerOrderStatus().equals(KfcOrderStatusEnum.CANCELLED.getCode())) {
                    return;
                }

                //否则更新订单状态
                OrderInfo updateOrderInfo = new OrderInfo();
                updateOrderInfo.setId(dbOrderInfo.getId());
                //将第三方订单状态转换为数据库中的订单状态
                updateOrderInfo.setOrderStatus(thirdPartyStatus2DbStatus(KfcOrderStatusEnum.of(thirdPartyOrder.getStatus())));
                updateOrderInfo.setSourceOwnerOrderStatus(thirdPartyOrder.getStatus());
                //更新原始订单信息
                updateOrderInfo.setSourceStr(JSON.toJSONString(thirdPartyOrder));
                orderInfoMapper.updateByPrimaryKeySelective(updateOrderInfo);
                //判断如果拉取的订单数据的状态为交易成功，则增加省钱金额,并更改数据库订单状态
                if (isSuccessStatus(thirdPartyOrder)
                        && thirdPartyOrder.getPlatformUniqueId() != null &&
                        dbOrderInfo.getSourceOwnerOrderStatus().equals(KfcOrderStatusEnum.WAIT_PAYMENT.getCode())) {
                    // 如果上一次的订单状态为待付款，这次为成功或排队中，则更新用户省钱金额，
                    pushMsgAndIncreaseSaveCash(thirdPartyOrder, dbOrderInfo, push, increase);
                }

            } else {
                //将拉取到的订单插入到订单表中
                OrderInfo newOrderInfo = convert2OrderInfo(thirdPartyOrder);
                orderInfoMapper.insertSelective(newOrderInfo);
                //判断订单状态如果为交易成功则增加省钱金额，并进行推送
                if (isSuccessStatus(thirdPartyOrder)
                        && thirdPartyOrder.getPlatformUniqueId() != null) {
                    pushMsgAndIncreaseSaveCash(thirdPartyOrder, newOrderInfo, push, increase);
                }
            }
        } catch (Exception ex) {
            log.error("千牛KFC第三方订单处理失败：{}, {}", JSON.toJSONString(thirdPartyOrder), ex);
        }

    }

    /**
     * 第三方订单状态转换为数据库中的订单状态
     *
     * @param kfcOrderStatusEnum kfc订单状态枚举
     * @return 数据库中的订单状态
     */
    public int thirdPartyStatus2DbStatus(KfcOrderStatusEnum kfcOrderStatusEnum) {
        if (kfcOrderStatusEnum == null) {
            return OrderStatusEnum.INVALID.getStatus();
        }
        OrderStatusEnum result;
        switch (kfcOrderStatusEnum) {
            case SUCCESS:
                result = OrderStatusEnum.VERIFICATION_PAY;
                break;
            case QUEUING:
                result = OrderStatusEnum.SUCCESS_PAY;
                break;
            case CANCELLED:
                result = OrderStatusEnum.CANCLE_PAY;
                break;
            case WAIT_PAYMENT:
                result = OrderStatusEnum.UNPAY;
                break;
            default:
                result = OrderStatusEnum.INVALID;
        }
        return result.getStatus();
    }

    /**
     * 判断订单是否交易成功
     *
     * @param thirdPartyOrder 第三方订单对象
     * @return 是否交易成功
     */
    private boolean isSuccessStatus(KfcOrderDTO.Order thirdPartyOrder) {
        return KfcOrderStatusEnum.SUCCESS.getCode().equals(thirdPartyOrder.getStatus())
                || KfcOrderStatusEnum.QUEUING.getCode().equals(thirdPartyOrder.getStatus());
    }

    @Override
    public void saveKfcOrderByTimeRange(String startTime, String endTime) {
        DateTimeFormatter df = DateTimeFormatter.ofPattern(DATE_FORMAT_STR);
        LocalDateTime start = LocalDateTime.parse(startTime, df);
        LocalDateTime end = LocalDateTime.parse(endTime, df);
        // 20分钟20分钟的同步
        while (start.isBefore(end)) {
            List<KfcOrderDTO.Order> thirdPartyOrderList = this.getThirdPartyOrder(start, (start = start.plusMinutes(20)));
            for (KfcOrderDTO.Order order : thirdPartyOrderList) {
                this.handleThirdPartyOrder(order, false, true);
            }
            // api调用有限流 半秒钟一次
            try {
                Thread.sleep(500);
            } catch (Exception e) {
                log.error("千牛KFC阻塞查询失败", e);
            }
        }
    }

    /**
     * 推送省钱消息并增加省钱金额
     *
     * @param thirdPartyOrder 第三方订单对象
     * @param orderInfo       db订单对象
     */
    private void pushMsgAndIncreaseSaveCash(KfcOrderDTO.Order thirdPartyOrder, OrderInfo orderInfo,
                                            boolean push, boolean increase) {
        BigDecimal saveAmount = calSaveMoney(thirdPartyOrder);
        if (BigDecimal.ZERO.compareTo(saveAmount) < 0) {
            //如果拉取的订单状态为交易成功，增加省钱金额
            if (increase) {
                userAccountIntegrationService.cashAccount(buildAccountCashParam(orderInfo, saveAmount));
            }
            if (push) {
                pushIntegrationService.pushSaveMoneyMsg(orderInfo, saveAmount);
                log.info("用户：{}, 增加省钱金额并推送，金额为{}", orderInfo.getOwnerUserId(), saveAmount);
            }
        }
    }

    /**
     * 构建调用local-user服务增加账户省钱金额接口入参
     *
     * @param orderInfo  订单信息
     * @param saveAmount 省钱金额
     * @return 参数对象
     */
    private AccountCashParam buildAccountCashParam(OrderInfo orderInfo, BigDecimal saveAmount) {
        AccountCashParam param = new AccountCashParam();
        param.setUserId(orderInfo.getOwnerUserId());
        param.setCashType(CashEnum.SAVE_CASH.name());
        param.setAddTotal(true);
        param.setCash(saveAmount);
        return param;
    }


    /**
     * 计算省钱金额:因为无法获取优惠金额
     * 使用 支付金额 * 优惠系数，来获取省钱金额，保留两位小数
     *
     * @param thirdPartyOrder 订单信息
     * @return 省钱金额
     */
    private BigDecimal calSaveMoney(KfcOrderDTO.Order thirdPartyOrder) {
        return (thirdPartyOrder.getAmount().
                multiply(new BigDecimal(kfcProperties.getDiscountCoefficient())))
                .setScale(2, RoundingMode.HALF_UP);
    }

    /**
     * 对第三方拉取饿订单信息进行转换，转换为数据库订单信息
     *
     * @param thirdPartyOrder 第三方订单信息对象
     * @return 数据库订单信息对象
     */
    private OrderInfo convert2OrderInfo(KfcOrderDTO.Order thirdPartyOrder) {
        OrderInfo dbOrderInfo = new OrderInfo();
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT_STR);
        try {
            dbOrderInfo.setSourceOrderCreateTime(sdf.parse(thirdPartyOrder.getCreateTime()));
            dbOrderInfo.setPayPrice(thirdPartyOrder.getAmount()
                    .setScale(2, RoundingMode.HALF_UP));
            dbOrderInfo.setOrderSn(thirdPartyOrder.getOrderNo());

            // 设置订单类型
            dbOrderInfo.setOrderType(OrderTypeEnum.QIANZHU_KFC.getCode());
            dbOrderInfo.setTbOrderType(OrderTypeEnum.QIANZHU_KFC.getDescription());

            dbOrderInfo.setSourceOwnerOrderStatus(thirdPartyOrder.getStatus());
            dbOrderInfo.setOrderStatus(thirdPartyStatus2DbStatus(KfcOrderStatusEnum.of(thirdPartyOrder.getStatus())));
            dbOrderInfo.setGoodsId(String.valueOf(sequenceCreater.nextLongId()));
            //保存原始数据
            dbOrderInfo.setSourceStr(JSON.toJSONString(thirdPartyOrder));
            dbOrderInfo.setGoodsPrice(thirdPartyOrder.getTotalPrice()
                    .setScale(2, RoundingMode.HALF_UP));
            //如果用户不等于空，设置用户id
            if (thirdPartyOrder.getPlatformUniqueId() != null) {
                dbOrderInfo.setOwnerUserId(Long.valueOf(thirdPartyOrder.getPlatformUniqueId()));
            }
            //获取商品信息
            List<KfcOrderDTO.Order.KfcPlaceOrder.Item> goodsItems = thirdPartyOrder.getKfcPlaceOrder().getItems();
            if (goodsItems != null && !goodsItems.isEmpty()) {
                dbOrderInfo.setGoodsName(goodsItems.get(0).getProductName()
                        + "等" + goodsItems.size() + "件商品");
                //这里保存第一个商品的信息
                dbOrderInfo.setImgUrl(goodsItems.get(0).getImageUrl());
            }
        } catch (Exception e) {
            log.error("千猪KFC第三方订单信息转换失败", e);
        }
        return dbOrderInfo;
    }

    /**
     * 构建请求参数
     */
    private Map<String, Object> buildParamMap(LocalDateTime startTime, LocalDateTime endTime, int pageIndex, int pageSize) {
        Map<String, Object> signMap = Maps.newHashMap();
        signMap.put("pageIndex", pageIndex);
        signMap.put("pageSize", pageSize);
        signMap.put("platformId", kfcProperties.getPlatformId());
        signMap.put("timestamp", System.currentTimeMillis());

        DateTimeFormatter df = DateTimeFormatter.ofPattern(DATE_FORMAT_STR);
        signMap.put("updateTimeBeginTime", startTime.format(df));
        signMap.put("updateTimeEndTime", endTime.format(df));

        String sign = SignUtils.generateSign(signMap, kfcProperties.getSecret());
        signMap.put("sign", sign.toLowerCase());

        return signMap;
    }
}
