package com.bxm.localnews.market.service.order;

import com.alibaba.fastjson.JSON;
import com.bxm.component.tbk.order.api.MeituanOrderApi;
import com.bxm.component.tbk.order.model.dto.MeituanOrderDetailInfo;
import com.bxm.component.tbk.order.model.dto.MeituanOrderInfo;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.component.tbk.order.service.impl.AbstractMeituanTakeOutOrderService;
import com.bxm.localnews.market.config.OrderProperties;
import com.bxm.localnews.market.config.TakeOutOrderCommissionProperties;
import com.bxm.localnews.market.domain.OrderInfoExtendMapper;
import com.bxm.localnews.market.model.constant.ProfitTypeConstant;
import com.bxm.localnews.market.model.dto.MeituanEuid;
import com.bxm.localnews.market.model.dto.MeituanOrderAllInfo;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.OrderTypeEnum;
import com.bxm.localnews.market.service.OrderCommissionService;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.Objects;

/**
 * 美团订单同步
 *
 * @author Gonzo
 * @date 2020-05-06 10:42
 * @since 2.0.0
 */
@Slf4j
@Service
public class MeituanTakeOutOrderServiceImpl extends AbstractMeituanTakeOutOrderService {

    @Autowired
    private OrderInfoExtendMapper orderInfoExtendMapper;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private MeituanOrderApi meituanOrderApi;

    @Autowired
    private TakeOutOrderCommissionProperties takeOutOrderCommissionProperties;

    @Autowired
    private OrderCommissionService orderCommissionService;

    @Autowired
    private OrderProperties orderProperties;

    @Override
    protected Integer doSyncOrderInfo(MeituanOrderInfo orderInfo) {

        // 查询订单详细信息
        MeituanOrderDetailInfo detailInfo = meituanOrderApi.getMeituanOrderDetailInfo(orderInfo.getOrderSn());

        if (Objects.isNull(detailInfo)) {
            log.warn("查询美团订单: {} 详情失败，无法添加订单数据", orderInfo.getOrderSn());
            return -1;
        }

        MeituanEuid euidObj = null;
        if (StringUtils.isNotBlank(orderInfo.getEuid())) {
            euidObj = JSON.parseObject(orderInfo.getEuid(), MeituanEuid.class);

            // 如果不是强制拉取所有订单 则判断是否当前业务主体订单
            if (!Objects.equals(orderProperties.getPullAllMeituanOrder(), Boolean.TRUE)) {
                // 判断是否是万事通的订单
                if (!Objects.equals(euidObj.getSiteId(), orderProperties.getMeituanOrderSiteId())) {
                    log.info("订单: {} 的site id: {} 不是配置的: {} 跳过处理", orderInfo.getOrderSn(), euidObj.getSiteId(),
                            orderProperties.getMeituanOrderSiteId());
                    return 1;
                }
            }
        }

        OrderInfo orderInfoByOrderId = orderInfoExtendMapper.getOrderInfoByOrderId(orderInfo.getOrderSn());

        if (Objects.nonNull(orderInfoByOrderId)) {
            return updateMeituanOrder(orderInfoByOrderId, orderInfo, detailInfo);
        } else {
            return createMeituanOrder(orderInfo, detailInfo, euidObj);
        }
    }


    /**
     * @param orderInfo
     * @return 1 表示 推送成功，并且订单已成功入库。0 表示 推送成功，但订单已存在。-1 表示 推送失败。
     */
    private int createMeituanOrder(MeituanOrderInfo orderInfo, MeituanOrderDetailInfo detailInfo, MeituanEuid euid) {

        OrderInfo takeOutOrder = build(orderInfo, detailInfo, euid, true);

        orderInfoExtendMapper.insertSelective(takeOutOrder);

        // 添加收益流水 账户金额
        orderCommissionService.addTakeOutOrderCommission(takeOutOrder, takeOutOrderCommissionProperties.getMeituanTakeOutOrder()
                .getCommonCommissionPurchaseRateForPayPrice(), takeOutOrder.getOwnerUserId(), true);

        return 1;
    }

    private OrderInfo build(MeituanOrderInfo orderInfo, MeituanOrderDetailInfo detailInfo, MeituanEuid euid, boolean create) {
        OrderInfo takeOutOrder = new OrderInfo();

        log.info("订单号: {} 的用户信息: {}", orderInfo.getOrderSn(), euid);
        Date now = new Date();
        if (create) {
            // 购物可得佣金
            BigDecimal purchaseCommission = orderInfo.getOrdersPrice()
                    .multiply(takeOutOrderCommissionProperties.getMeituanTakeOutOrder()
                            .getCommonCommissionPurchaseRateForPayPrice())
                    .setScale(2, RoundingMode.DOWN);

            takeOutOrder.setPurchaseCommission(purchaseCommission);

            // 订单时间
            Date orderTime = null;
            try {
                orderTime = DateUtils.DATE_TIME_FORMAT_THREAD_LOCAL.get().parse(orderInfo.getOrderTime());
            } catch (Exception e) {
                log.error("转换美团的订单创建时间失败", e);
            }

            Date chargeTime = null;
            try {
                if (!Objects.equals(orderInfo.getChargeTime(), "0000-00-00 00:00:00")) {
                    chargeTime = DateUtils.DATE_TIME_FORMAT_THREAD_LOCAL.get().parse(orderInfo.getChargeTime());
                }
            } catch (Exception e) {
                log.error("转换美团的订单结算时间失败", e);
            }

            takeOutOrder.setSourceOrderEarningTime(chargeTime);
            takeOutOrder.setSourceOrderCreateTime(orderTime);

            // 用户id
            Long userId = null;
            if (Objects.nonNull(euid)) {
                userId = euid.getUserId();
            }
            takeOutOrder.setOwnerUserId(userId);

            // 订单类型
            takeOutOrder.setOrderType(OrderTypeEnum.MEITUAN.getCode());

            takeOutOrder.setId(sequenceCreater.nextLongId());
            takeOutOrder.setGoodsId(detailInfo.getGoodsId());
            takeOutOrder.setGoodsName(detailInfo.getGoodsName());
            takeOutOrder.setImgUrl(null);
            //
            takeOutOrder.setRelationId(null);
            takeOutOrder.setSpecialId(null);
            takeOutOrder.setOrderSn(orderInfo.getOrderSn());
            //
            takeOutOrder.setOrderParentSn(null);
            takeOutOrder.setGoodsPrice(detailInfo.getGoodsPrice());
            takeOutOrder.setGoodsNum(detailInfo.getGoodsTa().longValue());
            takeOutOrder.setPayPrice(detailInfo.getOrdersPrice());


            takeOutOrder.setCommission(detailInfo.getOrderCommission());
            takeOutOrder.setVipPurchaseCommission(purchaseCommission);
            takeOutOrder.setParentCommission(null);
            takeOutOrder.setGrandparentCommission(null);


            takeOutOrder.setSource("MEITUAN");
            takeOutOrder.setTbOrderType("美团");
            takeOutOrder.setPid(StringUtils.join(orderInfo.getAdsId(), "_", orderInfo.getSiteId()));

            takeOutOrder.setType(null);
            takeOutOrder.setCreateTime(now);
            // 美团暂时全部为自购订单
            takeOutOrder.setOrderProfitType(ProfitTypeConstant.PURCHASE);

            // 源数据
            takeOutOrder.setSourceStr(JSON.toJSONString(MeituanOrderAllInfo.builder()
                    .detailInfo(detailInfo)
                    .orderInfo(orderInfo)
                    .createTime(now)
                    .build()));
        }

        // 订单状态
        takeOutOrder.setOrderStatus(getOrderStatus(orderInfo.getStatus()).getStatus());
        takeOutOrder.setSourceOwnerOrderStatus(orderInfo.getStatus());
        takeOutOrder.setModifyTime(now);

        return takeOutOrder;
    }

    /**
     * @param status 订单状态 -1 无效 0 未确认 1 确认 2 结算
     * @return 本地订单状态
     */
    private OrderStatusEnum getOrderStatus(Integer status) {
        switch (status) {

            case -1:
                return OrderStatusEnum.INVALID;
            case 0:
            case 1:
                // 当订单未未确认和确认时，本地状态都为未结算
            case 2:
                // 三方订单已计算时 本地状态都为未结算
                return OrderStatusEnum.UNSETTLED;
            default:
                // 默认无效订单
                return OrderStatusEnum.INVALID;
        }
    }

    /**
     * @param orderInfoByOrderId
     * @param orderInfo          1 表示 推送成功，并且订单已成功入库。0 表示 推送成功，但订单已存在。-1 表示 推送失败。
     * @return
     */
    private int updateMeituanOrder(OrderInfo orderInfoByOrderId, MeituanOrderInfo orderInfo, MeituanOrderDetailInfo detailInfo) {
        if (Objects.equals(orderInfo.getStatus(), orderInfoByOrderId.getSourceOwnerOrderStatus())) {
            log.info("订单: {}已存在库内，且状态: {} 相同，跳过处理", orderInfo.getOrderSn(), orderInfo.getStatus());
            return 1;
        }

        // 如果订单状态不是失效 则判断是否已经结算
        if (orderInfo.getStatus() != -1
                // 如果本地数据为三方已结算 或者本地数据为本地和三方都已结算 则不再做处理
                && (Objects.equals(orderInfoByOrderId.getOrderStatus(), OrderStatusEnum.HAVE_SETTLED.getStatus()))) {
            log.info("美团订单: {} 本地状态: {}, 回调状态: {} 为已结算，跳过处理", orderInfo.getOrderSn(),
                    orderInfoByOrderId.getOrderStatus(), orderInfo.getStatus());
            return 0;
        }

        // 更新状态
        OrderInfo update = build(orderInfo, detailInfo, null, false);
        update.setId(orderInfoByOrderId.getId());
        // 更新
        orderInfoExtendMapper.updateByPrimaryKeySelective(update);

        // 如果更新的状态为失效 移除待结算金额
        if (orderInfo.getStatus() == -1) {
            // 更新佣金流水和账户信息
            orderCommissionService.updateInvalidOrderCommission(orderInfo.getOrderSn());
        }
        return 1;
    }
}
