package com.bxm.localnews.market.service.order;

import com.alibaba.fastjson.JSON;
import com.bxm.component.httpclient.utils.OkHttpUtils;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.localnews.market.config.OilGroupProperties;
import com.bxm.localnews.market.domain.OrderInfoMapper;
import com.bxm.localnews.market.dto.UserPhoneDTO;
import com.bxm.localnews.market.integration.PushIntegrationService;
import com.bxm.localnews.market.integration.UserAccountIntegrationService;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.model.dto.OilGroupOrderDTO;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.OilGroupOrderStatusEnum;
import com.bxm.localnews.market.model.enums.OrderTypeEnum;
import com.bxm.localnews.market.param.AccountCashParam;
import com.bxm.localnews.market.service.OilGroupOrderAcquireService;
import com.bxm.localnews.market.util.SignUtils;
import com.bxm.localnews.user.enums.CashEnum;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.Maps;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 团油第三方订单拉取服务实现类
 *
 * @author wzy
 * @date 2020/8/13 14:10
 **/
@Service
@Slf4j
public class OilGroupOrderAcquireServiceImpl implements OilGroupOrderAcquireService {

    private final OilGroupProperties oilGroupProperties;

    private final OrderInfoMapper orderInfoMapper;

    private final SequenceCreater sequenceCreater;

    private final PushIntegrationService pushIntegrationService;

    private final UserAccountIntegrationService userAccountIntegrationService;

    private final UserIntegrationService userIntegrationService;

    @Autowired
    public OilGroupOrderAcquireServiceImpl(OilGroupProperties oilGroupProperties,
                                           OrderInfoMapper orderInfoMapper,
                                           SequenceCreater sequenceCreater,
                                           PushIntegrationService pushIntegrationService,
                                           UserAccountIntegrationService userAccountIntegrationService,
                                           UserIntegrationService userIntegrationService) {
        this.oilGroupProperties = oilGroupProperties;
        this.orderInfoMapper = orderInfoMapper;
        this.sequenceCreater = sequenceCreater;
        this.pushIntegrationService = pushIntegrationService;
        this.userAccountIntegrationService = userAccountIntegrationService;
        this.userIntegrationService = userIntegrationService;
    }

    @Override
    public void handleThirdPartyOrder(OilGroupOrderDTO.Order thirdPartyOrder,
                                      LocalDateTime startTime, LocalDateTime endTime,
                                      boolean push, boolean increase, boolean sensitive) {
        try {
            Integer currentOrderStatus =
                    OilGroupOrderStatusEnum.of(thirdPartyOrder.getOrderStatusName()).getCode();
            //根据类型和订单号查询是否有这笔订单
            OrderInfo dbOrderInfo = orderInfoMapper.selectByOrderSnAndType(thirdPartyOrder.getOrderId(),
                    OrderTypeEnum.OIL_GROUP.getCode());

            if (dbOrderInfo != null) {
                //如果数据库中的订单状态和第三方获取订单状态相同则不进行操作
                if (currentOrderStatus.equals(dbOrderInfo.getSourceOwnerOrderStatus())) {
                    return;
                }
                //更新订单状态
                OrderInfo updateOrderInfo = new OrderInfo();
                updateOrderInfo.setId(dbOrderInfo.getId());
                updateOrderInfo.setSourceOwnerOrderStatus(currentOrderStatus);
                //将第三方订单状态转换为数据库订单状态
                updateOrderInfo.setOrderStatus(thirdPartyStatus2DbStatus(OilGroupOrderStatusEnum.of(thirdPartyOrder.getOrderStatusName())));
                //更新原始订单信息
                updateOrderInfo.setSourceStr(JSON.toJSONString(thirdPartyOrder));
                //更新DB订单数据
                orderInfoMapper.updateByPrimaryKeySelective(updateOrderInfo);

            } else {
                OrderInfo newOrderInfo = convert2OrderInfo(thirdPartyOrder, startTime, endTime, sensitive);
                orderInfoMapper.insertSelective(newOrderInfo);
                if (newOrderInfo.getSourceOwnerOrderStatus().equals(OilGroupOrderStatusEnum.HAS_PAYMENT.getCode())
                        && newOrderInfo.getOwnerUserId() != null) {
                    pushMsgAndIncreaseSaveCash(thirdPartyOrder, newOrderInfo, push, increase);
                }
            }
        } catch (Exception ex) {
            log.error("团油第三方订单处理失败：{}, {}", JSON.toJSONString(thirdPartyOrder), ex);
        }
    }

    /**
     * 将第三方订单状态转换为DB中的订单状态
     *
     * @param oilGroupOrderStatusEnum 团油订单状态枚举
     * @return 数据库订单状态
     */
    public int thirdPartyStatus2DbStatus(OilGroupOrderStatusEnum oilGroupOrderStatusEnum) {
        if (oilGroupOrderStatusEnum == null) {
            return OrderStatusEnum.INVALID.getStatus();
        }
        OrderStatusEnum result;
        switch (oilGroupOrderStatusEnum) {
            case HAS_REFUND:
                result = OrderStatusEnum.REFUND_SUCCESS;
                break;
            case HAS_PAYMENT:
                result = OrderStatusEnum.VERIFICATION_PAY;
                break;
            case REFUND_FAIL:
            case REFUND_APPLYING:
                result = OrderStatusEnum.REFUNDING;
                break;
            default:
                result = OrderStatusEnum.INVALID;
        }
        return result.getStatus();
    }

    @Override
    public void saveOilOrderByTimeRange(String startTime, String endTime) {
        DateTimeFormatter df = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);
        LocalDateTime start = LocalDateTime.parse(startTime, df);
        LocalDateTime end = LocalDateTime.parse(endTime, df);
        // 20分钟20分钟的同步
        while (start.isBefore(end)) {
            List<OilGroupOrderDTO.Order> thirdPartyOrderList = this.getThirdPartyOrder
                    (start, start.plusMinutes(20), null);
            for (OilGroupOrderDTO.Order order : thirdPartyOrderList) {
                this.handleThirdPartyOrder(order,
                        start, start.plusMinutes(20), false, true, true);
            }
            start = start.plusMinutes(20);
            // api调用有限流 半秒钟一次
            try {
                Thread.sleep(500);
            } catch (Exception e) {
                log.error("团油阻塞查询失败", e);
            }
        }
    }

    /**
     * 推送消息并增加省钱金额
     *
     * @param thirdPartyOrder 第三方订单对象
     * @param orderInfo       db订单对象
     */
    private void pushMsgAndIncreaseSaveCash(OilGroupOrderDTO.Order thirdPartyOrder, OrderInfo orderInfo,
                                            boolean push, boolean increase) {
        BigDecimal saveAmount = calSaveMoney(thirdPartyOrder);

        if (BigDecimal.ZERO.compareTo(saveAmount) < 0) {
            //增加省钱金额并推送
            if (increase) {
                userAccountIntegrationService.cashAccount(buildAccountCashParam(orderInfo, saveAmount));
            }
            if (push) {
                pushIntegrationService.pushSaveMoneyMsg(orderInfo, saveAmount);
                log.info("用户：{}, 通过团油订单增加省钱金额并推送，金额为{}", orderInfo.getOwnerUserId(), saveAmount);
            }
        }
    }

    /**
     * 构建调用local-user 增加用户省钱余额接口入参
     *
     * @param orderInfo  订单信息
     * @param saveAmount 省钱金额
     * @return 入参对象
     */
    private AccountCashParam buildAccountCashParam(OrderInfo orderInfo,
                                                   BigDecimal saveAmount) {
        AccountCashParam param = new AccountCashParam();
        param.setUserId(orderInfo.getOwnerUserId());
        param.setCashType(CashEnum.SAVE_CASH.name());
        param.setAddTotal(true);
        param.setCash(saveAmount);
        return param;
    }

    /**
     * 计算省钱金额:订单总价 - 支付金额，保留两位小数
     *
     * @param thirdPartyOrder 订单信息
     * @return 省钱金额
     */
    private BigDecimal calSaveMoney(OilGroupOrderDTO.Order thirdPartyOrder) {
        return (new BigDecimal(thirdPartyOrder.getAmountGun())
                .subtract(new BigDecimal(thirdPartyOrder.getAmountPay())))
                .setScale(2, RoundingMode.HALF_UP);
    }

    /**
     * 将团油订单对象转化为数据库订单对象
     *
     * @param thirdPartyOrder 第三方订单对象
     * @param startTime       开始时间
     * @param endTime         结束时间
     * @return 构建的数据库订单对象
     */
    private OrderInfo convert2OrderInfo(OilGroupOrderDTO.Order thirdPartyOrder,
                                        LocalDateTime startTime, LocalDateTime endTime, boolean sensitive) {
        OrderInfo dbOrderInfo = new OrderInfo();
        SimpleDateFormat sdf = new SimpleDateFormat(DateUtils.DATE_TIME_FORMAT);
        try {
            dbOrderInfo.setSourceOrderCreateTime(sdf.parse(thirdPartyOrder.getOrderTime()));
            //支付金额
            dbOrderInfo.setPayPrice(new BigDecimal(thirdPartyOrder.getAmountPay())
                    .setScale(2, RoundingMode.HALF_UP));
            dbOrderInfo.setOrderSn(thirdPartyOrder.getOrderId());

            //修改订单类型
            dbOrderInfo.setOrderType(OrderTypeEnum.OIL_GROUP.getCode());
            dbOrderInfo.setTbOrderType(OrderTypeEnum.OIL_GROUP.getDescription());

            dbOrderInfo.setSourceOwnerOrderStatus(
                    OilGroupOrderStatusEnum.of(thirdPartyOrder.getOrderStatusName()).getCode());
            //保存原始数据
            dbOrderInfo.setSourceStr(JSON.toJSONString(thirdPartyOrder));
            //将第三方订单状态转换为数据库订单状态
            dbOrderInfo.setOrderStatus(thirdPartyStatus2DbStatus(OilGroupOrderStatusEnum.of(thirdPartyOrder.getOrderStatusName())));
            dbOrderInfo.setGoodsPrice(new BigDecimal(thirdPartyOrder.getAmountGun())
                    .setScale(2, RoundingMode.HALF_UP));
            dbOrderInfo.setGoodsId(String.valueOf(sequenceCreater.nextLongId()));
            dbOrderInfo.setGoodsName(
                    thirdPartyOrder.getGasName() + thirdPartyOrder.getOilNo() + thirdPartyOrder.getLitre() + "升");
            Long userId = locationUserId(thirdPartyOrder, startTime, endTime, sensitive);
            dbOrderInfo.setOwnerUserId(userId);
        } catch (Exception e) {
            log.error("团油第三方订单信息转换失败", e);
        }
        return dbOrderInfo;
    }

    /**
     * 根据用户脱敏手机号去定位用户id，如果查询出多个用户id，
     * 则根据用户手机号去调用查询第三方订单接口，判断当前订单号
     * 在哪个用户的订单中，来确定用户id，如果没有匹配到则返回null
     *
     * @param thirdPartyOrder 第三方订单对象
     * @param startTime       开始时间
     * @param endTime         结束时间
     * @return 用户id
     */
    private Long locationUserId(OilGroupOrderDTO.Order thirdPartyOrder, LocalDateTime startTime,
                                LocalDateTime endTime, boolean sensitive) {
        String sensitivePhone = thirdPartyOrder.getPhone();
        if (!checkPhone(sensitivePhone)) {
            return null;
        }
        // 如果是脱敏的手机号码调用脱敏的api
        if (sensitive) {
            List<UserPhoneDTO> userPhoneInfoList =
                    userIntegrationService.getUserInfoBySensitivePhone(sensitivePhone);

            if (userPhoneInfoList == null || userPhoneInfoList.isEmpty()) {
                return null;
            } else if (userPhoneInfoList.size() == 1) {
                // 如果查询结果只有一个用户则直接返回这个用户的id
                return userPhoneInfoList.get(0).getUserId();
            } else {
                //如果存在多个匹配用户
                return traverseOrderListGetUserId(userPhoneInfoList, thirdPartyOrder, startTime, endTime);
            }
        } else {
            // 如果不是脱敏手机号码则调用查询手机号码的api
            UserPhoneDTO userInfoByPhone = userIntegrationService.getUserInfoByPhone(sensitivePhone);
            if (userInfoByPhone != null && userInfoByPhone.getUserId() != null) {
                return userInfoByPhone.getUserId();
            } else {
                return null;
            }
        }
    }

    /**
     * 遍历用户的订单匹配订单对应的用户id
     *
     * @param userPhoneInfoList 用户手机信息List
     * @param thirdPartyOrder   第三方订单对象
     * @param startTime         开始时间
     * @param endTime           结束时间
     * @return 用户id
     */
    private Long traverseOrderListGetUserId(List<UserPhoneDTO> userPhoneInfoList,
                                            OilGroupOrderDTO.Order thirdPartyOrder, LocalDateTime startTime,
                                            LocalDateTime endTime) {
        for (UserPhoneDTO currentUser : userPhoneInfoList) {
            //根据这几个用户的手机号去获取他们的订单，判断当前订单属于哪个用户
            String currentUserPhone = currentUser.getPhone();
            List<OilGroupOrderDTO.Order> currentUserOrderList = getThirdPartyOrder(startTime, endTime, currentUserPhone);
            //如果订单列表不为空
            if (!currentUserOrderList.isEmpty()) {
                for (OilGroupOrderDTO.Order order : currentUserOrderList) {
                    if (order.getOrderId().equals(thirdPartyOrder.getOrderId())) {
                        return currentUser.getUserId();
                    }
                }
            }
        }
        return null;
    }

    /**
     * 检查脱敏手机是否合法
     *
     * @param sensitivePhone 脱敏手机号码
     * @return 是否合法
     */
    private boolean checkPhone(String sensitivePhone) {
        return sensitivePhone != null && sensitivePhone.length() == 11;
    }

    @Override
    public List<OilGroupOrderDTO.Order> getThirdPartyOrder(LocalDateTime startTime, LocalDateTime endTime, String phone) {
        List<OilGroupOrderDTO.Order> thirdPartyOrderList = new ArrayList<>();
        //这里分页数量最大值为100
        int pageSize = 100;
        Integer successCode = 200;
        //因为是分页所以要循环去拿
        for (int i = 1; i < Integer.MAX_VALUE; i++) {
            Map<String, Object> signMap = buildParamMap(startTime, endTime, i, pageSize, phone);
            String result = OkHttpUtils.postForm(oilGroupProperties.getPagedOrderUrl(),
                    signMap, Maps.newHashMap());

            OilGroupOrderDTO oilGroupOrderDTO = JSON.parseObject(result, OilGroupOrderDTO.class);
            if (!successCode.equals(oilGroupOrderDTO.getCode())) {
                log.error("获取团油第三方订单失败，code:{}, message:{}",
                        oilGroupOrderDTO.getCode(), oilGroupOrderDTO.getMessage());
                return thirdPartyOrderList;
            }
            List<OilGroupOrderDTO.Order> currentOrderResult = oilGroupOrderDTO.getResult();
            //如果当前循环获取的数据为null或者size等于0则退出循环
            if (currentOrderResult == null || currentOrderResult.isEmpty()) {
                break;
            }
            thirdPartyOrderList.addAll(currentOrderResult);
        }
        return thirdPartyOrderList;
    }

    /**
     * 构建查询第三方订单请求Map
     *
     * @param startTime 开始时间
     * @param endTime   结束时间
     * @param pageIndex 第几页
     * @param pageSize  每页多少条数据
     * @return 参数Map
     */
    private Map<String, Object> buildParamMap(LocalDateTime startTime, LocalDateTime endTime,
                                              int pageIndex, int pageSize, String phone) {
        DateTimeFormatter df = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);
        Map<String, Object> signMap = Maps.newTreeMap();
        signMap.put("app_key", oilGroupProperties.getAppKey());
        signMap.put("beginTime", startTime.format(df));
        signMap.put("endTime", endTime.format(df));
        signMap.put("orderSource", oilGroupProperties.getOrderSource());
        signMap.put("pageIndex", pageIndex);
        signMap.put("pageSize", pageSize);
        if (StringUtils.isNotBlank(phone)) {
            signMap.put("phone", phone);
        }
        signMap.put("timestamp", String.valueOf(System.currentTimeMillis()));

        String sign = SignUtils.generateConnectSign(signMap, oilGroupProperties.getAppSecret());
        signMap.put("sign", sign.toLowerCase());

        return signMap;
    }
}
