package com.bxm.localnews.market.service.order;

import com.alibaba.fastjson.JSON;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.localnews.market.constant.CommissionType;
import com.bxm.localnews.market.domain.OrderInfoExtendMapper;
import com.bxm.localnews.market.domain.OrderProfitExtendMapper;
import com.bxm.localnews.market.dto.CommissionPlatformFacadeDTO;
import com.bxm.localnews.market.dto.UserTbkInfo;
import com.bxm.localnews.market.model.enums.ProfitTypeEnum;
import com.bxm.localnews.market.integration.MessageUserIntegrationService;
import com.bxm.localnews.market.integration.PushIntegrationService;
import com.bxm.localnews.market.integration.UserAccountIntegrationService;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.entity.OrderInfoForQuartz;
import com.bxm.localnews.market.model.entity.OrderProfit;
import com.bxm.localnews.market.model.enums.OrderTypeEnum;
import com.bxm.localnews.market.param.AccountCashParam;
import com.bxm.localnews.market.param.CashAccountParam;
import com.bxm.localnews.market.service.OrderCommissionService;
import com.bxm.localnews.user.enums.AccountActionEnum;
import com.bxm.localnews.user.enums.CashEnum;
import com.bxm.localnews.user.enums.CashFlowTypeEnum;
import com.bxm.localnews.user.enums.LocalNewsUserJudgeMarkerEnum;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.List;
import java.util.Objects;

@Slf4j
@Service
public class OrderCommissionServiceImpl implements OrderCommissionService {

    @Autowired
    private OrderProfitExtendMapper orderProfitExtendMapper;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Autowired
    private MessageUserIntegrationService messageUserIntegrationService;

    @Autowired
    private OrderInfoExtendMapper orderInfoExtendMapper;

    @Autowired
    private UserAccountIntegrationService userAccountIntegrationService;

    @Autowired
    private PushIntegrationService pushIntegrationService;


    @Override
    public boolean addCommission(OrderInfo orderInfo, UserTbkInfo userTbkInfo,
                                 CommissionPlatformFacadeDTO commissionPlatformFacadeDTO) {

        log.info("添加返佣信息, orderInfo: {}, userTbkInfo: {}, commissionPlatformFacadeDTO: {}, ",
                JSON.toJSONString(orderInfo),
                JSON.toJSONString(userTbkInfo), JSON.toJSONString(commissionPlatformFacadeDTO));

        if (Objects.isNull(orderInfo)) {
            log.warn("添加返佣信息失败，订单信息为null");
            return false;
        }

        // 如果relation id 没有对应的用户 则订单不添加返佣信息
        if (Objects.isNull(userTbkInfo) || Objects.isNull(userTbkInfo.getId())) {
            log.info("用户relationId({}) 未查询到对应平台的id", orderInfo.getRelationId());
            return false;
        }

        // 是否失效订单
        boolean isInvalidOrder = OrderStatusEnum.isInvalidOrder(orderInfo.getOrderStatus());
        Long userId = userTbkInfo.getId();
        // special id是购买者id 如果和relation 获取到的user id一致，则说明是自购
        boolean isPurchaseBySelf = Objects.equals(userTbkInfo.getTbkSpecialId(), orderInfo.getSpecialId());
        // 是否vip
        boolean isVip = Objects.equals(userTbkInfo.getIsVip(), Boolean.TRUE);
        boolean isNovice = Objects.equals(userTbkInfo.getIsNovice(), Boolean.TRUE);

        log.info("添加返佣信息 userId; {} superiorUserId: {} topUserId: {} isPurchaseBySelf: {} isInvalidOrder: {} " +
                        "isNovice: {} isVip: {}",
                userId, userTbkInfo.getSuperiorUserId(), userTbkInfo.getTopUserId(),
                isPurchaseBySelf, isInvalidOrder, isVip, isNovice);

        // 自购佣金金额
        BigDecimal commission = isVip ? orderInfo.getVipPurchaseCommission() : orderInfo.getPurchaseCommission();

        // 添加自购佣金等信息
        // 记录profit
        recordOrderProfit(orderInfo, userId, userId,
                // 自购还是分享单
                isPurchaseBySelf ? ProfitTypeEnum.PURCHASE : ProfitTypeEnum.SHARE,
                commission,
                // 自购比例
                isVip ? commissionPlatformFacadeDTO.getPlatformCommissionRageConfigInfo().getVipPurchaseRate()
                        : commissionPlatformFacadeDTO.getPlatformCommissionRageConfigInfo().getPurchaseRate());
        purchaseInfoRecode(userId, isInvalidOrder, orderInfo, commission, isVip,
                isPurchaseBySelf);

        // 非新人首单才给上级进行返佣
        if (!Objects.equals(commissionPlatformFacadeDTO.getCommissionType(), CommissionType.NOVICE)) {
            // 上级返佣
            if (Objects.nonNull(userTbkInfo.getSuperiorUserId())) {
                // 添加profit 信息
                recordOrderProfit(orderInfo, userTbkInfo.getSuperiorUserId(), userTbkInfo.getId(),
                        ProfitTypeEnum.PARENT, orderInfo.getParentCommission(),
                        commissionPlatformFacadeDTO.getPlatformCommissionRageConfigInfo().getParentRate());
                superiorInfoRecode(userTbkInfo.getSuperiorUserId(), orderInfo, isInvalidOrder);
            }

            // 上上级返佣
            if (Objects.nonNull(userTbkInfo.getTopUserId())) {
                // 添加profit 信息
                recordOrderProfit(orderInfo, userTbkInfo.getTopUserId(), userTbkInfo.getId(),
                        ProfitTypeEnum.GRANDPARENT, orderInfo.getGrandparentCommission(),
                        commissionPlatformFacadeDTO.getPlatformCommissionRageConfigInfo().getGrandParentRate());
                topInfoRecode(userTbkInfo.getTopUserId(), orderInfo, isInvalidOrder);
            }

            // 内容上级
            if (Objects.nonNull(userTbkInfo.getContentSuperiorUserId())) {
                // 和正统上级、上上级都不一样才返佣
                if (!Objects.equals(userTbkInfo.getContentSuperiorUserId(), userTbkInfo.getSuperiorUserId())
                        && !Objects.equals(userTbkInfo.getContentSuperiorUserId(), userTbkInfo.getTopUserId())) {

                    // 添加profit 信息
                    recordOrderProfit(orderInfo, userTbkInfo.getContentSuperiorUserId(), userTbkInfo.getId(),
                            ProfitTypeEnum.CONTENT_PARENT, commissionPlatformFacadeDTO.getContentParentPrice(),
                            commissionPlatformFacadeDTO.getPlatformCommissionRageConfigInfo().getContentParentRate());
                    // 佣金 推送
                    contentParentCommissionInfoRecode(userTbkInfo.getContentSuperiorUserId(), orderInfo.getOrderSn(),
                            commissionPlatformFacadeDTO.getContentParentPrice(), isInvalidOrder);
                }
            }

            // 内容上上级
            if (Objects.nonNull(userTbkInfo.getContentTopUserId())) {
                // 和内容上级、正统上上级都不一样才返佣
                if (!Objects.equals(userTbkInfo.getContentSuperiorUserId(), userTbkInfo.getContentTopUserId())
                        && !Objects.equals(userTbkInfo.getContentSuperiorUserId(), userTbkInfo.getTopUserId())) {
                    // 添加profit 信息
                    recordOrderProfit(orderInfo, userTbkInfo.getContentTopUserId(), userTbkInfo.getId(),
                            ProfitTypeEnum.CONTENT_GRANDPARENT, commissionPlatformFacadeDTO.getContentGrandParentPrice(),
                            commissionPlatformFacadeDTO.getPlatformCommissionRageConfigInfo().getContentGrandParentRate());
                    // 佣金 推送
                    contentParentCommissionInfoRecode(userTbkInfo.getContentTopUserId(), orderInfo.getOrderSn(),
                            commissionPlatformFacadeDTO.getContentGrandParentPrice(), isInvalidOrder);
                }

            }
        }

        return true;
    }


    @Override
    public boolean addTakeOutOrderCommission(OrderInfo orderInfo, BigDecimal commissionRate,
                                             Long userId, boolean isPurchaseBySelf) {

        log.info("添加返佣信息, orderInfo: {}, userId: {}, isPurchaseBySelf: {}, ", JSON.toJSONString(orderInfo),
                userId, isPurchaseBySelf);

        if (Objects.isNull(orderInfo)) {
            log.warn("添加返佣信息失败，订单信息为null");
            return false;
        }

        // 如果relation id 没有对应的用户 则订单不添加返佣信息
        if (Objects.isNull(userId)) {
            log.info("用户 userId为null 无法添加佣金");
            return false;
        }

        // 添加自购佣金等信息
        // 记录profit
        recordOrderProfit(orderInfo, userId, userId,
                // 自购还是分享单
                isPurchaseBySelf ? ProfitTypeEnum.PURCHASE : ProfitTypeEnum.SHARE,
                orderInfo.getPurchaseCommission(),
                // 自购比例
                commissionRate);

        if (Objects.equals(orderInfo.getOrderStatus(), OrderStatusEnum.INVALID.getStatus())) {
            log.info("订单: {} 已失效，不进行返佣", orderInfo.getOrderSn());
            return false;
        }

        // 增加账户金额、发推送
        takeOutOrderPurchaseInfoRecode(userId, orderInfo, orderInfo.getPurchaseCommission(), isPurchaseBySelf);
        // 增加用户账户省钱金额，省钱金额为佣金，并推送
        pushMsgAndIncreaseSaveCash(orderInfo, userId, orderInfo.getPurchaseCommission());

        return true;
    }

    @Override
    public void updateInvalidOrderCommission(String orderSn) {
        log.info("订单: {} 失效 更新返佣信息", orderSn);

        try {
            OrderInfo orderInfo = orderInfoExtendMapper.getOrderInfoByOrderId(orderSn);
            if (Objects.isNull(orderInfo)) {
                log.warn("订单号: {} 查找不到对应的订单信息", orderSn);
                return;
            }

            // 更改订单状态（待结算->已结算）
            orderInfoExtendMapper.updateProfitStatus(orderInfo.getOrderSn(), OrderStatusEnum.INVALID.getStatus());

            // 取消返佣
            List<OrderInfoForQuartz> infoList = orderProfitExtendMapper.getProfitInfo(orderSn);
            if (!CollectionUtils.isEmpty(infoList)) {
                infoList.forEach(p -> {
                    // 更改账户信息
                    CashAccountParam param = new CashAccountParam();
                    param.setUserId(p.getUserId());
                    param.setAccountAction(AccountActionEnum.SUB_REBATE_CASH);
                    param.setCashFlowType(getCashFlowTypeByTbOrderType(orderInfo.getOrderType()));
                    param.setAmount(p.getAmount());
                    param.setRelationId(p.getOrderId());
                    param.setRemark("佣金退还");
                    userAccountIntegrationService.cashAccountOperation(param);
                });
            }

        } catch (Exception e) {
            log.error("订单: {} 失效 更新返佣信息 出现错误", orderSn, e);
        }
    }

    /**
     * 根据订单类型 获取账户流水类型
     *
     * @param orderType
     * @return
     */
    private CashFlowTypeEnum getCashFlowTypeByTbOrderType(Integer orderType) {
        if (Objects.nonNull(orderType) && orderType.equals(OrderTypeEnum.MEITUAN.getCode())) {
            return CashFlowTypeEnum.TAKE_OUT_REBATE_CASH;
        }

        return CashFlowTypeEnum.REBATE_CASH;
    }

    @Override
    public void payCashForLastMonthOrder(OrderInfoForQuartz orderInfo) {
        log.info("订单号: {} 定时返佣 返佣订单信息 {}", orderInfo.getOrderSn(), JSON.toJSONString(orderInfo));
        try {

            // 更改订单状态（待结算->已结算）
            orderInfoExtendMapper.updateOrderStatus(orderInfo.getOrderSn(), OrderStatusEnum.HAVE_SETTLED.getStatus());

            // 更改账户信息
            CashAccountParam param = new CashAccountParam();
            param.setUserId(orderInfo.getUserId());
            param.setAccountAction(AccountActionEnum.REBATE_SETTLE);
            param.setCashFlowType(CashFlowTypeEnum.REBATE_CASH);
            param.setAmount(orderInfo.getAmount());
            param.setRelationId(Long.parseLong(orderInfo.getOrderSn()));
            param.setRemark("返佣增加");
            userAccountIntegrationService.cashAccountOperation(param);

            log.info("订单号: {} 定时返佣金额: {} 返佣完成", orderInfo.getOrderSn(), orderInfo.getAmount());
        } catch (Exception e) {
            log.error("订单号: {} 定时返佣 出现错误", orderInfo.getOrderSn(), e);
        }
    }

    @Override
    public void payCashForLastMonthTakeOutOrder(OrderInfo orderInfo) {
        log.info("外卖订单号: {} 定时返佣 返佣订单信息 {}", orderInfo.getOrderSn(), JSON.toJSONString(orderInfo));
        try {

            // 更改订单状态（待结算->已结算）
            orderInfoExtendMapper.updateOrderStatus(orderInfo.getOrderSn(), OrderStatusEnum.HAVE_SETTLED.getStatus());

            List<OrderInfoForQuartz> profitInfo = orderProfitExtendMapper.getProfitInfo(orderInfo.getOrderSn());

            profitInfo.forEach(p -> {
                // 更改账户信息
                CashAccountParam param = new CashAccountParam();
                param.setUserId(orderInfo.getOwnerUserId());
                param.setAccountAction(AccountActionEnum.REBATE_SETTLE);
                param.setCashFlowType(CashFlowTypeEnum.TAKE_OUT_REBATE_CASH);
                param.setAmount(p.getAmount());
                param.setRelationId(Long.parseLong(orderInfo.getOrderSn()));
                param.setRemark("外卖返佣增加");
                userAccountIntegrationService.cashAccountOperation(param);

                log.info("外卖订单号: {} 定时返佣金额: {} 返佣完成", orderInfo.getOrderSn(), p.getAmount());
            });

        } catch (Exception e) {
            log.error("订单号: {} 定时返佣 出现错误", orderInfo.getOrderSn(), e);
        }
    }

    /**
     * 记录订单收益
     *
     * @param orderInfo       订单信息
     * @param profitUserId    受益人用户id
     * @param userId          订单发起人id
     * @param profitType      订单收益类型
     * @param commissionPrice 实际收益数额
     * @param commissionRate  佣金比例
     */
    private void recordOrderProfit(OrderInfo orderInfo, Long profitUserId, Long userId,
                                   ProfitTypeEnum profitType, BigDecimal commissionPrice, BigDecimal commissionRate) {

        Date now = new Date();

        OrderProfit profit = new OrderProfit();
        profit.setUserId(profitUserId);
        profit.setOrderId(orderInfo.getId());
        profit.setGoodsId(orderInfo.getGoodsId());
        profit.setType(profitType.getCode());
        profit.setOrderSn(orderInfo.getOrderSn());
        profit.setOrderOwnerUserId(userId);
        profit.setOrderStatus(orderInfo.getOrderStatus());
        profit.setProfitAmount(commissionPrice);
        profit.setProfitRate(commissionRate);
        profit.setCreateTime(now);
        profit.setModifyTime(now);

        orderProfitExtendMapper.insertSelective(profit);
    }

    /**
     * 自购信息记录
     * 添加profit
     * 推送等
     *
     * @param userId           userId
     * @param isInvalidOrder   订单是否失效
     * @param orderInfo        orderInfo
     * @param isVip            是否vip
     * @param isPurchaseBySelf 是否自购
     */
    private void purchaseInfoRecode(Long userId, boolean isInvalidOrder, OrderInfo orderInfo, BigDecimal commission,
                                    boolean isVip, boolean isPurchaseBySelf) {

        if (!isInvalidOrder) {
            // 添加账户金额
            addCheckCash(userId, commission, orderInfo.getOrderSn());

            // 增加省钱金额和推送省钱消息
            BigDecimal saveCash = calTbOrderSaveCash(orderInfo, commission);
            pushMsgAndIncreaseSaveCash(orderInfo, userId, saveCash);
            if (isPurchaseBySelf) {
                // 记录下单信息
                addOrderedRecode(userId);
            }

            // 给自购人推送佣金到账
            String content;
            if (isPurchaseBySelf) {
                content = StringUtils.join("你有一笔自购佣金已经跟单成功（淘宝订单", orderInfo.getOrderSn(), "），坐等结算提现吧～");
            } else {
                content = StringUtils.join("你有一笔分享佣金已经跟单成功（淘宝订单", orderInfo.getOrderSn(), "），坐等结算提现吧～");
            }
            messageUserIntegrationService.pushOrderCommissionCheckMsg(content, userId);
        }
    }

    /**
     * 计算淘宝订单省钱金额
     * 省钱金额=佣金 + 优惠券金额
     *
     * @param orderInfo  淘宝订单对象
     * @param commission 佣金
     * @return 省钱金额
     */
    private BigDecimal calTbOrderSaveCash(OrderInfo orderInfo, BigDecimal commission) {
        BigDecimal saveCash = commission;
        if (orderInfo.getCoupon() != null) {
            saveCash = commission.add(orderInfo.getCoupon());
        }
        return saveCash.setScale(2, RoundingMode.HALF_UP);
    }

    /**
     * 增加省钱金额并推送省钱消息
     *
     * @param orderInfo 订单对象
     * @param userId    用户id
     * @param saveCash  省钱金额
     */
    private void pushMsgAndIncreaseSaveCash(OrderInfo orderInfo, Long userId,
                                            BigDecimal saveCash) {
        if (BigDecimal.ZERO.compareTo(saveCash) < 0) {
            // 增加省钱金额
            userAccountIntegrationService.cashAccount(buildAccountCashParam(userId, saveCash));
            // 推送消息
            pushIntegrationService.pushSaveMoneyMsg(orderInfo, saveCash);
        }
    }


    /**
     * 构建调用local-user服务增加账户省钱金额接口入参
     *
     * @param userId   用户id
     * @param saveCash 省钱金额
     * @return 参数对象
     */
    private AccountCashParam buildAccountCashParam(Long userId, BigDecimal saveCash) {
        AccountCashParam param = new AccountCashParam();
        param.setUserId(userId);
        param.setCashType(CashEnum.SAVE_CASH.name());
        param.setAddTotal(true);
        param.setCash(saveCash);
        return param;
    }

    /**
     * 自购信息记录
     * 添加profit
     * 推送等
     *
     * @param userId           userId
     * @param orderInfo        orderInfo
     * @param isPurchaseBySelf 是否自购
     */
    private void takeOutOrderPurchaseInfoRecode(Long userId, OrderInfo orderInfo,
                                                BigDecimal commission, boolean isPurchaseBySelf) {
        // 添加账户金额
        addTakeoutOrderCheckCash(userId, commission, orderInfo.getOrderSn());

        // 给自购人推送佣金到账
        String content = StringUtils.join("你有一笔外卖补贴已经跟单成功（订单", orderInfo.getOrderSn(), "），坐等结算提现吧～");

        messageUserIntegrationService.pushOrderCommissionCheckMsg(content, userId);
    }

    /**
     * 给上级用户返佣
     *
     * @param superiorUserId 上级信息id
     * @param orderInfo      订单信息
     * @param isInvalidOrder 订单是否失效
     */
    private void superiorInfoRecode(Long superiorUserId, OrderInfo orderInfo, boolean isInvalidOrder) {

        // 给上级购买者返利
        if (!isInvalidOrder && orderInfo.getParentCommission().compareTo(BigDecimal.ZERO) > 0) {
            // 增加返利
            addCheckCash(superiorUserId, orderInfo.getParentCommission(), orderInfo.getOrderSn());

            // 给订单所属用户的上级用户推送佣金到账
            messageUserIntegrationService.pushOrderCommissionCheckMsg(StringUtils.join(
                    "你有一笔好友贡献佣金已经跟单成功（淘宝订单", orderInfo.getOrderSn(), "），坐等结算提现吧～"),
                    superiorUserId);
        }
    }

    /**
     * 增加审核佣金
     *
     * @param userId     用户id
     * @param commission 佣金金额
     * @param orderSn    订单号
     */
    private void addCheckCash(Long userId, BigDecimal commission, String orderSn) {
        // 添加拟结算金额
        CashAccountParam param = new CashAccountParam();
        param.setUserId(userId);
        param.setAccountAction(AccountActionEnum.ADD_REBATE_CASH);
        param.setCashFlowType(CashFlowTypeEnum.REBATE_CASH);
        param.setAmount(commission);
        param.setRelationId(Long.parseLong(orderSn));
        param.setRemark("购物返佣增加");
        userAccountIntegrationService.cashAccountOperation(param);
    }

    /**
     * 增加审核佣金
     *
     * @param userId     用户id
     * @param commission 佣金金额
     * @param orderSn    订单号
     */
    private void addTakeoutOrderCheckCash(Long userId, BigDecimal commission, String orderSn) {
        // 添加拟结算金额
        CashAccountParam param = new CashAccountParam();
        param.setUserId(userId);
        param.setAccountAction(AccountActionEnum.ADD_REBATE_CASH);
        param.setCashFlowType(CashFlowTypeEnum.TAKE_OUT_REBATE_CASH);
        param.setAmount(commission);
        param.setRelationId(Long.parseLong(orderSn));
        param.setRemark("外卖返佣增加");
        userAccountIntegrationService.cashAccountOperation(param);
    }

    /**
     * 给顶级添加佣金 推送等
     *
     * @param topUserId      顶级id
     * @param orderInfo      订单信息
     * @param isInvalidOrder 订单是否失效
     */
    private void topInfoRecode(Long topUserId, OrderInfo orderInfo, boolean isInvalidOrder) {

        // 给顶级购买者返利
        if (!isInvalidOrder && orderInfo.getParentCommission().compareTo(BigDecimal.ZERO) > 0) {
            // 添加拟结算金额
            addCheckCash(topUserId, orderInfo.getGrandparentCommission(), orderInfo.getOrderSn());

            // 给订单所属用户的上级用户推送佣金到账
            messageUserIntegrationService.pushOrderCommissionCheckMsg(StringUtils.join(
                    "你有一笔好友贡献佣金已经跟单成功（淘宝订单", orderInfo.getOrderSn(), "），坐等结算提现吧～"),
                    topUserId);
        }
    }

    /**
     * 给顶级添加佣金 推送等
     *
     * @param topUserId      顶级id
     * @param orderSn        订单号
     * @param isInvalidOrder 订单是否失效
     */
    private void contentParentCommissionInfoRecode(Long topUserId, String orderSn, BigDecimal commission, boolean isInvalidOrder) {

        // 给内容上级返利
        if (!isInvalidOrder && commission.compareTo(BigDecimal.ZERO) > 0) {
            // 添加拟结算金额
            addCheckCash(topUserId, commission, orderSn);

            // 给订单所属用户的上级用户推送佣金到账
            messageUserIntegrationService.pushOrderCommissionCheckMsg(StringUtils.join(
                    "你有一笔好友贡献佣金已经跟单成功（淘宝订单", orderSn, "），坐等结算提现吧～"),
                    topUserId);
        }
    }


    /**
     * 记录下单信息
     * 下单记录-用于新人首单
     *
     * @param userId userId
     */
    private void addOrderedRecode(Long userId) {

        // 添加用户标志位
        userIntegrationService.updateUserJudgeMarker(userId, LocalNewsUserJudgeMarkerEnum.HAS_ORDERED, true);
    }
}
