package com.bxm.localnews.market.service.order;

import com.alibaba.fastjson.JSON;
import com.bxm.component.tbk.order.model.dto.LongTimeNoCheckOrderInfo;
import com.bxm.component.tbk.order.model.dto.TbkOrderInfo;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.component.tbk.order.model.enums.OrderTypeEnum;
import com.bxm.component.tbk.order.model.enums.TbkOrderStatusEnum;
import com.bxm.component.tbk.order.service.impl.AbstractPullOrderService;
import com.bxm.localnews.market.domain.OrderInfoExtendMapper;
import com.bxm.localnews.market.dto.CommissionPlatformFacadeDTO;
import com.bxm.localnews.market.dto.UserTbkInfo;
import com.bxm.localnews.market.integration.CommodityCommissionFacadeService;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.model.constant.ProfitTypeConstant;
import com.bxm.localnews.market.model.entity.OrderExtendInfoBean;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.OrderExtendType;
import com.bxm.localnews.market.param.GetPlatformCommissionParam;
import com.bxm.localnews.market.service.OrderCommissionService;
import com.bxm.newidea.component.enums.AppNameEnum;
import com.bxm.newidea.component.uuid.SequenceCreater;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 订单拉取
 *
 * @author Gonzo
 * @date 2020-05-06 10:43
 * @since 2.0.0
 */
@Slf4j
@Service
public class PullOrderServiceImpl extends AbstractPullOrderService {

    @Autowired
    private OrderInfoExtendMapper orderInfoExtendMapper;

    @Autowired
    private OrderCommissionService orderCommissionService;

    @Autowired
    private SequenceCreater sequenceCreater;

    @Autowired
    private CommodityCommissionFacadeService commodityCommissionFacadeService;

    @Autowired
    private UserIntegrationService userIntegrationService;


    @Override
    protected List<LongTimeNoCheckOrderInfo> getLongTimeNoCheckOrders(OrderTypeEnum type) {
        // 获取30天以上的库内未结算订单进行查询二次更新
        List<OrderInfo> orderInfos = orderInfoExtendMapper.selectCreateTimeGreatThen(type.getCode(), 30);
        return orderInfos.stream().map(this::convert).collect(Collectors.toList());
    }

    private LongTimeNoCheckOrderInfo convert(OrderInfo orderInfo) {
        LongTimeNoCheckOrderInfo info = new LongTimeNoCheckOrderInfo();
        info.setOrderSn(orderInfo.getOrderSn());
        info.setSourceOrderCreateTime(orderInfo.getSourceOrderCreateTime());
        return info;
    }

    @Override
    @Transactional(rollbackFor = Exception.class)
    protected boolean doHandleCommodityOrderInfo(TbkOrderInfo orderInfo) {

        // 先查询是否已有订单
        OrderInfo orderInfoBean = orderInfoExtendMapper.getOrderInfoByOrderId(orderInfo.getOrderSn());
        if (Objects.nonNull(orderInfoBean)) {
            if (orderInfoBean.getSourceOwnerOrderStatus() == orderInfo.getSourceOwnerOrderStatus()) {
                log.info("订单号: {} 已有该订单信息 且状态未变更, 不再处理", orderInfo.getOrderSn());
                return false;
            }

            // 更新订单信息
            if (orderInfo.getSourceOwnerOrderStatus() == TbkOrderStatusEnum.INVALID.getStatus()) {
                // 订单失效
                invalidStatusAndCommission(orderInfo.getOrderSn());
            } else {
                // 订单状态更新
                orderInfoExtendMapper.updateSourceOrderStatus(orderInfo.getOrderSn(), orderInfo.getSourceOwnerOrderStatus());
            }
            return true;
        }

        boolean isVip = false;
        Long userId = null;
        boolean isPurchaseBySelf = false;
        Long superiorUserId = null;

        // 查询用户信息
        UserTbkInfo userTbkInfo = userIntegrationService.selectUserFromCacheByRelationId(orderInfo.getRelationId());
        if (Objects.nonNull(userTbkInfo)) {
            isVip = Objects.equals(userTbkInfo.getIsVip(), Boolean.TRUE);

            userId = userTbkInfo.getId();
            // 是否自购
            isPurchaseBySelf = Objects.equals(userTbkInfo.getTbkSpecialId(), orderInfo.getSpecialId());
            superiorUserId = userTbkInfo.getSuperiorUserId();
        }

        //佣金信息查询
        CommissionPlatformFacadeDTO platformCommission =
                commodityCommissionFacadeService.getPlatformCommission(GetPlatformCommissionParam.builder()
                        .commissionPrice(orderInfo.getCommission())
                        .goodsNum(orderInfo.getGoodsNum())
                        .payPrice(orderInfo.getPayPrice())
                        .goodsId(Long.valueOf(orderInfo.getGoodsId()))
                        .vip(isVip)
                        .userId(userId)
                        .purchaseBySelf(isPurchaseBySelf)
                        .superiorUserId(superiorUserId)
                        .srcApp(AppNameEnum.LOCAL_NEWS.name())
                        .build());

        if (Objects.isNull(platformCommission)
                || Objects.isNull(platformCommission.getPlatformCommissionRageConfigInfo())) {

            log.error("订单号: {} 生成订单时 获取佣金信息失败！platformCommission: {}", orderInfo.getOrderSn(),
                    JSON.toJSONString(platformCommission));

            return false;
        }

        // 构建插入信息
        OrderInfo orderInfoForInsert = build(orderInfo, platformCommission, userId, isPurchaseBySelf);
        // 新建
        addOrder(orderInfoForInsert, OrderExtendType.getByCommissionType(platformCommission.getCommissionType()));

        // 添加佣金信息
        if (Objects.isNull(userId)) {
            log.warn("订单号: {} relationId: {} 没有获取到用户 跳过添加佣金信息", orderInfo.getOrderSn(),
                    orderInfo.getRelationId());
            return false;
        }

        orderCommissionService.addCommission(orderInfoForInsert, userTbkInfo, platformCommission);
        return true;
    }

    /**
     * 更新订单状态为“失效”；同时更新账户金额
     *
     * @param orderSn 订单号
     */
    private void invalidStatusAndCommission(String orderSn) {
        orderInfoExtendMapper.updateOrderStatus(orderSn, OrderStatusEnum.INVALID.getStatus());
        orderInfoExtendMapper.updateSourceOrderStatus(orderSn, TbkOrderStatusEnum.INVALID.getStatus());

        // 同时更新账户金额
        orderCommissionService.updateInvalidOrderCommission(orderSn);
    }

    /**
     * 添加订单以及extend信息
     *
     * @param orderInfo       订单数据
     * @param orderExtendType 订单extend信息
     */
    public void addOrder(OrderInfo orderInfo, OrderExtendType orderExtendType) {
        // 添加订单信息
        orderInfoExtendMapper.insertSelective(orderInfo);

        // 添加extend info
        OrderExtendInfoBean infoBean = new OrderExtendInfoBean();
        Date now = new Date();
        infoBean.setId(sequenceCreater.nextLongId());
        infoBean.setCreateTime(now);
        infoBean.setModifyTime(now);
        infoBean.setTOrderInfoId(orderInfo.getId());
        infoBean.setType(orderExtendType.getType());
        infoBean.setExtendJson("");
        infoBean.setOwnerUserId(orderInfo.getOwnerUserId());

    }

    /**
     * 根据淘宝客订单数据和佣金信息，构建本地订单数据
     *
     * @param tbkOrderInfo       源数据
     * @param platformCommission 佣金信息
     * @return 订单数据
     */
    private OrderInfo build(TbkOrderInfo tbkOrderInfo, CommissionPlatformFacadeDTO platformCommission, Long userId, boolean isPurchaseBySelf) {
        BigDecimal couponPrice = commodityCommissionFacadeService.getCouponPrice(Long.parseLong(tbkOrderInfo.getGoodsId()));

        Date now = new Date();
        OrderInfo orderInfo = new OrderInfo();
        orderInfo.setId(sequenceCreater.nextLongId());
        orderInfo.setGoodsId(tbkOrderInfo.getGoodsId());
        orderInfo.setGoodsName(tbkOrderInfo.getGoodsName());
        orderInfo.setImgUrl(tbkOrderInfo.getImgUrl());
        orderInfo.setRelationId(tbkOrderInfo.getRelationId());
        orderInfo.setSpecialId(tbkOrderInfo.getSpecialId());
        orderInfo.setOrderSn(tbkOrderInfo.getOrderSn());
        orderInfo.setOrderParentSn(tbkOrderInfo.getOrderParentSn());
        orderInfo.setGoodsPrice(tbkOrderInfo.getGoodsPrice());
        orderInfo.setGoodsNum(tbkOrderInfo.getGoodsNum());
        orderInfo.setPayPrice(tbkOrderInfo.getPayPrice());
        orderInfo.setSourceOrderCreateTime(tbkOrderInfo.getSourceOrderCreateTime());
        orderInfo.setSourceOrderEarningTime(tbkOrderInfo.getSourceOrderEarningTime());
        orderInfo.setOrderStatus(OrderStatusEnum.getOrderStatus(tbkOrderInfo.getSourceOwnerOrderStatus()));
        orderInfo.setSourceOwnerOrderStatus(tbkOrderInfo.getSourceOwnerOrderStatus());
        orderInfo.setCommission(tbkOrderInfo.getCommission());
        //判断是否为自购订单
        orderInfo.setOrderProfitType(isPurchaseBySelf ? ProfitTypeConstant.PURCHASE : ProfitTypeConstant.SHARE);

        // 数据库添加成功再增加佣金拟结算
        orderInfo.setVipPurchaseCommission(platformCommission.getVipPurchasePrice());
        orderInfo.setPurchaseCommission(platformCommission.getPurchasePrice());
        orderInfo.setParentCommission(platformCommission.getParentPrice());
        orderInfo.setGrandparentCommission(platformCommission.getGrandParentPrice());

        orderInfo.setSource(tbkOrderInfo.getSource());
        orderInfo.setTbOrderType(tbkOrderInfo.getTbOrderType());
        orderInfo.setPid(tbkOrderInfo.getPid());
        orderInfo.setType(tbkOrderInfo.getType());
        // 购物订单
        orderInfo.setOrderType(com.bxm.localnews.market.model.enums.OrderTypeEnum.TAOBAO.getCode());
        // 优惠券金额
        orderInfo.setCoupon(couponPrice);
        orderInfo.setOwnerUserId(userId);
        orderInfo.setSourceStr(tbkOrderInfo.getSourceStr());
        orderInfo.setCreateTime(now);
        orderInfo.setModifyTime(now);

        return orderInfo;
    }
}
