package com.bxm.localnews.market.service.order;

import com.alibaba.fastjson.JSON;
import com.bxm.component.tbk.order.config.DuoMaiMeituanProperties;
import com.bxm.localnews.market.config.EleProperties;
import com.bxm.localnews.market.dto.UserTbkInfo;
import com.bxm.localnews.market.model.dto.MeituanEuid;
import com.bxm.localnews.market.integration.CommodityInfoFacadeService;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.model.param.TakeOutJumpInfoParam;
import com.bxm.localnews.market.model.param.UserIdParam;
import com.bxm.localnews.market.model.vo.EleTaoWord;
import com.bxm.localnews.market.model.vo.JumpInfo;
import com.bxm.localnews.market.service.TakeOutOrderService;
import com.bxm.newidea.component.enums.AppNameEnum;
import com.bxm.newidea.component.vo.ResponseJson;
import com.taobao.api.request.TbkActivitylinkGetRequest;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.http.client.utils.URIBuilder;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.net.URISyntaxException;
import java.util.Date;
import java.util.Objects;

@Service
@Slf4j
public class TakeOutOrderServiceImpl implements TakeOutOrderService {

    @Autowired
    private DuoMaiMeituanProperties duoMaiMeituanProperties;

    @Autowired
    private CommodityInfoFacadeService commodityFacadeService;

    @Autowired
    private EleProperties eleProperties;

    @Autowired
    private UserIntegrationService userIntegrationService;

    @Override
    public ResponseJson<JumpInfo> getJumpInfo(TakeOutJumpInfoParam param) {

        if (Objects.isNull(param.getUserId())
                || Objects.isNull(param.getJumpType())) {
            log.warn("请求参数: {} 缺失", JSON.toJSONString(param));
            return ResponseJson.badReqeuset("缺少必要参数").build();
        }

        try {
            if (param.getJumpType() == 2) {
                return ResponseJson.ok(getMeituanJumpUrl(param.getUserId()));
            }

            UserTbkInfo tbkInfo = userIntegrationService.selectUserFromCacheByUserId(param.getUserId());
            if (Objects.isNull(tbkInfo) || StringUtils.isBlank(tbkInfo.getTbkRelationId())) {
                log.warn("userId: {} 获取relation id 失败", param.getUserId());
                return ResponseJson.badReqeuset("请先绑定淘宝").build();
            }

            return ResponseJson.ok(getEleJumpUrl(tbkInfo.getTbkRelationId(), true));
        } catch (Exception e) {
            log.error("转换跳转链接失败 请求参数: {}", JSON.toJSONString(param), e);
        }

        return ResponseJson.badReqeuset("获取失败").build();
    }


    private JumpInfo getEleJumpUrl(String tbkRelationId, boolean switchToProtocol) {

        TbkActivitylinkGetRequest req = new TbkActivitylinkGetRequest();
        // 推广位id
        req.setAdzoneId(eleProperties.getAdZoneId());
        // relation_id
        req.setRelationId(tbkRelationId);
        // 官方活动id
        req.setPromotionSceneId(eleProperties.getPromotionSceneId());

        String taoBaoUnionStr = commodityFacadeService.getTaoBaoUnionStr(eleProperties.getAdZoneId(),
                tbkRelationId, eleProperties.getPromotionSceneId());

        if (log.isDebugEnabled()) {
            log.debug("请求参数: {}, 获取饿了么跳转链接返回: {}", JSON.toJSONString(req), taoBaoUnionStr);
        }

        if (StringUtils.isNotBlank(taoBaoUnionStr)) {
            String url = taoBaoUnionStr.replaceAll("\\\\", "");
            // 是否转化协议
            if (switchToProtocol && StringUtils.isNotBlank(url)) {
                if (url.contains("https")) {
                    url = url.replaceAll("https", "wst://function/openApp?url=tbopen");
                } else {
                    url = url.replaceAll("http", "wst://function/openApp?url=tbopen");
                }
            }

            return JumpInfo.builder()
                    .jumpUrl(url)
                    .build();
        }

        return null;
    }


    private JumpInfo getMeituanJumpUrl(Long userId) throws URISyntaxException {
        MeituanEuid euid = new MeituanEuid();
        euid.setUserId(userId);
        euid.setTime(new Date());
        euid.setSiteId(Objects.toString(AppNameEnum.LOCAL_NEWS));

        // http://c.duomai.com/track.php?site_id=47&aid=69&euid=&t=http%3A%2F%2Fwww.duomai.com%2F
        URIBuilder uriBuilder = new URIBuilder(duoMaiMeituanProperties.getTrackUrl());
        // 多麦的媒体id
        uriBuilder.addParameter("site_id", duoMaiMeituanProperties.getSiteId());
        // 美团的推广计划id
        uriBuilder.addParameter("aid", duoMaiMeituanProperties.getAid());
        // 自定义参数
        uriBuilder.addParameter("euid", JSON.toJSONString(euid));
        // 美团推广的目标地址
        uriBuilder.addParameter("t", duoMaiMeituanProperties.getTargetUrl());

        return JumpInfo.builder()
                .jumpUrl(uriBuilder.build().toString())
                .build();
    }

    @Override
    public ResponseJson<EleTaoWord> getEleTaoWord(UserIdParam param) {

        UserTbkInfo tbkInfo = userIntegrationService.selectUserFromCacheByUserId(param.getUserId());
        if (Objects.isNull(tbkInfo) || StringUtils.isBlank(tbkInfo.getTbkRelationId())) {
            log.warn("userId: {} 获取relation id 失败", param.getUserId());
            return ResponseJson.badReqeuset("请先绑定淘宝").build();
        }

        JumpInfo eleJumpUrl = getEleJumpUrl(tbkInfo.getTbkRelationId(), false);

        if (Objects.isNull(eleJumpUrl)) {
            return ResponseJson.badReqeuset("获取淘口令失败").build();
        }

        if (log.isDebugEnabled()) {
            log.debug("生成淘口令, url: {} imgUrl: {}, text: {}", eleJumpUrl.getJumpUrl(),
                    eleProperties.getEleTaoWordImgUrl(), eleProperties.getEleTaoWordText());
        }

        return ResponseJson.ok(EleTaoWord.builder()
                .taoWord(commodityFacadeService.createTaoWord(eleJumpUrl.getJumpUrl(),
                        eleProperties.getEleTaoWordImgUrl(), eleProperties.getEleTaoWordText()))
                .build());

    }

}
