package com.bxm.localnews.market.service.order;

import com.alibaba.fastjson.JSON;
import com.bxm.component.httpclient.utils.OkHttpUtils;
import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.localnews.market.config.Taoquan365Properties;
import com.bxm.localnews.market.domain.OrderInfoMapper;
import com.bxm.localnews.market.integration.PushIntegrationService;
import com.bxm.localnews.market.integration.UserAccountIntegrationService;
import com.bxm.localnews.market.model.dto.CardCouponOrderDTO;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.OrderTypeEnum;
import com.bxm.localnews.market.model.enums.Taoquan365OrderStatusEnum;
import com.bxm.localnews.market.param.AccountCashParam;
import com.bxm.localnews.market.service.TaoQuanOrderAcquireService;
import com.bxm.localnews.market.util.Md5Utils;
import com.bxm.localnews.user.enums.CashEnum;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.google.common.collect.ImmutableMap;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.utils.URLEncodedUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import static java.nio.charset.StandardCharsets.UTF_8;
import static org.apache.commons.lang3.StringUtils.*;

/**
 * 365淘券订单拉取服务实现类
 *
 * @author wzy
 * @date 2020/8/13 14:16
 **/
@Service
@Slf4j
public class TaoQuanOrderAcquireServiceImpl implements TaoQuanOrderAcquireService {

    private final Taoquan365Properties taoquan365Properties;

    private final OrderInfoMapper orderInfoMapper;

    private final SequenceCreater sequenceCreater;

    private final PushIntegrationService pushIntegrationService;

    private final UserAccountIntegrationService userAccountIntegrationService;

    @Autowired
    public TaoQuanOrderAcquireServiceImpl(Taoquan365Properties taoquan365Properties,
                                          OrderInfoMapper orderInfoMapper,
                                          SequenceCreater sequenceCreater,
                                          PushIntegrationService pushIntegrationService,
                                          UserAccountIntegrationService userAccountIntegrationService) {
        this.taoquan365Properties = taoquan365Properties;
        this.orderInfoMapper = orderInfoMapper;
        this.sequenceCreater = sequenceCreater;
        this.pushIntegrationService = pushIntegrationService;
        this.userAccountIntegrationService = userAccountIntegrationService;
    }

    @Override
    public List<CardCouponOrderDTO.Order> getThirdPartyOrder(LocalDateTime startTime,
                                                             LocalDateTime endTime) {
        Integer successCode = 200;
        Map<String, String> headerMap = ImmutableMap.of("Content-Type", "Application/x-www-form-urlencode");

        String result = OkHttpUtils.get(parseUrl(taoquan365Properties.getOrderUrl(), startTime, endTime), headerMap);

        if (StringUtils.isBlank(result)) {
            log.error("获取淘券365第三方订单失败");
            return Collections.emptyList();
        }

        CardCouponOrderDTO cardCouponOrderDTO = JSON.parseObject(result, CardCouponOrderDTO.class);
        if (!successCode.equals(cardCouponOrderDTO.getCode())) {
            log.error("获取淘券365第三方订单失败，code:{}, message:{}",
                    cardCouponOrderDTO.getCode(), cardCouponOrderDTO.getMessage());
            return Collections.emptyList();
        }
        return cardCouponOrderDTO.getData();
    }

    @Override
    public void handleThirdPartyOrder(CardCouponOrderDTO.Order thirdPartyOrder,
                                      boolean push, boolean increase) {
        try {
            Taoquan365OrderStatusEnum currentOrderStatus = Taoquan365OrderStatusEnum.valueOf(thirdPartyOrder.getOrderStatus());
            OrderInfo dbOrderInfo = orderInfoMapper.selectByOrderSnAndType(thirdPartyOrder.getSellOrderNo(),
                    OrderTypeEnum.TAOQUAN365.getCode());

            //判断订单是否存在，如果存在，则更改订单状态
            if (dbOrderInfo != null) {
                //如果订单状态和数据库里面的订单状态相同，则不修改订单状态
                if (dbOrderInfo.getSourceOwnerOrderStatus().equals(currentOrderStatus.getCode())) {
                    return;
                }

                //只去更改订单的状态，因为这里的所有状态都是付完款的，退款不减去省钱金额
                OrderInfo updateOrderInfo = new OrderInfo();
                updateOrderInfo.setId(dbOrderInfo.getId());
                updateOrderInfo.setSourceOwnerOrderStatus(currentOrderStatus.getCode());
                //更新原始订单信息
                updateOrderInfo.setSourceStr(JSON.toJSONString(thirdPartyOrder));
                //更新db订单状态
                updateOrderInfo.setOrderStatus(thirdPartyStatus2DbStatus(currentOrderStatus));

                orderInfoMapper.updateByPrimaryKeySelective(updateOrderInfo);
            } else {
                //如果不存在则插入记录，并增加省钱金额，并推送消息
                OrderInfo newOrderInfo = convert2DbOrderInfo(thirdPartyOrder);

                orderInfoMapper.insertSelective(newOrderInfo);

                //如果用户id为null则不需要增加省钱金额和推送省钱金额
                if (newOrderInfo.getOwnerUserId() != null && isSuccessStatus(thirdPartyOrder)) {
                    pushMsgAndIncreaseSaveCash(thirdPartyOrder, newOrderInfo, push, increase);
                }
            }
        } catch (Exception ex) {
            log.error("365淘券订单处理失败：{}, {}", JSON.toJSONString(thirdPartyOrder), ex);
        }
    }

    /**
     * 第三方订单状态转换为DB中的orderStatus
     *
     * @param sourceOrderStatus 第三方原始订单状态
     * @return 数据库中的订单状态
     */
    private int thirdPartyStatus2DbStatus(Taoquan365OrderStatusEnum sourceOrderStatus) {
        if (sourceOrderStatus == null) {
            return OrderStatusEnum.INVALID.getStatus();
        }
        OrderStatusEnum result;
        switch (sourceOrderStatus) {
            case SOLD:
                result = OrderStatusEnum.SUCCESS_PAY;
                break;
            case SUCCESS:
                result = OrderStatusEnum.VERIFICATION_PAY;
                break;
            case REFUNDED:
                result = OrderStatusEnum.REFUNDING;
                break;
            case AFTER_SALE:
                result = OrderStatusEnum.REFUND_SUCCESS;
                break;
            case EXCHANGE_CLOSE:
            default:
                result = OrderStatusEnum.CANCLE_PAY;

        }
        return result.getStatus();
    }

    /**
     * 判断订单是否为成功状态
     *
     * @param thirdPartyOrder 第三方订单对象
     * @return 是否为成功状态
     */
    private boolean isSuccessStatus(CardCouponOrderDTO.Order thirdPartyOrder) {
        return Taoquan365OrderStatusEnum.SUCCESS.name().equals(thirdPartyOrder.getOrderStatus())
                || Taoquan365OrderStatusEnum.SOLD.name().equals(thirdPartyOrder.getOrderStatus());
    }

    @Override
    public void saveTaoQuanOrderByTimeRange(String startTime, String endTime) {
        DateTimeFormatter dateTimeFormatter = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);
        LocalDateTime start = LocalDateTime.parse(startTime, dateTimeFormatter);
        LocalDateTime end = LocalDateTime.parse(endTime, dateTimeFormatter);

        // 20分钟20分钟的同步
        while (start.isBefore(end)) {
            List<CardCouponOrderDTO.Order> thirdPartyOrderList = this.getThirdPartyOrder(start, (start = start.plusMinutes(20)));
            for (CardCouponOrderDTO.Order order : thirdPartyOrderList) {
                this.handleThirdPartyOrder(order, false, true);
            }

            // api调用有限流 半秒钟一次
            try {
                TimeUnit.MILLISECONDS.sleep(500);
            } catch (InterruptedException e) {
                log.error("淘券阻塞查询失败", e);
            }
        }

    }

    /**
     * 推送消息并增加省钱金额
     *
     * @param thirdPartyOrder 第三方订单对象
     * @param orderInfo       db订单对象
     */
    private void pushMsgAndIncreaseSaveCash(CardCouponOrderDTO.Order thirdPartyOrder,
                                            OrderInfo orderInfo,
                                            boolean push,
                                            boolean increase) {
        BigDecimal saveAmount = calSaveMoney(thirdPartyOrder);
        if (BigDecimal.ZERO.compareTo(saveAmount) < 0) {
            //增加省钱金额，并推送消息
            if (increase) {
                userAccountIntegrationService.cashAccount(buildAccountCashParam(orderInfo, saveAmount));
            }

            if (push) {
                log.info("用户：{}, 通过365淘券增加省钱金额并推送，金额为{}", orderInfo.getOwnerUserId(), saveAmount);
                pushIntegrationService.pushSaveMoneyMsg(orderInfo, saveAmount);
            }
        }
    }

    /**
     * 构建调用local-user服务增加账户省钱金额接口入参
     *
     * @param dbOrderInfo 订单信息
     * @param saveAmount  省钱金额
     * @return 参数对象
     */
    private AccountCashParam buildAccountCashParam(OrderInfo dbOrderInfo, BigDecimal saveAmount) {
        AccountCashParam param = new AccountCashParam();
        param.setUserId(dbOrderInfo.getOwnerUserId());
        param.setCashType(CashEnum.SAVE_CASH.name());
        param.setAddTotal(true);
        param.setCash(saveAmount);
        return param;
    }

    /**
     * 计算订单省钱的金额，保留两位小数
     *
     * @param order 订单信息
     * @return 省钱金额
     */
    private BigDecimal calSaveMoney(CardCouponOrderDTO.Order order) {
        return (order.getOfficialPrice().subtract(order.getAmount())).setScale(2, RoundingMode.HALF_UP);
    }

    /**
     * 将淘气365第三方拉取订单对象转化为DB订单对象
     *
     * @param thirdPartyOrder 第三方订单对象
     * @return 数据库订单对象
     */
    private OrderInfo convert2DbOrderInfo(CardCouponOrderDTO.Order thirdPartyOrder) {
        OrderInfo dbOrderInfo = new OrderInfo();

        SimpleDateFormat sdf = new SimpleDateFormat(DateUtils.DATE_TIME_FORMAT);
        try {
            dbOrderInfo.setOrderSn(thirdPartyOrder.getSellOrderNo());
            dbOrderInfo.setOrderStatus(thirdPartyStatus2DbStatus(
                    Taoquan365OrderStatusEnum.valueOf(thirdPartyOrder.getOrderStatus())));
            dbOrderInfo.setSourceOwnerOrderStatus(
                    Taoquan365OrderStatusEnum.valueOf(thirdPartyOrder.getOrderStatus()).getCode());

            //修改订单状态
            dbOrderInfo.setOrderType(OrderTypeEnum.TAOQUAN365.getCode());
            dbOrderInfo.setTbOrderType(OrderTypeEnum.TAOQUAN365.getDescription());

            dbOrderInfo.setSourceOrderCreateTime(sdf.parse(thirdPartyOrder.getCreateTime()));
            //保存原始数据
            dbOrderInfo.setSourceStr(JSON.toJSONString(thirdPartyOrder));
            dbOrderInfo.setPayPrice(thirdPartyOrder.getAmount()
                    .setScale(2, RoundingMode.HALF_UP));
            dbOrderInfo.setGoodsPrice(thirdPartyOrder.getOfficialPrice()
                    .setScale(2, RoundingMode.HALF_UP));
            dbOrderInfo.setGoodsName(thirdPartyOrder.getCommodityName());
            dbOrderInfo.setCommission(thirdPartyOrder.getAgentRebate());
            dbOrderInfo.setGoodsId(String.valueOf(sequenceCreater.nextLongId()));

            Long ownerUserId = null;
            ownerUserId = getOwnerUserId(thirdPartyOrder, ownerUserId);
            dbOrderInfo.setOwnerUserId(ownerUserId);
        } catch (Exception e) {
            log.error("365卡券订单对象转换失败：{}", thirdPartyOrder.getSellOrderNo());
        }

        return dbOrderInfo;
    }

    /**
     * 如果返回的用户id不合法则返回null
     *
     * @param thirdPartyOrder 订单信息
     * @param ownerUserId     用户id
     * @return 用户id
     */
    private Long getOwnerUserId(CardCouponOrderDTO.Order thirdPartyOrder, Long ownerUserId) {
        try {
            ownerUserId = Long.parseLong(thirdPartyOrder.getOpenid());
        } catch (NumberFormatException e) {
            log.warn("365淘券返回用户id格式不正确：{}", thirdPartyOrder.getOpenid());
        }
        return ownerUserId;
    }


    /**
     * 拼接查询订单URL
     *
     * @param sourceUrl 原始url
     * @return 查询订单URL
     */
    private String parseUrl(String sourceUrl, LocalDateTime startTime,
                            LocalDateTime endTime) {
        String urlParamPrefix = "?";
        if (isBlank(sourceUrl)) {
            return sourceUrl;
        }
        //拼接参数
        if (sourceUrl.contains(urlParamPrefix)) {
            sourceUrl += "&";
        } else {
            sourceUrl += "?";
        }
        //登录账户
        sourceUrl += "login_account=" + taoquan365Properties.getLoginAccount();

        //时间戳
        sourceUrl += "&time_stamp=" + System.currentTimeMillis();

        DateTimeFormatter df = DateTimeFormatter.ofPattern(DateUtils.DATE_TIME_FORMAT);
        //开始时间
        sourceUrl += "&query_start_time=" + startTime.format(df);

        //结束时间
        sourceUrl += "&query_end_time=" + endTime.format(df);

        //解析URL获取参数信息，计算签名
        List<NameValuePair> paramList = URLEncodedUtils.parse(substring(sourceUrl, sourceUrl.indexOf("?") + 1), UTF_8);

        paramList = paramList.stream().sorted(Comparator.comparing(NameValuePair::getName)).collect(Collectors.toList());

        StringBuilder signSource = new StringBuilder();

        for (NameValuePair nameValuePair : paramList) {
            if (equalsAny(nameValuePair.getName(), taoquan365Properties.getExcludeParam())) {
                continue;
            }

            signSource.append(nameValuePair.getName()).append("=").append(nameValuePair.getValue());
            signSource.append("&");
        }

        signSource.append("secretKey").append("=").append(taoquan365Properties.getSecretKey());

        //附加签名
        sourceUrl += "&sign=" + Md5Utils.md5(signSource.toString()).toUpperCase();

        return sourceUrl;
    }
}
