/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.market.service.order.usergoods.event;

import com.bxm.localnews.market.integration.MerchantGoodsIntegrationService;
import com.bxm.localnews.market.model.constant.CommonConstant;
import com.bxm.localnews.market.model.constant.DistributedKey;
import com.bxm.localnews.market.model.dto.GoodsOrderEventDTO;
import com.bxm.localnews.market.model.dto.MerchantGoodsInfoDTO;
import com.bxm.localnews.market.model.dto.PushMessageByOrderDTO;
import com.bxm.localnews.market.model.dto.UserGoodsParam;
import com.bxm.localnews.market.model.enums.GoodsOrderEventEnum;
import com.bxm.newidea.component.redis.DistributedLock;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;
import static java.util.concurrent.TimeUnit.SECONDS;
import static org.apache.commons.lang3.StringUtils.isNotBlank;

/**
 * @author jieliGG
 * @date 2020/8/25 11:39
 * 商品订单的事件变更
 **/
@Slf4j
@Component
public abstract class AbstractGoodsOrderEvent {

    @Autowired
    private MerchantGoodsIntegrationService merchantGoodsIntegrationService;

    @Autowired
    private DistributedLock lock;

    @Autowired
    private SequenceCreater sequenceCreater;


    /**
     * 前置处理:参数校验
     * 子类可以重写
     *
     * @param orderEventDTO ： 入参
     * @return ： 验证结果
     */
    protected Message prepare(GoodsOrderEventDTO orderEventDTO) {

        String key = DistributedKey.ORDER_LOCK.copy()
                .appendKey(orderEventDTO.getUserId())
                .appendKey(orderEventDTO.getGoodsId())
                .gen();
        orderEventDTO.setKey(key);
        orderEventDTO.setRequestId(sequenceCreater.nextStringId());
        // 加锁
        if (!lock.lock(orderEventDTO.getKey(), orderEventDTO.getRequestId(), 1, SECONDS)) {
            return Message.build(false, "下单过于频繁，请稍后再试~");
        }

        return Message.build(true);
    }

    /**
     * 处理的方法
     *
     * @param orderEventDTO ： 入参
     * @return 处理结果
     */
    protected abstract Message handler(GoodsOrderEventDTO orderEventDTO);

    /**
     * 处理完成后的操作：比如发推送等
     *
     * @param orderEventDTO ： 入参
     */
    protected abstract void after(GoodsOrderEventDTO orderEventDTO);

    /**
     * 处理失败后的动作
     * 需要调用的方法去实现
     *
     * @param orderEventDTO ： 入参
     */
    protected void handlerFail(GoodsOrderEventDTO orderEventDTO) {

    }

    /**
     * FIXME 这里还是需要重构一下，把相关的校验都放到prepare里面，外面的方法就不需要那么多的校验了
     * 执行的流程方法
     *
     * @param orderEventDTO ： 入参
     * @return ： 执行结果
     */
    public Message exec(GoodsOrderEventDTO orderEventDTO) {
        //前置处理参数信息
        Message message = this.prepare(orderEventDTO);

        try {
            if (Objects.equals(Boolean.FALSE, message.isSuccess())) {
                return releaseLockAndReturn(orderEventDTO, message);
            }
            //真正的处理方法【涉及到相关的账户和订单操作】，子类去实现
            message = handler(orderEventDTO);
            if (Objects.equals(Boolean.FALSE, message.isSuccess())) {
                handlerFail(orderEventDTO);
                return releaseLockAndReturn(orderEventDTO, message);
            } else {
                orderEventDTO.setOrderInfo(message.getParam(CommonConstant.CREATE_ORDER_INFO_KEY));
            }
            //处理的后续动作
            after(orderEventDTO);

            return releaseLockAndReturn(orderEventDTO, message);
        } catch(Exception e) {
            log.error("处理订单出现异常 orderEventDTO: {}", toJSONString(orderEventDTO), e);
        }

        return releaseLockAndReturn(orderEventDTO, Message.build(false, "处理订单失败"));
    }

    /**
     * 释放资源并返回
     *
     * @param orderEventDTO 包含资源key的请求对象
     * @param message 处理结果
     * @return 处理结果
     */
    private Message releaseLockAndReturn(GoodsOrderEventDTO orderEventDTO, Message message) {

        if (isNotBlank(orderEventDTO.getKey()) && isNotBlank(orderEventDTO.getRequestId())) {
            lock.unlock(orderEventDTO.getKey(), orderEventDTO.getRequestId());
        }

        return message;
    }

    /**
     * 获取动作事件
     *
     * @return ： 事件类型
     */
    protected abstract GoodsOrderEventEnum goodsOrderEventEnum();

    /**
     * 初始化用户下单参数
     *
     * @return
     */
    protected UserGoodsParam getUserGoodsParam(GoodsOrderEventDTO orderEventDTO) {
        UserGoodsParam userGoodsParam = new UserGoodsParam();
        userGoodsParam.setGoodsId(orderEventDTO.getGoodsId());
        userGoodsParam.setOpenVip(orderEventDTO.getOpenVip());
        userGoodsParam.setShareUserId(orderEventDTO.getShareUserId());
        userGoodsParam.setUserId(orderEventDTO.getUserId());
        userGoodsParam.setOrderNo(orderEventDTO.getOrderNo());
        userGoodsParam.setUserName(orderEventDTO.getUserName());
        userGoodsParam.setUserPhone(orderEventDTO.getPhone());
        userGoodsParam.setNum(orderEventDTO.getNum());
        userGoodsParam.setOrderInfo(orderEventDTO.getOrderInfo());
        return userGoodsParam;
    }

    /**
     * 获取订单推送的数据
     *
     * @param orderEventDTO
     * @return
     */
    protected PushMessageByOrderDTO getOrderDTO(GoodsOrderEventDTO orderEventDTO) {
        MerchantGoodsInfoDTO goodsInfoDTO = merchantGoodsIntegrationService.getMerchantGoodsById(Long.parseLong(orderEventDTO.getOrderInfo().getGoodsId()));
        PushMessageByOrderDTO orderDTO = new PushMessageByOrderDTO();
        orderDTO.setNickname(orderEventDTO.getUserName());
        orderDTO.setPushUserId(goodsInfoDTO.getUserId());
        orderDTO.setMerchantId(goodsInfoDTO.getMerchantId());
        orderDTO.setGoodsName(goodsInfoDTO.getGoodsName());
        return orderDTO;
    }

}
