/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.market.service.order.usergoods.impl;

import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.localnews.market.domain.OrderInfoMapper;
import com.bxm.localnews.market.dto.MerchantRateDTO;
import com.bxm.localnews.market.integration.MerchantGoodsIntegrationService;
import com.bxm.localnews.market.model.dto.UserGoodsParam;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.ProfitTypeEnum;
import com.bxm.localnews.market.service.OrderCommissionService;
import com.bxm.localnews.market.service.order.usergoods.OrderStateService;
import com.bxm.newidea.component.vo.Message;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Date;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;
import static java.math.BigDecimal.ZERO;
import static java.util.Objects.isNull;

/**
 * @author jieliGG
 * @date 2020/8/25 10:44
 **/
@Slf4j
@Component
public abstract class AbstractOrderState implements OrderStateService {

    @Autowired
    protected MerchantGoodsIntegrationService merchantGoodsIntegrationService;

    @Autowired
    protected OrderInfoMapper orderInfoMapper;

    @Autowired
    private OrderCommissionService orderCommissionService;

    @Override
    public Message initOrder(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    @Override
    public Message unPayOrder(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    @Override
    public Message pay(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    @Override
    public Message refund(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    @Override
    public Message notRefund(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    @Override
    public Message yesRefund(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    @Override
    public Message verification(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    @Override
    public Message cancel(UserGoodsParam userGoodsParam) {
        return Message.build(false);
    }

    /**
     * 佣金=（支付金额-手续费）* 佣金比率
     *
     * @param payPrice       : 支付金额
     * @param commissionRate : 佣金比率
     * @return 获取佣金
     */
    protected final BigDecimal getCommission(BigDecimal payPrice, BigDecimal commissionRate) {
        payPrice = payPrice.subtract(getServiceMoney(payPrice));
        return payPrice.multiply(commissionRate).setScale(2, RoundingMode.DOWN);
    }

    /**
     * 提现费 = 支付金额 * 费率 || 不足1分钱不收取手续费 ，所以这里可能返回0
     *
     * @return 获取提现费用
     */
    protected final BigDecimal getServiceMoney(BigDecimal payPrice) {
        BigDecimal serviceMoney = payPrice.multiply(getRate(false, ProfitTypeEnum.SERVICE));
        //小于0.01的服务费，不收钱
        if (Objects.equals(serviceMoney.compareTo(BigDecimal.valueOf(0.01)), -1)) {
            serviceMoney = ZERO;
        }
        return serviceMoney;
    }

    /**
     * @param payPrice       : 支付金额
     * @param commissionRate : 佣金比率
     * @return 获取商家收到金额
     */
    protected final BigDecimal getReceiveMoney(BigDecimal payPrice, BigDecimal commissionRate) {
        BigDecimal commission = this.getCommission(payPrice, commissionRate);
        BigDecimal serviceMoney = this.getServiceMoney(payPrice);
        return payPrice.subtract(serviceMoney).subtract(commission);
    }

    /**
     * 分享人/购买人 奖励 ： 向下取整
     *
     * @param commissionPrice : 金额
     * @param vip             : 分享人是否是VIP
     * @return 获取分享人奖励金额
     */
    protected final BigDecimal getShareMoney(BigDecimal commissionPrice, boolean vip) {
        BigDecimal rate = getRate(vip, ProfitTypeEnum.SHARE);
        return commissionPrice.multiply(rate)
                .setScale(2, RoundingMode.DOWN);
    }

    /**
     * 获取发送奖励金额 ： 向下取整
     *
     * @param commissionPrice : 金额
     * @param vip             : 分享人是否是VIP
     * @return 获取金额
     */
    protected final BigDecimal getMasterMoney(BigDecimal commissionPrice, boolean vip, ProfitTypeEnum profitTypeEnum) {
        BigDecimal rate = getRate(vip, profitTypeEnum);
        return commissionPrice.multiply(rate).setScale(2, RoundingMode.DOWN);
    }


    /**
     * 是否是VIP，来返回不同的比率
     *
     * @param vip            ： true 是vip
     * @param profitTypeEnum : 收益类型
     * @return : 默认返回VIP分成比例
     */
    protected final BigDecimal getRate(Boolean vip, ProfitTypeEnum profitTypeEnum) {
        MerchantRateDTO merchantRateDTO = merchantGoodsIntegrationService.getMerchantGoodsRate();
        if (isNull(merchantRateDTO)) {
            return ZERO;
        }
        boolean isVip = Objects.equals(vip, Boolean.TRUE);


        if (Objects.equals(profitTypeEnum, ProfitTypeEnum.SHARE) || Objects.equals(profitTypeEnum, ProfitTypeEnum.PURCHASE)) {
            if (isVip) {
                return merchantRateDTO.getVipUserRate();
            } else {
                return merchantRateDTO.getUserRate();
            }
        } else if (Objects.equals(profitTypeEnum, ProfitTypeEnum.PARENT)) {
            if (isVip) {
                return merchantRateDTO.getVipUserInviteRate();
            } else {
                return merchantRateDTO.getUserInviteRate();
            }
        } else if (Objects.equals(profitTypeEnum, ProfitTypeEnum.GRANDPARENT)) {
            if (isVip) {
                return merchantRateDTO.getVipUserTwoInviteRate();
            } else {
                return merchantRateDTO.getUserTwoInviteRate();
            }
        } else if (Objects.equals(profitTypeEnum, ProfitTypeEnum.SERVICE)) {
            return merchantRateDTO.getServiceRate();
        } else {
            return merchantRateDTO.getVipRate();
        }
    }

    /**
     *
     * @param parentSn 团购订单的订单号
     * @param orderType 订单类型
     */
    protected void cancelCommission(String parentSn, Integer orderType) {

        log.info("团购订单订单: {} 类型: {} 取消分佣", parentSn, orderType);

        // 当前订单号作为parentOrderSn进行查询
        OrderInfo orderInfo = orderInfoMapper.selectByParentOrderSn(parentSn);
        if (isNull(orderInfo)) {
            log.warn("订单: {}类型: {} 不存在", parentSn, orderType);
            return;
        }

        try {
            // 更新订单状态
            OrderInfo update = new OrderInfo();
            Date now = new Date();
            update.setOrderStatus(OrderStatusEnum.INVALID.getStatus());
            update.setModifyTime(now);
            update.setId(orderInfo.getId());

            orderInfoMapper.updateByPrimaryKeySelective(update);

            // 扣钱
            orderCommissionService.updateInvalidOrderCommission(orderInfo.getOrderSn());
        } catch(Exception e) {
            log.error("扣除佣金失败, 订单: {} 类型: {}", parentSn, orderType, e);
        }
    }
}

