/*
 * Copyright 2017 bianxianmao.com All right reserved. This software is the confidential and proprietary information of
 * bianxianmao.com ("Confidential Information"). You shall not disclose such Confidential Information and shall use it
 * only in accordance with the terms of the license agreement you entered into with bianxianmao.com.
 */
package com.bxm.localnews.market.service.order.usergoods.impl;

import com.bxm.component.tbk.order.model.enums.OrderStatusEnum;
import com.bxm.localnews.common.util.NidGeneratorUtil;
import com.bxm.localnews.constants.RedisConfig;
import com.bxm.localnews.market.config.WstOrderProperties;
import com.bxm.localnews.market.domain.OrderGoodsInfoExtMapper;
import com.bxm.localnews.market.dto.MerchantRateDTO;
import com.bxm.localnews.market.dto.UserInfoDTO;
import com.bxm.localnews.market.integration.UserIntegrationService;
import com.bxm.localnews.market.integration.UserVipIntegrationService;
import com.bxm.localnews.market.model.constant.CommonConstant;
import com.bxm.localnews.market.model.dto.MerchantGoodsInfoDTO;
import com.bxm.localnews.market.model.dto.MerchantOrderInfoDTO;
import com.bxm.localnews.market.model.dto.UserGoodsParam;
import com.bxm.localnews.market.model.entity.OrderGoodsInfoExt;
import com.bxm.localnews.market.model.entity.OrderInfo;
import com.bxm.localnews.market.model.enums.OrderTypeEnum;
import com.bxm.localnews.market.model.param.OrderCancelTaskParam;
import com.bxm.localnews.market.timer.OrderCancelTask;
import com.bxm.localnews.user.enums.AppConst;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.schedule.ScheduleService;
import com.bxm.newidea.component.schedule.builder.OnceTask;
import com.bxm.newidea.component.schedule.builder.OnceTaskBuilder;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.RandomUtils;
import com.bxm.newidea.component.tools.SpringContextHolder;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Calendar;
import java.util.Date;
import java.util.Objects;

import static com.alibaba.fastjson.JSON.toJSONString;
import static com.bxm.localnews.market.model.enums.MerchantOrderStatusEnum.ORDERED;
import static org.apache.commons.lang3.StringUtils.join;

/**
 * @author jieliGG
 * @date 2020/8/25 10:43
 * 初始化订单，代付款
 **/
@Slf4j
@Component
@AllArgsConstructor
public class InitOrderState extends AbstractOrderState {

    private final SequenceCreater sequenceCreater;

    private final UserVipIntegrationService userVipIntegrationService;

    private final PaySuccessOrderState paySuccessOrderState;

    private final UnpayOrderState unpayOrderState;

    private final RedisStringAdapter redisStringAdapter;

    private final OrderGoodsInfoExtMapper orderGoodsInfoExtMapper;

    private final UserIntegrationService userIntegrationService;

    private final ScheduleService scheduleService;

    private final WstOrderProperties wstOrderProperties;

    /**
     * 一天的超时时间
     **/
    private static final Long onedayTimeout = 60 * 60 * 24L + 10L;

    private static final BigDecimal RATE = new BigDecimal("0.01");

    @Override
    public Message initOrder(UserGoodsParam userGoodsParam) {
        //如果有订单号，说明是去支付的功能
        //特殊说明： vip卡的价格我保存在了orderInfo的 openVip字段，所以返回的payPrice需要加上这个VIP的价格
        if (null != userGoodsParam.getOrderNo()) {
            OrderInfo orderInfo = orderInfoMapper.selectByOrderSnAndOrderType(userGoodsParam.getOrderNo(), OrderTypeEnum.WANSHITONG_ONE.getCode());
            if (null != orderInfo && Objects.equals(orderInfo.getOrderStatus(), OrderStatusEnum.UNPAY.getStatus())) {
                orderInfo.setPayPrice(orderInfo.getPayPrice().add(orderInfo.getCoupon()));
            }
            userGoodsParam.setOrderInfo(orderInfo);
            return Message.build(true);
        }

        //同一件商品是否已经购买过了
        OrderInfo orderInfo = orderInfoMapper.selectLastOrderByGidStatus(userGoodsParam.getGoodsId(), OrderStatusEnum.UNPAY.getStatus(), userGoodsParam.getUserId());
        if (null != orderInfo) {
            //修改订单的部分信息
            resetOrderInfo(orderInfo, userGoodsParam);
        } else {
            //保存订单信息
            orderInfo = this.getOrderInfo(userGoodsParam);
            orderInfoMapper.insert(orderInfo);
            //增加商家订单信息
            userGoodsParam.setOrderInfo(orderInfo);
            insertMerchantOrder(userGoodsParam, orderInfo);
        }
        userGoodsParam.setOrderInfo(orderInfo);

        //缓存分享人信息
        if (Objects.nonNull(userGoodsParam.getShareUserId())) {
            redisStringAdapter.set(RedisConfig.MARKET_RECORD_GOODS_SHARE_USER.copy().appendKey(orderInfo.getOrderSn()), userGoodsParam.getShareUserId(), onedayTimeout);
        }

        //判断是否有开通vip
        if (Objects.equals(userGoodsParam.getOpenVip(), Boolean.TRUE)) {
            userGoodsParam.getOrderInfo().setPayPrice(userGoodsParam.getOrderInfo().getPayPrice().add(userGoodsParam.getOrderInfo().getOpenVip()));
        }

        // 处理超时未支付
        orderCancelTask(orderInfo);

        return Message.build(true).addParam(CommonConstant.CREATE_ORDER_INFO_KEY, orderInfo);
    }

    private void orderCancelTask(OrderInfo orderInfo) {

        try {
            OrderCancelTask orderCancelTask = SpringContextHolder.getBean(OrderCancelTask.class);
            if (Objects.isNull(orderCancelTask)) {
                log.warn("bean: orderCancelTask 不存在 无法生成订单未超时任务");
            }

            OrderCancelTaskParam param = new OrderCancelTaskParam();
            param.setOrderSn(orderInfo.getOrderSn());

            String taskName = join("订单超时未支付任务", ": ", orderInfo.getOwnerUserId(), "-", orderInfo.getGoodsName());
            String author = "gonzo";
            Date fireTime = DateUtils.addField(new Date(), Calendar.MINUTE, wstOrderProperties.getOrderExpireTimeMinute().intValue());

            log.info("生成订单超时未支付任务, param: {} taskName: {}, author: {}, fireTime: {}", toJSONString(param),
                    taskName, author, fireTime);

            //手动注册的定时任务
            OnceTask onceTask = OnceTaskBuilder.builder(new Date(), orderCancelTask)
                    .callbackParam(param)
                    .taskName(taskName)
                    .author(author)
                    .description("订单超时未支付任务")
                    .fireTime(fireTime)
                    .build();

            scheduleService.push(onceTask);
        } catch (Exception e) {
            log.error("发送订单超时未支付任务失败, orderInfo: {}", toJSONString(orderInfo), e);
        }
    }

    @Override
    public Message pay(UserGoodsParam userGoodsParam) {
        return paySuccessOrderState.pay(userGoodsParam);
    }

    @Override
    public Message unPayOrder(UserGoodsParam userGoodsParam) {
        return unpayOrderState.unPayOrder(userGoodsParam);
    }

    /**
     * 重新设置orderInfo的信息
     *
     * @param orderInfo      ： 要设置的订单信息
     * @param userGoodsParam ： 主要是要拿他的入参。
     */
    private void resetOrderInfo(OrderInfo orderInfo, UserGoodsParam userGoodsParam) {
        if (Objects.equals(orderInfo.getGoodsNum(), userGoodsParam.getNum())) {
            //不用重新算价格了，我只是去更改时间和，爽歪歪
            orderInfoMapper.updateSctimeNumById(orderInfo.getId(), userGoodsParam.getNum());
        } else {
            //订单数量不同，就要根据数量重新计算订单价格，扣库存或者换库存。
        }
    }

    /**
     * 根据传过来的参数初始化一个订单信息
     *
     * @param goodsParam ：
     * @return 订单信息
     */
    private OrderInfo getOrderInfo(UserGoodsParam goodsParam) {
        OrderInfo orderInfo = new OrderInfo();
        MerchantGoodsInfoDTO goodsInfoDTO = merchantGoodsIntegrationService.getMerchantGoodsById(goodsParam.getGoodsId());
        orderInfo.setId(sequenceCreater.nextLongId());
        orderInfo.setGoodsId(String.valueOf(goodsParam.getGoodsId()));
        orderInfo.setGoodsName(goodsInfoDTO.getGoodsName());
        orderInfo.setImgUrl(goodsInfoDTO.getGoodsImg());
        goodsParam.setGoodsInfoDTO(goodsInfoDTO);

        orderInfo.setOrderSn(NidGeneratorUtil.getOrderNo(OrderTypeEnum.WANSHITONG_ONE.getCode().toString()));
        orderInfo.setGoodsNum(goodsParam.getNum());
        MerchantRateDTO merchantRateDTO = merchantGoodsIntegrationService.getMerchantGoodsRate();
        if (Objects.equals(goodsInfoDTO.getVipDiscount(), AppConst.ENABLE)
                && !Objects.equals(merchantRateDTO.getVipRate(), BigDecimal.ZERO)
                && (goodsParam.getOpenVip() || userVipIntegrationService.isVip(goodsParam.getUserId()))) {
            // VIP 自购返佣=折扣价格 单价 * 数量 * vip比率 向上取两位小数点
            orderInfo.setVipPurchaseCommission(goodsInfoDTO.getPrice()
                    .multiply(new BigDecimal(goodsParam.getNum()))
                    .multiply(merchantRateDTO.getVipRate())
                    .setScale(2, RoundingMode.DOWN));
            //支付价格
            orderInfo.setPayPrice(goodsInfoDTO.getPrice().multiply(new BigDecimal(goodsParam.getNum())).subtract(orderInfo.getVipPurchaseCommission()));
        } else {
            orderInfo.setVipPurchaseCommission(BigDecimal.ZERO);
            orderInfo.setPayPrice(goodsInfoDTO.getPrice().multiply(new BigDecimal(goodsParam.getNum())));
        }
        if (Objects.equals(goodsParam.getOpenVip(), Boolean.TRUE)) {
            orderInfo.setOpenVip(userVipIntegrationService.getVipInfo().getVipPrice());
        } else {
            orderInfo.setOpenVip(BigDecimal.ZERO);
        }
        //佣金= 支付价格 * (1-手续费率) * 商家配置的佣金比例 向下取两位小数
        if (Objects.nonNull(goodsInfoDTO.getCommissionRate())) {
            BigDecimal rate = goodsInfoDTO.getCommissionRate().multiply(RATE);
            orderInfo.setCommission(orderInfo.getPayPrice().multiply(rate).setScale(2, RoundingMode.DOWN));
        }

        orderInfo.setGoodsPrice(goodsInfoDTO.getPrice());
        orderInfo.setOrderStatus(com.bxm.component.tbk.order.model.enums.OrderStatusEnum.UNPAY.getStatus());
        orderInfo.setSource(OrderTypeEnum.WANSHITONG_ONE.name());
        orderInfo.setTbOrderType(OrderTypeEnum.WANSHITONG_ONE.getDescription());
        orderInfo.setOrderType(OrderTypeEnum.WANSHITONG_ONE.getCode());
        orderInfo.setType(com.bxm.component.tbk.order.model.enums.OrderTypeEnum.CHANNEL_ORDER.getCode());
        orderInfo.setOwnerUserId(goodsParam.getUserId());
        orderInfo.setCreateTime(new Date());
        orderInfo.setVerificationCode(generateVerification());
        orderInfo.setSourceOrderCreateTime(new Date());
        //添加用户信息
        if (StringUtils.isEmpty(goodsParam.getUserName()) || StringUtils.isEmpty(goodsParam.getUserPhone())) {
            UserInfoDTO userInfoDTO = userIntegrationService.selectUserFromCache(goodsParam.getUserId());
            if (null != userInfoDTO) {
                orderInfo.setOwnerUserName(userInfoDTO.getNickname());
                orderInfo.setOwnerUserPhone(userInfoDTO.getPhone());
            }
        } else {
            orderInfo.setOwnerUserName(goodsParam.getUserName());
            orderInfo.setOwnerUserPhone(goodsParam.getUserPhone());
        }
        // 新增各项配置的比率
        OrderGoodsInfoExt orderGoodsInfoExt = new OrderGoodsInfoExt();
        orderGoodsInfoExt.setCommissionRate(goodsInfoDTO.getCommissionRate());
        orderGoodsInfoExt.setOrderSn(orderInfo.getOrderSn());
        orderGoodsInfoExt.setServiceRate(merchantRateDTO.getServiceRate());
        orderGoodsInfoExt.setVipRate(merchantRateDTO.getVipRate());
        orderGoodsInfoExt.setType(OrderTypeEnum.WANSHITONG_ONE.getCode());
        orderGoodsInfoExtMapper.insert(orderGoodsInfoExt);
        return orderInfo;
    }

    /**
     * 返回核销码
     *
     * @return ： 生成核销码
     */
    private Long generateVerification() {
        int bucketIndex = RandomUtils.nextInt(10, 99);

        KeyGenerator key = RedisConfig.VERIFICATION_CODE_BUCKET.copy().appendKey(bucketIndex);

        if (!redisStringAdapter.hasKey(key)) {
            long initValue = bucketIndex * 1000000 + RandomUtils.nextInt(10000, 99999);
            redisStringAdapter.incrementWithDefault(key, initValue, 1);
            return initValue;
        }

        Long codeValue = redisStringAdapter.increment(key);
        // 越界后重试其他bucket
        if (codeValue >= (bucketIndex + 1) * 1000000) {
            log.warn("触发越界：{}，生成码：{}", bucketIndex, codeValue);
            return generateVerification();
        }
        return codeValue;
    }

    /**
     * 新增商家订单信息
     *
     * @param userGoodsParam ： 订单
     */
    private void insertMerchantOrder(UserGoodsParam userGoodsParam, OrderInfo orderInfo) {
        MerchantOrderInfoDTO merchantOrder = new MerchantOrderInfoDTO();
        merchantOrder.setOrderNo(userGoodsParam.getOrderInfo().getOrderSn());
        merchantOrder.setGoodsId(Long.parseLong(userGoodsParam.getOrderInfo().getGoodsId()));
        // 0:代付款
        merchantOrder.setState(ORDERED.getStatus());
        merchantOrder.setOrderNo(userGoodsParam.getOrderInfo().getOrderSn());
        merchantOrder.setGoodsId(Long.parseLong(userGoodsParam.getOrderInfo().getGoodsId()));
        merchantOrder.setCreateTime(new Date());
        merchantOrder.setGoodsName(userGoodsParam.getOrderInfo().getGoodsName());
        merchantOrder.setGoodsImg(orderInfo.getImgUrl());
        //获取商家信息
        if (null == userGoodsParam.getGoodsInfoDTO()) {
            MerchantGoodsInfoDTO goodsInfoDTO = merchantGoodsIntegrationService.getMerchantGoodsById(Long.parseLong(userGoodsParam.getOrderInfo().getGoodsId()));
            if (null == goodsInfoDTO) {
                goodsInfoDTO = new MerchantGoodsInfoDTO();
            }
            userGoodsParam.setGoodsInfoDTO(goodsInfoDTO);
        }
        merchantOrder.setMerchantId(userGoodsParam.getGoodsInfoDTO().getMerchantId());
        merchantOrder.setMerchantName(userGoodsParam.getGoodsInfoDTO().getMerchantName());
        // 剔除开通vip的金额
        if (Objects.equals(Boolean.TRUE, userGoodsParam.getOpenVip())) {
            merchantOrder.setPayMoney(userGoodsParam.getOrderInfo().getPayPrice().subtract(orderInfo.getOpenVip()));
        } else {
            merchantOrder.setPayMoney(userGoodsParam.getOrderInfo().getPayPrice());
        }
        merchantOrder.setGoodsNum(userGoodsParam.getNum());
        merchantOrder.setUserId(userGoodsParam.getOrderInfo().getOwnerUserId());
        merchantOrder.setUserName(userGoodsParam.getOrderInfo().getOwnerUserName());
        merchantOrder.setUserPhone(userGoodsParam.getOrderInfo().getOwnerUserPhone());
        merchantGoodsIntegrationService.addMerchantOrder(merchantOrder);
    }

}
