package com.bxm.localnews.merchant.controller;

import com.bxm.localnews.merchant.coupon.emnus.UserCouponEventEnum;
import com.bxm.localnews.merchant.coupon.flow.adapter.UserCouponStateMachineAdapter;
import com.bxm.localnews.merchant.coupon.service.UserCouponService;
import com.bxm.localnews.merchant.dto.coupon.CashCouponDTO;
import com.bxm.localnews.merchant.param.coupon.UserCouponOrderParam;
import com.bxm.newidea.component.vo.Message;
import io.swagger.annotations.Api;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.*;

/**
 * @author liujia
 * @date 2020-09-28 11:21
 **/
@Api(tags = "11-91 [内部接口] 优惠券相关接口")
@RequestMapping("facade/merchant/coupon")
@RestController
@Slf4j
public class CouponFacadeController {

    private final UserCouponService userCouponService;

    private final UserCouponStateMachineAdapter userCouponStateMachineAdapter;

    @Autowired
    public CouponFacadeController(UserCouponService userCouponService,
                                  UserCouponStateMachineAdapter userCouponStateMachineAdapter) {
        this.userCouponService = userCouponService;
        this.userCouponStateMachineAdapter = userCouponStateMachineAdapter;
    }

    @ApiOperation(value = "11-91-01 获取可用的现金优惠券信息", notes = "如果返回null则表示优惠券不存在或已不可使用")
    @RequestMapping("getUserCouponInfo")
    public ResponseEntity<CashCouponDTO> getUserCouponInfo(@RequestParam("userCouponId") Long userCouponId) {
        return ResponseEntity.ok(userCouponService.getUsableCashCoupon(userCouponId));
    }

    @ApiOperation(value = "11-91-02 用户下单时锁定优惠券", notes = "锁定优惠券信息，将其更新为待确认状态")
    @PostMapping("lock")
    public ResponseEntity<Message> lockUserCoupon(@RequestBody UserCouponOrderParam param) {
        log.info("调用【用户下单时锁定优惠券】接口，参数：{}", param);
        boolean accept = userCouponStateMachineAdapter.postEvent(param, UserCouponEventEnum.ORDER);
        return ResponseEntity.ok(Message.build(accept));
    }

    @ApiOperation(value = "11-91-03 用户取消订单数，释放优惠券", notes = "将用户优惠券还原恢复原始状态")
    @PostMapping("release")
    public ResponseEntity<Message> releaseUserCoupon(@RequestBody UserCouponOrderParam param) {
        boolean accept = userCouponStateMachineAdapter.postEvent(param, UserCouponEventEnum.CANCEL);
        return ResponseEntity.ok(Message.build(accept));
    }

    @ApiOperation(value = "11-91-04 订单付款成功，消费优惠券")
    @PostMapping("consume")
    public ResponseEntity<Message> consumeUserCoupon(@RequestBody UserCouponOrderParam param) {
        boolean accept = userCouponStateMachineAdapter.postEvent(param, UserCouponEventEnum.PAY);
        return ResponseEntity.ok(Message.build(accept));
    }
}
