package com.bxm.localnews.merchants.cache.impl;

import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.domain.MerchantCollectMapper;
import com.bxm.localnews.merchants.cache.collected.MerchantCollectRedisRefresh;
import com.bxm.localnews.merchants.vo.MerchantCollectedVo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import com.fasterxml.jackson.core.type.TypeReference;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.Collections;
import java.util.List;
import java.util.Objects;

/**
 * @author pf.w
 * @date 2020/9/22 17:38
 **/
@Service
@Log4j2
@AllArgsConstructor
public class MerchantCollectRedisRefreshImpl implements MerchantCollectRedisRefresh {

    private final RedisHashMapAdapter redisHashMapAdapter;

    private final MerchantCollectMapper merchantCollectMapper;

    @Override
    public List<MerchantCollectedVo> getCollectedMerchants(Long userId) {
        TypeReference<List<MerchantCollectedVo>> typeReference = new TypeReference<List<MerchantCollectedVo>>() {
        };
        List<MerchantCollectedVo> collectedMerchants = redisHashMapAdapter.get(buildKey(), userId.toString(), typeReference);

        if (CollectionUtils.isEmpty(collectedMerchants)) {
            //缓存没有的话就重新从库里取一次
            collectedMerchants = this.loadInfoFromDb(userId);

            //还没有返回空对象
            if (CollectionUtils.isEmpty(collectedMerchants)) {
                return Collections.emptyList();
            }
        }
        return collectedMerchants;
    }

    @Override
    public boolean checkMerchantIsCollected(Long userId, Long merchantId) {
        TypeReference<List<MerchantCollectedVo>> typeReference = new TypeReference<List<MerchantCollectedVo>>() {
        };
        List<MerchantCollectedVo> collectedMerchants = redisHashMapAdapter.get(buildKey(), userId.toString(), typeReference);

        if (CollectionUtils.isEmpty(collectedMerchants)) {
            return false;
        }

        boolean flag = false;
        for (MerchantCollectedVo collected : collectedMerchants) {
            if (Objects.equals(collected.getMerchantId(), merchantId)) {
                flag = true;
                break;
            }
        }
        return flag;
    }

    @Async
    @Override
    public void refreshMerchantCollect(Long userId) {
        this.loadInfoFromDb(userId);
    }

    @Override
    public List<MerchantCollectedVo> loadInfoFromDb(Long userId) {
        List<MerchantCollectedVo> collectedMerchants = merchantCollectMapper.getCollectMerchantIds(userId);

        if (!CollectionUtils.isEmpty(collectedMerchants)) {
            redisHashMapAdapter.put(buildKey(), userId.toString(), collectedMerchants);
            return collectedMerchants;
        } else {
            redisHashMapAdapter.remove(buildKey(),userId.toString());
            return Collections.emptyList();
        }
    }

    private KeyGenerator buildKey() {
        return RedisConfig.MERCHANT_COLLECT_MERCHANTS.copy();
    }
}
