package com.bxm.localnews.merchants.cache.impl;

import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.domain.MerchantInfoMapper;
import com.bxm.localnews.merchants.cache.detail.MerchantInfoRedisRefresh;
import com.bxm.localnews.merchants.dto.MerchantInfoCacheDTO;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeanUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import java.util.Objects;

/**
 * @author: pf.w
 * @date: 2020/9/24 19:34
 **/
@Service
@Log4j2
@AllArgsConstructor
public class MerchantInfoRedisRefreshImpl implements MerchantInfoRedisRefresh {

    private final MerchantInfoMapper merchantInfoMapper;

    private final RedisHashMapAdapter redisHashMapAdapter;

    @Override
    public MerchantInfoCacheDTO getMerchantInfoCache(Long merchantId) {
        MerchantInfoCacheDTO info = redisHashMapAdapter.get(cacheKey(), merchantId.toString(), MerchantInfoCacheDTO.class);
        if (Objects.isNull(info)) {
            //重新获取下信息
            info = this.reloadInfo(merchantId);
            //还没有返回空对象
            if (Objects.isNull(info)) {
                return null;
            }
        }
        return info;
    }

    @Override
    public MerchantInfo getMerchantInfo(Long merchantId) {
        MerchantInfoCacheDTO cacheInfo = this.getMerchantInfoCache(merchantId);
        if (Objects.isNull(cacheInfo)) {
            return null;
        }
        MerchantInfo info = new MerchantInfo();
        BeanUtils.copyProperties(cacheInfo, info);
        return info;
    }

    @Async
    @Override
    public void merchantInfoRefresh(Long merchantId) {
        if (Objects.isNull(merchantId)) {
            return;
        }
        this.reloadInfo(merchantId);
    }

    private MerchantInfoCacheDTO reloadInfo(Long merchantId) {
        MerchantInfoCacheDTO info = merchantInfoMapper.getMerchantCacheInfo(merchantId);
        if (Objects.isNull(info)) {
            return null;
        }
        redisHashMapAdapter.put(cacheKey(), merchantId.toString(), info);
        return info;
    }

    private KeyGenerator cacheKey() {
        return RedisConfig.MERCHANT_INFO_CACHE.copy();
    }
}
