package com.bxm.localnews.merchants.cache.impl;

import com.bxm.localnews.merchant.domain.MerchantInfoMapper;
import com.bxm.localnews.merchants.cache.MerchantRedisRefresh;
import com.bxm.localnews.merchants.dto.MerchantUserDTO;
import com.bxm.localnews.merchants.facade.service.MerchantRedisRefreshFacade;
import com.bxm.localnews.merchants.param.MerChantListParam;
import com.bxm.localnews.merchants.service.MerchantInfoDbService;
import com.bxm.localnews.merchants.service.MerchantListService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.redis.RedisHashMapAdapter;
import org.apache.commons.lang3.StringUtils;
import org.springframework.scheduling.annotation.Async;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.Objects;

import static com.bxm.localnews.merchant.common.config.RedisConfig.MERCHANT_USER_CACHE_KEY;
import static com.bxm.localnews.merchants.common.emuns.MerchantStatusEnum.QUALIFICATION_SUCCESS;

/**
 * @author pf.w
 * @date 2020/7/14 16:12
 */
@Service
public class MerchantRedisRefreshImpl implements MerchantRedisRefresh, MerchantRedisRefreshFacade {
    @Resource
    private MerchantListService merchantListService;

    @Resource
    private RedisHashMapAdapter redisHashMapAdapter;

    @Resource
    private MerchantInfoMapper merchantInfoMapper;

    @Resource
    private MerchantInfoDbService merchantInfoDbService;

    @Async
    @Override
    public void refreshMerchantList(Long merchantId) {
        MerchantInfo info = merchantInfoDbService.getMerchantInfo(merchantId);
        if (Objects.isNull(info)) {
            return;
        }

        MerChantListParam param = new MerChantListParam();
        param.setCategoryId(info.getCategoryId());
        param.setAreaCode(info.getAreaCode());
        merchantListService.getFromDb(param);

        //刷新分类商家列表同时刷新总的商家列表
        MerChantListParam paramTotal = new MerChantListParam();
        paramTotal.setCategoryId(0L);
        paramTotal.setAreaCode(info.getAreaCode());
        merchantListService.getFromDb(paramTotal);
    }

    @Override
    public void removeMerchantList(String areaCode, Long categoryId) {
        merchantListService.removeCache(areaCode, categoryId);
    }

    @Override
    public void removeMerchantUserCache(Long userId) {
        if (null == userId) {
            return;
        }

        reloadCache(userId);
    }

    @Override
    public void initCache() {
        List<MerchantInfo> merchantInfos = merchantInfoMapper.queryPassQualificationList();
        merchantInfos.stream().parallel().forEach(merchantInfo -> reloadCache(merchantInfo.getUserId()));
    }

    /**
     * 清除缓存时，重新加载商户对象
     *
     * @param userId 用户ID
     * @return 用户商户对象 如果不存在则返回null
     */
    private MerchantUserDTO reloadCache(Long userId) {
        MerchantInfo merchantInfo = merchantInfoMapper.selectByUserId(userId);

        if (Objects.isNull(merchantInfo)) {
            return null;
        }

        // 审核通过的才可以显示
        int merchantStatus = Objects.equals(merchantInfo.getQualificationStatus(), QUALIFICATION_SUCCESS.getType()) ? 1 : 0;

        String merchantName = merchantInfo.getShortName();

        if (StringUtils.isBlank(merchantName)) {
            merchantName = merchantInfo.getMerchantName();
        }

        MerchantUserDTO merchantUserDTO = MerchantUserDTO.builder()
                .v(MerchantUserDTO.CURRENT_VERSION)
                .userId(userId)
                .merchantId(merchantInfo.getId())
                .isShelf(merchantInfo.getIsShelf())
                .merchantName(merchantName)
                .merchantStatus(merchantStatus)
                .build();

        redisHashMapAdapter.put(MERCHANT_USER_CACHE_KEY, userId.toString(), merchantUserDTO);

        return merchantUserDTO;
    }

    @Override
    public MerchantUserDTO getMerchantUserCache(Long userId) {
        MerchantUserDTO merchantUser = redisHashMapAdapter.get(MERCHANT_USER_CACHE_KEY, userId.toString(), MerchantUserDTO.class);

        // 大部分用户均不是商户，返回空对象
        if (merchantUser == null) {
            merchantUser = new MerchantUserDTO();
            merchantUser.setUserId(userId);
            merchantUser.setMerchantStatus(0);
        }

        // 信息实体变更，重新加载一次
        if (!Objects.equals(MerchantUserDTO.CURRENT_VERSION, merchantUser.getV())) {
            merchantUser = reloadCache(userId);
        }

        return merchantUser;
    }
}
