package com.bxm.localnews.merchants.manage.impl;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.common.param.PointReportParam;
import com.bxm.localnews.merchant.common.enums.MerchantAccountTypeEnum;
import com.bxm.localnews.merchant.common.utils.ImportUtils;
import com.bxm.localnews.merchant.config.MerchantProperties;
import com.bxm.localnews.merchant.domain.*;
import com.bxm.localnews.merchant.dto.ManageMerchantInfoDto;
import com.bxm.localnews.merchant.dto.MechantTeamInfosDTO;
import com.bxm.localnews.merchant.dto.MerchantAuthInfoDto;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.account.MerchantAccountInfoDTO;
import com.bxm.localnews.merchant.integration.BizLogIntegrationService;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.*;
import com.bxm.localnews.merchant.param.manager.MemberTeamParam;
import com.bxm.localnews.merchant.service.account.UserProAccountStrategy;
import com.bxm.localnews.merchant.service.goods.GoodsService;
import com.bxm.localnews.merchant.service.members.MerchantUserService;
import com.bxm.localnews.merchant.service.promote.MerchantUserProAccountService;
import com.bxm.localnews.merchant.vo.MerchantAuthInfo;
import com.bxm.localnews.merchant.vo.MerchantOperationLog;
import com.bxm.localnews.merchant.vo.goods.GoodsSortVo;
import com.bxm.localnews.merchant.vo.manager.MemberShortInfoVo;
import com.bxm.localnews.merchants.common.emuns.MerchantStatusEnum;
import com.bxm.localnews.merchants.dto.MerchantAdvertInfoDTO;
import com.bxm.localnews.merchants.facade.service.MerchantInfoAdvertFacade;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.facade.service.MerchantRedisRefreshFacade;
import com.bxm.localnews.merchants.manage.MerchantInfoManageService;
import com.bxm.localnews.merchants.param.MerchantManageListParam;
import com.bxm.localnews.merchants.push.MerchantPushService;
import com.bxm.localnews.merchants.service.impl.BaseMerchantInfoService;
import com.bxm.localnews.merchants.vo.MerchantAdvertInfoVo;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.localnews.merchants.vo.MerchantInfoListVo;
import com.bxm.localnews.merchants.vo.MerchantMatchVO;
import com.bxm.localnews.mq.common.param.DingtalkMessage;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import com.bxm.newidea.component.vo.PageWarper;
import com.google.common.collect.Lists;
import lombok.AllArgsConstructor;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.web.multipart.MultipartFile;

import java.math.BigDecimal;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.stream.Collectors;

import static com.alibaba.fastjson.JSON.toJSONString;
import static org.springframework.beans.BeanUtils.copyProperties;

/**
 * @author zhoujincheng
 */
@Service
@AllArgsConstructor
public class MerchantInfoManageServiceImpl extends BaseMerchantInfoService implements MerchantInfoManageService, MerchantInfoAdvertFacade {

    private SequenceCreater sequenceCreater;

    private MerchantInfoMapper merchantInfoMapper;

    private MerchantAuthInfoMapper merchantAuthInfoMapper;

    private MerchantRedisRefreshFacade merchantRedisRefresh;

    private UserProAccountStrategy userProAccountStrategy;

    private BizLogIntegrationService bizLogIntegrationService;

    private MerchantOperationLogMapper merchantOperationLogMapper;

    private MerchantProperties merchantProperties;

    private MessageSender messageSender;

    private MerchantUserProAccountService merchantUserProAccountService;

    private MerchantActivityMapper merchantActivityMapper;

    private MerchantMemberMapper merchantMemberMapper;

    private UserIntegrationService userIntegrationService;

    private MerchantUserService merchantUserService;

    private MerchantGoodsMapper merchantGoodsMapper;

    private GoodsService goodsService;

    private MerchantInfoFacadeService merchantInfoDbService;

    private final MerchantPushService merchantPushService;

    @Override
    public Message getAuthInfo(Long merchantId) {
        MerchantInfo merchantInfo = merchantInfoDbService.getMerchantInfo(merchantId);
        if (Objects.isNull(merchantId)) {
            logger.info("未获取到商户信息:[{}]", merchantId);
            return Message.build(false, "商户信息不存在");
        }

        MerchantAuthInfo merchantAuthInfo = merchantAuthInfoMapper.selectByMerchantId(merchantId);
        MerchantAuthInfoDto merchantAuthInfoDto = new MerchantAuthInfoDto();
        merchantAuthInfoDto.setMerchantName(merchantInfo.getMerchantName());
        merchantAuthInfoDto.setCategoryId(merchantInfo.getCategoryId());

        //用户提交过资质信息
        if (Objects.nonNull(merchantAuthInfo)) {
            copyProperties(merchantAuthInfo, merchantAuthInfoDto);
        }
        return Message.build().addParam("merchantAuthInfoDto", merchantAuthInfoDto);
    }

    @Override
    public Message auditAuthInfo(AuditAuthInfoParam auditAuthInfoParam) {
        MerchantInfo merchantInfo = merchantInfoDbService.getMerchantInfo(auditAuthInfoParam.getMerchantId());
        if (merchantInfo == null) {
            logger.info("获取店铺信息失败:[{}]", toJSONString(auditAuthInfoParam));
            return Message.build(false, "获取店铺信息失败");
        }

        MerchantAuthInfo merchantAuthInfo = new MerchantAuthInfo();
        copyProperties(auditAuthInfoParam, merchantAuthInfo);
        merchantAuthInfo.setModifyTime(new Date());

        //后台运营人员新增的商户 初始化没有认证审核信息 走插入认证审核
        if (Objects.isNull(merchantAuthInfoMapper.selectByMerchantId(auditAuthInfoParam.getMerchantId()))) {
            merchantAuthInfoMapper.insertSelective(merchantAuthInfo);
        } else {
            merchantAuthInfoMapper.updateByPrimaryKeySelective(merchantAuthInfo);
        }

        List<MerchantOperationLog> merchantOperationLogs = merchantOperationLogMapper.selectByMerchantId(auditAuthInfoParam.getMerchantId(),
                MerchantStatusEnum.OPERATOR_SHELF_SUCCESS.getType());

        //审核通过  店铺上架  存缓存 推送消息 qualification
        if (Objects.equals(MerchantStatusEnum.QUALIFICATION_SUCCESS.getType(), auditAuthInfoParam.getQualificationStatus())) {
            afterLogicForQualificationSuccess(merchantInfo, merchantOperationLogs, auditAuthInfoParam);
        }
        //审核拒绝 推送消息
        if (Objects.equals(MerchantStatusEnum.QUALIFICATION_FAIL.getType(), auditAuthInfoParam.getQualificationStatus())) {
            afterLogicForQualificationFalse(merchantInfo, merchantOperationLogs, auditAuthInfoParam);
        }

        merchantInfoDbService.updateMerchant(merchantInfo);
        return Message.build(Boolean.TRUE);
    }


    private void afterLogicForQualificationSuccess(MerchantInfo merchantInfo, List<MerchantOperationLog> merchantOperationLogs,
                                                   AuditAuthInfoParam auditAuthInfoParam) {
        //设置主表冗余字段状态
        merchantInfo.setQualificationStatus(MerchantStatusEnum.QUALIFICATION_SUCCESS.getType());
        //查询日志判断是否上架过  店铺已上架后下架   恢复上架
        if (CollectionUtils.isNotEmpty(merchantOperationLogs)) {
            merchantPushService.pushShelfStatusMsg(merchantInfo, MerchantStatusEnum.SHELF_SUCCESS, null);
        }
        //2第一次上架  装修店铺
        if (Objects.equals(merchantInfo.getIsShelf(), MerchantStatusEnum.SHELF_NOTAUDIT.getType())) {
            //认证成功 推广金额增加
            MerchantAccountInfoDTO merchantAccountInfoDTO = new MerchantAccountInfoDTO();
            merchantAccountInfoDTO.setType(MerchantAccountTypeEnum.QUALIFICATION_SUCCESS);
            merchantAccountInfoDTO.setMerchantId(auditAuthInfoParam.getMerchantId());
            merchantAccountInfoDTO.setUserId(merchantInfo.getUserId());
            userProAccountStrategy.exection(merchantAccountInfoDTO);

            //推送
            merchantPushService.pushQualificationStatusMsg(merchantInfo,
                    MerchantStatusEnum.QUALIFICATION_SUCCESS,
                    auditAuthInfoParam.getRemark());

        }
        //审核通过 店铺上架
        merchantInfo.setIsShelf(MerchantStatusEnum.SHELF_SUCCESS.getType());
        //埋点
        pointAdd(merchantInfo);
        //上架刷新缓存
        refreshCache(merchantInfo);
        //记录资质认证成功日志
        addLog(auditAuthInfoParam.getMerchantId(), MerchantStatusEnum.OPERATOR_QUALIFICATION_SUCCESS.getType(), auditAuthInfoParam.getAuditUserId());
        //记录店铺上架成功日志
        addLog(auditAuthInfoParam.getMerchantId(), MerchantStatusEnum.OPERATOR_SHELF_SUCCESS.getType(), auditAuthInfoParam.getAuditUserId());
    }

    private void afterLogicForQualificationFalse(MerchantInfo merchantInfo, List<MerchantOperationLog> merchantOperationLogs,
                                                 AuditAuthInfoParam auditAuthInfoParam) {
        //设置主表冗余字段状态
        merchantInfo.setQualificationStatus(MerchantStatusEnum.QUALIFICATION_FAIL.getType());
        //店铺下架
        // 1.判断是第一次审核失败的下架  只推送审核失败消息
        if (Objects.equals(merchantInfo.getIsShelf(), MerchantStatusEnum.SHELF_NOTAUDIT.getType())) {
            merchantPushService.pushQualificationStatusMsg(merchantInfo, MerchantStatusEnum.QUALIFICATION_FAIL, auditAuthInfoParam.getRemark());
        }
        //2.上架之后再次审核失败的下架   店铺处于上架状态  把下架消息放入缓存和推送下架消息
        if (CollectionUtils.isNotEmpty(merchantOperationLogs)) {
            //下架所有已上架的商品
            goodsService.shelfDownAllGoods(merchantInfo.getId());

            merchantInfo.setIsShelf(MerchantStatusEnum.SHELF_FAIL.getType());
            merchantPushService.pushShelfStatusMsg(merchantInfo, MerchantStatusEnum.SHELF_FAIL, auditAuthInfoParam.getRemark());

            //刷新缓存
            refreshCache(merchantInfo);

            //记录店铺下架日志
            addLog(auditAuthInfoParam.getMerchantId(), MerchantStatusEnum.OPERATOR_SHELF_FAIL.getType(), auditAuthInfoParam.getAuditUserId());
        }
        //记录日志
        addLog(auditAuthInfoParam.getMerchantId(), MerchantStatusEnum.OPERATOR_QUALIFICATION_FAIL.getType(), auditAuthInfoParam.getAuditUserId());
    }

    @Override
    public void addMerchantRemark(Long merchantId, String remark) {
        MerchantInfo merchantInfo = new MerchantInfo();
        merchantInfo.setId(merchantId);
        merchantInfo.setRemark(remark);
        boolean flag = merchantInfoDbService.updateMerchant(merchantInfo);
        if (!flag) {
            logger.info("添加备注操作失败:[{}]", JSONObject.toJSONString(merchantInfo));
        }
    }

    @Override
    public Message getMerchantInfo(Long merchantId) {
        MerchantInfo merchantInfo = merchantInfoDbService.getMerchantInfo(merchantId);

        if (merchantInfo == null) {
            logger.info("查询商户信息失败:[{}]", merchantId);
            return Message.build(false, "商户信息不存在");
        }

        ManageMerchantInfoDto manageMerchantInfoDto = new ManageMerchantInfoDto();
        copyProperties(merchantInfo, manageMerchantInfoDto);
        return Message.build().addParam("manageMerchantInfoDto", manageMerchantInfoDto);
    }

    @Override
    public Message saveMerchantInfo(ManageMerchantInfoParam manageMerchantInfoParam) {
        MerchantInfo merchant = merchantInfoDbService.getMerchantInfo(manageMerchantInfoParam.getMerchantId());
        if (merchant == null) {
            logger.info("查询商户信息失败:[{}]", manageMerchantInfoParam.getMerchantId());
            return Message.build(false, "商户信息不存在");
        }
        MerchantInfo merchantInfo = this.fillMerchantInfo(manageMerchantInfoParam);

        // 如果用户的定位发生了变更，则删除缓存
        if (!StringUtils.equals(merchantInfo.getAreaCode(), merchant.getAreaCode())) {
            merchantRedisRefresh.removeMerchantList(merchant.getAreaCode(), merchant.getCategoryId());
        }

        //只修改店铺信息
        if (Objects.equals(manageMerchantInfoParam.getIsShelf(), merchant.getIsShelf())) {
            merchantInfoDbService.updateMerchant(merchantInfo);
            merchantRedisRefresh.removeMerchantUserCache(merchant.getUserId());
            refreshCache(merchantInfo);
            return Message.build(true, "店铺信息修改成功");
        }
        //修改了上下架状态
        //判断用户资质审核未通过
        if (!Objects.equals(merchant.getQualificationStatus(), MerchantStatusEnum.QUALIFICATION_SUCCESS.getType())) {
            logger.info("资质认证未通过,不能操作上下架:[{}]", JSONObject.toJSONString(manageMerchantInfoParam));
            //不能修改店铺上下架状态
            merchantInfo.setIsShelf(merchant.getIsShelf());
            merchantInfoDbService.updateMerchant(merchantInfo);
            refreshCache(merchantInfo);
            return Message.build(true, "店铺信息修改成功,但资质认证未通过,不能操作上下架");
        }

        merchantInfoDbService.updateMerchant(merchantInfo);
        //上架
        if (Objects.equals(MerchantStatusEnum.SHELF_SUCCESS.getType(), manageMerchantInfoParam.getIsShelf())) {
            //推送消息
            merchantPushService.pushShelfStatusMsg(merchantInfo, MerchantStatusEnum.SHELF_SUCCESS, null);
            refreshCache(merchantInfo);
            addLog(manageMerchantInfoParam.getMerchantId(), MerchantStatusEnum.OPERATOR_SHELF_SUCCESS.getType(), manageMerchantInfoParam.getAuditUserId());
        }
        //下架
        if (Objects.equals(MerchantStatusEnum.SHELF_FAIL.getType(), manageMerchantInfoParam.getIsShelf())) {
            //下架所有已上架的商品
            goodsService.shelfDownAllGoods(merchant.getId());
            //推送消息
            merchantPushService.pushShelfStatusMsg(merchantInfo, MerchantStatusEnum.SHELF_FAIL, manageMerchantInfoParam.getRemovalReason());
            //刷新缓存
            refreshCache(merchantInfo);
            addLog(manageMerchantInfoParam.getMerchantId(), MerchantStatusEnum.OPERATOR_SHELF_FAIL.getType(), manageMerchantInfoParam.getAuditUserId());
        }
        return Message.build(true, "店铺信息修改成功");
    }

    /**
     * 记录日志
     *
     * @param merchantId:商户id type:日志类型  currentUserId:审核人
     */
    private void addLog(Long merchantId, Integer type, Long currentUserId) {
        MerchantOperationLog merchantOperationLog = new MerchantOperationLog();
        merchantOperationLog.setUserId(currentUserId);
        merchantOperationLog.setMerchantId(merchantId);
        merchantOperationLog.setCreateTime(new Date());
        merchantOperationLog.setType(type);
        merchantOperationLogMapper.insertSelective(merchantOperationLog);
    }

    /**
     * 资质审核成功的埋点
     *
     * @param merchantInfo 商户信息
     */
    private void pointAdd(MerchantInfo merchantInfo) {
        PointReportParam reportParam = PointReportParam.build()
                .e("3034")
                .ev("121." + merchantInfo.getId())
                .put("uid", merchantInfo.getUserId() + "");
        bizLogIntegrationService.point(reportParam);
        if (logger.isDebugEnabled()) {
            logger.debug("商家通过资质认证事件，请求参数: {}", reportParam);
        }
    }

    /**
     * 刷新缓存
     *
     * @param merchant 商户信息
     */
    private void refreshCache(MerchantInfo merchant) {
        merchantRedisRefresh.removeMerchantUserCache(merchant.getUserId());
        merchantRedisRefresh.refreshMerchantList(merchant.getId());
    }

    @Override
    public PageWarper<MerchantInfoListVo> getMerchantList(MerchantManageListParam merchantManageListParam) {
        List<MerchantInfoListVo> merchantInfoList = merchantInfoMapper.queryMerchantListByPage(merchantManageListParam);
        if (CollectionUtils.isNotEmpty(merchantInfoList)) {
            //根据商户id查询资质
            List<Long> merchantIds = merchantInfoList.stream().map(MerchantInfoListVo::getId).collect(Collectors.toList());
            List<MerchantAuthInfo> merchantAuthInfos = merchantAuthInfoMapper.selectByMerchantIds(merchantIds);
            merchantInfoList.forEach(merchant -> {
                if (merchant.getAblePromotionCash() != null) {
                    merchant.setAblePromotionCash(merchant.getAblePromotionCash().setScale(2, BigDecimal.ROUND_DOWN));
                }
                Optional<MerchantAuthInfo> match = merchantAuthInfos.stream().filter(authInfo ->
                        Objects.equals(authInfo.getMerchantId(), merchant.getId())).findFirst();

                match.ifPresent(merchantAuthInfo -> merchant.setBusinessLicenseType(merchantAuthInfo.getType()));
                //商家广告上下架状态
                List<MerchantAdvertInfoVo> infos = merchantActivityMapper.selectByMerchantId(merchant.getId());
                merchant.setMerchantAdvertStatus(CollectionUtils.isEmpty(infos) ? 0 : infos.get(0).getStatus() == 0 ? 2 : 1);
                //团队成员数量
                merchant.setMerchantTeamNum(merchantMemberMapper.memberNum(merchant.getId()));
                //商品数
                GoodsSortVo goodsSortVo = merchantGoodsMapper.selectTotalByMerchantId(merchant.getId());
                merchant.setGoodsNum(Objects.nonNull(goodsSortVo) ? goodsSortVo.getTotal() : null);
            });
            return new PageWarper<>(merchantInfoList);
        }
        return new PageWarper<>(Lists.newArrayList());
    }

    @Override
    public Message insertMerchantInfo(ManageMerchantInfoAddParam manageMerchantInfoAddParam) {
        if (CollectionUtils.isNotEmpty(merchantInfoMapper.selectByMobile(manageMerchantInfoAddParam.getMobile()))) {
            return Message.build(false, "该用户已绑定该手机号,不能重复绑定");
        }
        //后台运营人员新增的商户需要手动绑定用户id
        //判断用户是否绑定过商户
        if (merchantInfoMapper.selectByUserId(manageMerchantInfoAddParam.getUserId()) != null) {
            return Message.build(false, "该用户已绑定商户,不能重复绑定");
        }
        MerchantInfo merchantInfo = new MerchantInfo();
        copyProperties(manageMerchantInfoAddParam, merchantInfo);
        merchantInfo.setId(sequenceCreater.nextLongId());
        merchantInfo.setCreateTime(new Date());

        if (merchantInfoDbService.saveMerchant(merchantInfo)) {
            //后续动作
            afterEnterSuccess(merchantInfo, manageMerchantInfoAddParam);
        } else {
            logger.info("新增商家信息失败:[{}]", JSONObject.toJSONString(manageMerchantInfoAddParam));
        }
        return Message.build(true, "新增店铺成功");
    }

    /**
     * 商户入驻填写成功后
     *
     * @param merchantInfo 商户信息
     */
    private void afterEnterSuccess(MerchantInfo merchantInfo, ManageMerchantInfoAddParam manageMerchantInfoAddParam) {
        if (merchantProperties.getEnableEnterNotify()) {
            messageSender.sendDingtalk(DingtalkMessage.builder()
                    .scene("mct")
                    .content("有新的商户申请入驻：" + merchantInfo.getMerchantName())
                    .build());
        }
        //初始化账户信息
        merchantUserProAccountService.initProAccount(manageMerchantInfoAddParam.getUserId());
        //初始化团队信息
        merchantUserService.bossCreate(merchantInfo.getId(), merchantInfo.getUserId());
    }

    @Override
    public MerchantAdvertInfoDTO merchantAdvertInfo(Long merchantId) {
        List<MerchantAdvertInfoDTO> result;
        List<MerchantAdvertInfoVo> infos = merchantActivityMapper.selectByMerchantId(merchantId);
        if (!CollectionUtils.isEmpty(infos)) {
            result = infos.stream().map(item -> {
                MerchantAdvertInfoDTO merchantAdvertInfoDTO = new MerchantAdvertInfoDTO();
                BeanUtils.copyProperties(item, merchantAdvertInfoDTO);
                return merchantAdvertInfoDTO;
            }).collect(Collectors.toList());
            return result.get(0);
        } else {
            return null;
        }
    }

    @Override
    public Boolean editMerchantAdvert(MerchantAdvertParam param) {
        MerchantAdvertInfoVo entity = new MerchantAdvertInfoVo();
        BeanUtils.copyProperties(param, entity);

        if (Objects.isNull(param.getId())) {
            merchantActivityMapper.save(entity);
        } else {
            merchantActivityMapper.updateMerchantAdvert(entity);
        }
        return Boolean.TRUE;
    }

    @Override
    public PageWarper<MechantTeamInfosDTO> getMemberInfo(MemberTeamParam param) {
        List<MemberShortInfoVo> listFromDb = merchantMemberMapper.getMemberForIdAndNickname(param);
        if (CollectionUtils.isEmpty(listFromDb)) {
            return new PageWarper<>();
        }

        List<MechantTeamInfosDTO> result = listFromDb.stream().map(item -> {
            MechantTeamInfosDTO mechantTeamInfosDTO = new MechantTeamInfosDTO();
            BeanUtils.copyProperties(item, mechantTeamInfosDTO);
            return mechantTeamInfosDTO;
        }).collect(Collectors.toList());

        result.forEach(bean -> {
            UserInfoDTO user = userIntegrationService.getUserFromRedisDB(bean.getUserId());
            if (Objects.nonNull(user)) {
                bean.setPhone(Objects.nonNull(user.getPhone()) ? user.getPhone() : null);
                bean.setAreaName(Objects.nonNull(user.getLocationName()) ? user.getLocationName() : null);
            }
        });
        return new PageWarper<>(result);
    }

    @Override
    public void readExcel(MultipartFile multipartFiles) {
        if (null == multipartFiles) {
            return;
        }
        Long merchantId = null;
        List<List<String>> resultList = ImportUtils.readExcelFile(multipartFiles);

        if (logger.isDebugEnabled()) {
            logger.debug("解析结果：" + JSON.toJSONString(resultList));
        }

        if (resultList == null) {
            return;
        }

        for (List<String> result : resultList) {
            logger.debug("解析结果：" + JSON.toJSONString(result));
            if (result.size() >= 2) {
                String bossPhone = result.get(1);
                merchantId = Long.valueOf(bossPhone);
            }
        }
        //批量绑定员工
        Long employeeId;
        CreateParam createParam;

        for (List<String> result : resultList) {
            if (StringUtils.isBlank(result.get(0))) {
                continue;
            }
            employeeId = Long.valueOf(result.get(0));
            createParam = new CreateParam();
            //不推送给老板
            createParam.setPush(0);
            createParam.setMerchantId(merchantId);
            createParam.setUserId(employeeId);
            merchantUserService.create(createParam);
        }
    }

    @Override
    public List<MerchantMatchVO> matchMerchantInfoByName(String merchantName) {
        return merchantInfoMapper.matchMerchantInfoByName(merchantName);
    }
}










