package com.bxm.localnews.merchants.service.impl;

import com.bxm.localnews.merchant.domain.MerchantCollectMapper;
import com.bxm.localnews.merchants.cache.collected.MerchantCollectRedisRefresh;
import com.bxm.localnews.merchants.dto.MerchantCollectInfoDetailDTO;
import com.bxm.localnews.merchants.param.CollectMerchantParam;
import com.bxm.localnews.merchants.param.MerchantCollectListParam;
import com.bxm.localnews.merchants.service.MerchantCollectService;
import com.bxm.localnews.merchants.service.MerchantInfoDbService;
import com.bxm.localnews.merchants.vo.MerchantCollectedVo;
import com.bxm.localnews.merchants.vo.MerchantFavoriteVo;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.vo.PageWarper;
import com.github.pagehelper.Page;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang.StringUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;

/**
 * @author: pf.w
 * @date: 2020/7/14 15:09
 */
@Service
@Log4j2
@AllArgsConstructor
public class MerchantCollectServiceImpl implements MerchantCollectService {

    private final MerchantCollectMapper merchantCollectMapper;

    private final MerchantCollectRedisRefresh merchantCollectRedisRefresh;

    private final MerchantInfoDbService merchantInfoDbService;

    @Override
    public boolean collectMerchant(CollectMerchantParam param) {
        if (log.isDebugEnabled()) {
            log.debug("userId : {},merchantId : {},type : {}", param.getUserId(), param.getMerchantId(),
                    param.getType());
        }

        if (Objects.isNull(param.getMerchantId()) || Objects.isNull(param.getUserId()) ||
                Objects.isNull(param.getType())) {
            return false;
        }

        //查看记录是否存在
        MerchantFavoriteVo merchantFavoriteVo = merchantCollectMapper.getFavoriteInfo(param.getMerchantId(),
                param.getUserId());

        if (Objects.isNull(merchantFavoriteVo)) {
            merchantCollectMapper.save(param.getMerchantId(), param.getUserId(), param.getType());
        } else {
            merchantCollectMapper.updateCollectInfo(param.getMerchantId(), param.getUserId(), param.getType());
        }
        //异步刷新
        merchantCollectRedisRefresh.refreshMerchantCollect(param.getUserId());
        return true;
    }

    @Override
    public PageWarper<MerchantCollectInfoDetailDTO> getCollectList(MerchantCollectListParam param) {
        if (log.isDebugEnabled()) {
            log.debug("param : {}", param);
        }

        List<MerchantCollectedVo> list = merchantCollectRedisRefresh.getCollectedMerchants(param.getUserId());
        if (CollectionUtils.isEmpty(list)) {
            return new PageWarper<>();
        }

        //分页
        int start = (param.getPageNum() - 1) * param.getPageSize();
        int end = param.getPageNum() * param.getPageSize();
        if (start < 0) {
            start = 0;
        }
        if (end > list.size()) {
            end = list.size();
        }
        List<MerchantCollectedVo> merchantIds = list.subList(start, end);
        //转化对象
        List<MerchantCollectInfoDetailDTO> result = covert(merchantIds);
        Page page = new Page(param.getPageNum(), param.getPageSize());
        page.setTotal(list.size());
        page.addAll(result);
        return new PageWarper<>(page);
    }

    private List<MerchantCollectInfoDetailDTO> covert(List<MerchantCollectedVo> merchantIds) {
        List<MerchantCollectInfoDetailDTO> result = new ArrayList<>();

        merchantIds.forEach(collectedVo -> {
            MerchantCollectInfoDetailDTO merchantCollectInfoDetailDTO = new MerchantCollectInfoDetailDTO();
            MerchantInfo info = merchantInfoDbService.getMerchantInfo(collectedVo.getMerchantId());
            if (Objects.nonNull(info)) {
                merchantCollectInfoDetailDTO.setMerchantId(collectedVo.getMerchantId());
                merchantCollectInfoDetailDTO.setMerchantName(info.getMerchantName());
                merchantCollectInfoDetailDTO.setShortDesc(info.getDes());
                merchantCollectInfoDetailDTO.setStatus(info.getIsShelf() == 1 ? 1 : 0);
                merchantCollectInfoDetailDTO.setImgUrl(StringUtils.isNotBlank(info.getHeadPics()) ? Arrays.asList
                        (info.getHeadPics().split(",")).get(0) : "");
                merchantCollectInfoDetailDTO.setCollectTime(collectedVo.getCollectedTime() != null ? DateUtils.timeConvertString
                        (collectedVo.getCollectedTime()) : null);
                result.add(merchantCollectInfoDetailDTO);
            }
        });

        return result;
    }

}
