package com.bxm.localnews.merchants.service.impl;

import com.bxm.localnews.merchant.domain.MerchantInfoMapper;
import com.bxm.localnews.merchants.cache.detail.MerchantInfoRedisRefresh;
import com.bxm.localnews.merchants.dto.MerchantInfoCacheDTO;
import com.bxm.localnews.merchants.dto.MerchantInfoDto;
import com.bxm.localnews.merchants.facade.service.MerchantInfoFacadeService;
import com.bxm.localnews.merchants.param.DeletedPhoneParam;
import com.bxm.localnews.merchants.service.MerchantInfoDbService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.google.common.base.Splitter;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;

/**
 * 商家信息入库更新动作封装
 *
 * @author pf.w
 * @date 2020/9/25 10:38
 **/
@Service
@Log4j2
@AllArgsConstructor
public class MerchantInfoDbServiceImpl implements MerchantInfoDbService, MerchantInfoFacadeService {

    private final MerchantInfoMapper merchantInfoMapper;

    private final MerchantInfoRedisRefresh merchantInfoRedisRefresh;

    @Override
    public boolean saveMerchant(MerchantInfo info) {
        boolean flag = merchantInfoMapper.insert(info) > 0;
        if (flag) {
            //异步刷新商家信息缓存
            merchantInfoRedisRefresh.merchantInfoRefresh(info.getId());
        }
        return flag;
    }

    @Override
    public boolean updateMerchant(MerchantInfo info) {
        boolean flag = merchantInfoMapper.updateByPrimaryKeySelective(info) > 0;
        if (flag) {
            //异步刷新商家信息缓存
            merchantInfoRedisRefresh.merchantInfoRefresh(info.getId());
        }
        return flag;
    }

    @Override
    public boolean deleteMerchantInfoByUid(Long userId) {
        DeletedPhoneParam param = new DeletedPhoneParam();
        param.setUserId(userId);

        Long id = merchantInfoMapper.deletePhoneByUserId(param);
        if (id > 0) {
            Long merchantId = param.getId();
            //异步刷新商家信息缓存
            merchantInfoRedisRefresh.merchantInfoRefresh(merchantId);
        }
        return true;
    }

    @Override
    public MerchantInfo getMerchantInfo(Long merchantId) {
        return merchantInfoRedisRefresh.getMerchantInfo(merchantId);
    }

    @Override
    public MerchantInfoDto getMerchantInfoDetail(Long merchantId) {
        MerchantInfoCacheDTO merchantInfo = merchantInfoRedisRefresh.getMerchantInfoCache(merchantId);

        if (merchantInfo == null) {
            return null;
        }

        MerchantInfoDto merchantInfoDto = new MerchantInfoDto();
        BeanUtils.copyProperties(merchantInfo, merchantInfoDto);

        if (StringUtils.isNotBlank(merchantInfo.getLogo())) {
            merchantInfoDto.setMasterPicture(merchantInfo.getLogo());
        } else if (StringUtils.isNotBlank(merchantInfo.getHeadPics())) {
            merchantInfoDto.setMasterPicture(Splitter.on(",").split(merchantInfo.getHeadPics()).iterator().next());
        }

        return merchantInfoDto;
    }
}
