package com.bxm.localnews.merchants.service.impl;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.base.integration.DomainIntegrationService;
import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.common.constant.MerchantMemberEnum;
import com.bxm.localnews.merchant.config.MerchantProperties;
import com.bxm.localnews.merchant.domain.MerchantAuthInfoMapper;
import com.bxm.localnews.merchant.domain.MerchantInfoMapper;
import com.bxm.localnews.merchant.domain.MerchantPrerogativeInfoMapper;
import com.bxm.localnews.merchant.dto.LocationDTO;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.integration.LocationIntegrationService;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.service.members.MerchantUserService;
import com.bxm.localnews.merchant.service.promote.MerchantUserProAccountService;
import com.bxm.localnews.merchant.vo.MerchantAuthInfo;
import com.bxm.localnews.merchant.vo.MerchantMemberVo;
import com.bxm.localnews.merchant.vo.MerchantUserAccountVO;
import com.bxm.localnews.merchants.cache.MerchantRedisRefresh;
import com.bxm.localnews.merchants.common.emuns.MerchantStatusEnum;
import com.bxm.localnews.merchants.dto.IntoApplyDTO;
import com.bxm.localnews.merchants.dto.MerchantBusinessInfo;
import com.bxm.localnews.merchants.dto.MerchantInfoDto;
import com.bxm.localnews.merchants.dto.MerchantStatusDto;
import com.bxm.localnews.merchants.param.*;
import com.bxm.localnews.merchants.service.MerchantInfoDbService;
import com.bxm.localnews.merchants.service.MerchantInfoService;
import com.bxm.localnews.merchants.vo.MerchantInfo;
import com.bxm.localnews.merchants.vo.MerchantInfoSimpleVO;
import com.bxm.localnews.mq.common.param.DingtalkMessage;
import com.bxm.localnews.msg.sender.MessageSender;
import com.bxm.newidea.component.redis.RedisStringAdapter;
import com.bxm.newidea.component.uuid.SequenceCreater;
import com.bxm.newidea.component.vo.Message;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.math.BigDecimal;
import java.util.Date;
import java.util.Objects;

import static java.util.Objects.isNull;
import static java.util.Objects.nonNull;
import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang3.StringUtils.join;

@Slf4j
@Service
@AllArgsConstructor
public class MerchantInfoServiceImpl extends BaseMerchantInfoService implements MerchantInfoService {

    private MerchantInfoMapper merchantInfoMapper;

    private MerchantAuthInfoMapper merchantAuthInfoMapper;

    private RedisStringAdapter redisStringAdapter;

    private MerchantProperties merchantProperties;

    private SequenceCreater sequenceCreater;

    private MessageSender messageSender;

    private MerchantUserProAccountService merchantUserProAccountService;

    private MerchantPrerogativeInfoMapper merchantPrerogativeInfoMapper;

    private UserIntegrationService userIntegrationService;

    private MerchantUserService merchantUserService;

    private DomainIntegrationService domainIntegrationService;

    private MerchantRedisRefresh merchantRedisRefresh;

    private final LocationIntegrationService locationIntegrationService;

    private final MerchantInfoDbService merchantInfoDbService;

    @Override
    public Message getMerchantInfoDetail(Long merchantId) {
        MerchantInfoDto merchantInfoDto = merchantInfoDbService.getMerchantInfoDetail(merchantId);
        if (merchantInfoDto == null) {
            logger.info("商户id获取不到商户信息:[{}]", merchantId);
            return Message.build(false, "商户信息不存在");
        }

        return Message.build(true).addParam("merchantInfoDto", merchantInfoDto);
    }

    @Override
    public Message enter(MerchantIntoApplyParam merchantIntoApplyParam) {
        if (!CollectionUtils.isEmpty(merchantInfoMapper.selectByMobile(merchantIntoApplyParam.getMobile()))) {
            return Message.build(false, "该用户已绑定该手机号,不能重复绑定");
        }
        if (merchantInfoMapper.selectByUserId(merchantIntoApplyParam.getUserId()) != null) {
            return Message.build(false, "该用户已绑定商户,不能重复绑定");
        }
        MerchantInfo merchantInfo = super.fillMerchantInfo(merchantIntoApplyParam);
        BeanUtils.copyProperties(merchantIntoApplyParam, merchantInfo);
        merchantInfo.setId(sequenceCreater.nextLongId());
        merchantInfo.setCreateTime(new Date());

        if (!merchantInfoDbService.saveMerchant(merchantInfo)) {
            logger.info("提交入驻失败:{}", merchantIntoApplyParam);
            return Message.build(false, "提交入驻失败");
        }
        //入驻成功后续操作
        afterEnterSuccess(merchantInfo, merchantIntoApplyParam);

        IntoApplyDTO intoApplyDTO = new IntoApplyDTO();
        intoApplyDTO.setMerchantId(merchantInfo.getId());
        return Message.build(true).addParam("intoApplyDTO", intoApplyDTO);
    }

    /**
     * 商户入驻填写成功后
     *
     * @param merchantInfo 商户信息
     */
    private void afterEnterSuccess(MerchantInfo merchantInfo, MerchantIntoApplyParam merchantIntoApplyParam) {
        //钉钉推送新上架入驻信息
        if (merchantProperties.getEnableEnterNotify()) {
            messageSender.sendDingtalk(DingtalkMessage.builder()
                    .scene("mct")
                    .content("有新的商户申请入驻：" + merchantInfo.getMerchantName())
                    .build());
        }

        // 入驻成功后，使用配置的账号ID关注商家，方便运营、产品跟进商家
        if (merchantProperties.getFollowUserIds().size() > 0 && merchantInfo.getUserId() != null) {
            merchantProperties.getFollowUserIds().forEach(fromUserId ->
                    userIntegrationService.follow(fromUserId, merchantInfo.getUserId())
            );
        }

        //初始化账户信息
        merchantUserProAccountService.initProAccount(merchantIntoApplyParam.getUserId());
        //商家老板初始化团队信息
        merchantUserService.bossCreate(merchantInfo.getId(), merchantInfo.getUserId());
    }

    @Override
    public Message updateMerchantInfo(MerchantInfoParam merchantInfoParam) {
        MerchantInfo merchantInfo = fillMerchantInfo(merchantInfoParam);

        merchantInfo.setSettlementStatus(MerchantStatusEnum.SETTLEMENT_ALREADY.getType());
        merchantInfo.setModifyTime(new Date());
        MerchantInfo fullMerchantInfo = merchantInfoDbService.getMerchantInfo(merchantInfoParam.getMerchantId());

        // 如果定位发生了变化，则删除原有的缓存
        if (!StringUtils.equals(merchantInfo.getAreaCode(), fullMerchantInfo.getAreaCode())) {
            merchantRedisRefresh.removeMerchantList(fullMerchantInfo.getAreaCode(), fullMerchantInfo.getCategoryId());
        }

        boolean flag = merchantInfoDbService.updateMerchant(merchantInfo);
        if (!flag) {
            logger.info("更新店铺信息失败:{}", merchantInfoParam);
            return Message.build(false, "更新店铺信息失败");
        }

        // 更新商户信息时清除缓存
        merchantRedisRefresh.removeMerchantUserCache(fullMerchantInfo.getUserId());
        merchantRedisRefresh.refreshMerchantList(fullMerchantInfo.getId());

        return Message.build(true);
    }


    @Override
    public Message authApply(MerchantAuthInfoParam merchantAuthInfoParam) {
        //判断商户id合法性
        if (Objects.isNull(merchantInfoDbService.getMerchantInfo(merchantAuthInfoParam.getMerchantId()))) {
            logger.info("商户id获取不到商户信息:{}", merchantAuthInfoParam);
            return Message.build(false, "商户id获取不到商户信息");
        }

        MerchantAuthInfo merchantAuthInfo = new MerchantAuthInfo();
        BeanUtils.copyProperties(merchantAuthInfoParam, merchantAuthInfo);
        merchantAuthInfo.setCreateTime(new Date());
        merchantAuthInfo.setQualificationStatus(MerchantStatusEnum.QUALIFICATION_WILL.getType());

        if (merchantAuthInfoMapper.insertSelective(merchantAuthInfo) <= 0) {
            logger.info("提交资质审核资料时失败:{}", merchantAuthInfoParam);
            return Message.build(false, "提交资质审核资料时失败");
        }

        //更新商户信息表冗余最近提交资质审核
        MerchantInfo merchantInfo = new MerchantInfo();
        merchantInfo.setId(merchantAuthInfoParam.getMerchantId());
        merchantInfo.setLastAuthTime(new Date());
        merchantInfo.setQualificationStatus(MerchantStatusEnum.QUALIFICATION_WILL.getType());
        merchantInfo.setCategoryId(merchantAuthInfoParam.getCategoryId());

        return Message.build(merchantInfoDbService.updateMerchant(merchantInfo));
    }


    @Override
    public MerchantIntoInfoDto getMerchantInfo(Long merchantId) {
        MerchantInfo merchantInfo = merchantInfoDbService.getMerchantInfo(merchantId);

        if (null != merchantInfo) {
            MerchantIntoInfoDto merchantIntoInfoDto = new MerchantIntoInfoDto();
            BeanUtils.copyProperties(merchantInfo, merchantIntoInfoDto);

            return merchantIntoInfoDto;
        }

        return null;
    }

    @Override
    public Message getBusinessInfo(Long merchantId) {
        MerchantInfo merchantInfo = merchantInfoDbService.getMerchantInfo(merchantId);
        if (isNull(merchantInfo)) {
            logger.info("获取商家中心基本信息失败:[{}]", merchantId);
            return Message.build(false, "获取商家中心基本信息失败");
        }

        MerchantBusinessInfo merchantBusinessInfo = new MerchantBusinessInfo();
        BeanUtils.copyProperties(merchantInfo, merchantBusinessInfo);

        MerchantUserAccountVO merchantUserAccountVO = merchantUserProAccountService.getUserProAccount(merchantInfo.getId());
        if (nonNull(merchantUserAccountVO)) {
            merchantBusinessInfo.setAblePromotionCash(merchantUserAccountVO.getAbleProCash().setScale(2, BigDecimal.ROUND_DOWN));
        }
        //从缓存取
        String string = redisStringAdapter.getString(RedisConfig.MERCHANT_ID_KEY.copy().appendKey(merchantId));
        merchantBusinessInfo.setIsInCenter(isBlank(string) ? MerchantStatusEnum.INCENTER_NOT.getType() :
                MerchantStatusEnum.INCENTER_YES.getType());

        merchantBusinessInfo.setIsVip(isNull(merchantPrerogativeInfoMapper.selectByMerchantId(merchantId, 0))
                ? MerchantStatusEnum.VIP_NO.getType() : MerchantStatusEnum.VIP_YES.getType());

        String msgTop = redisStringAdapter.getString(RedisConfig.MERCHANT_MESSAGE_KEY.copy().appendKey(merchantId));
        merchantBusinessInfo.setBindType(isBlank(msgTop) ? 0 : 1);

        // 商家粉丝数量 也就是商家绑定的用户粉丝数量
        UserInfoDTO userFromRedisDB = userIntegrationService.getUserFromRedisDB(merchantInfo.getUserId());
        if (nonNull(userFromRedisDB) && nonNull(userFromRedisDB.getFanNum())) {
            merchantBusinessInfo.setFansNum(userFromRedisDB.getFanNum());
        }

        return Message.build(true).addParam("merchantBusinessInfo", merchantBusinessInfo);
    }

    @Override
    public MerchantStatusDto getStatus(MerchantStatusParam param) {
        // 产品要求个人主页的入口支持配置 暂时先做个开关 后期可能需要根据用户、区域进行返回
        if (Boolean.FALSE.equals(merchantProperties.getPersonalEnterSwitch())) {
            // 如果开关已关 且不在特殊开启的城市中 则返回空信息 客户端不展示入口
            if (merchantProperties.getOpenCityCodes().stream().noneMatch(p -> Objects.equals(p, param.getAreaCode()))) {
                return null;
            }
        }

        // 获取h5base url
        String innerH5BaseUrl = domainIntegrationService.getInnerH5BaseUrl();

        Long userId = param.getUserId();
        MerchantInfo merchantInfo = merchantInfoMapper.selectByUserId(userId);
        MerchantStatusDto merchantStatusDto = new MerchantStatusDto();

        //获取用户团队信息并且是员工
        MerchantMemberVo merchantMemberVo = merchantUserService.getMemberInfo(userId);
        if (merchantMemberVo != null && Objects.equals(merchantMemberVo.getRole(), MerchantMemberEnum.MEMBER_ROLE_EMPLOYEE.getType())) {
            merchantStatusDto.setContent("我的营销");
            String baseUrl = join(innerH5BaseUrl, merchantProperties.getMarketUrlBase());
            String url = replaceUserIdAndMerchant(baseUrl, merchantMemberVo.getUserId(), merchantMemberVo.getMerchantId());

            if (log.isDebugEnabled()) {
                logger.debug("innerH5BaseUrl : {},baseUrl : {},url : {}", innerH5BaseUrl, baseUrl, url);
            }
            merchantStatusDto.setProtocolAddress(url);
            // 處理佔位符信息
            return fillInfoIfNecessary(merchantStatusDto, param);
        }

        //未申请 资质未提交
        if (merchantInfo == null) {
            merchantStatusDto.setContent("商家合作");
            UserInfoDTO user = userIntegrationService.getUserFromRedisDB(userId);
            merchantStatusDto.setProtocolAddress(join(innerH5BaseUrl, merchantProperties.getMerchantApplyPath()));
            // 替换手机号
            if (Objects.nonNull(user) && isNotBlank(user.getPhone())) {
                String url = replacePhone(join(innerH5BaseUrl, merchantProperties.getMerchantApplyPath()),
                        user.getPhone());

                merchantStatusDto.setProtocolAddress(fillLocationParam(url, param.getAreaCode()));
            }
            // 處理佔位符信息
            return fillInfoIfNecessary(merchantStatusDto, param);
        }

        //已申请 完成资质
        if (merchantInfo.getQualificationStatus().equals(MerchantStatusEnum.QUALIFICATION_SUCCESS.getType())) {
            merchantStatusDto.setContent("商家中心");
        } else {
            //已申请 未完成资质、
            merchantStatusDto.setContent("商家合作");
        }

        // 其他状态都跳转到商家中心
        merchantStatusDto.setProtocolAddress(replaceMerchant(join(innerH5BaseUrl, merchantProperties.getMerchantCenterPath()),
                merchantInfo.getId()));
        // 處理佔位符信息
        return fillInfoIfNecessary(merchantStatusDto, param);
    }

    private MerchantStatusDto fillInfoIfNecessary(MerchantStatusDto merchantStatusDto, MerchantStatusParam param) {
        //如果传递了这个参数，占位符由服务端填充,为了兼容H5使用
        if (Objects.equals(Boolean.TRUE, param.getFillPlaceholder())) {
            String resultUrl = merchantStatusDto.getProtocolAddress();
            LocationDTO areaCodeInfo = locationIntegrationService.getLocationByGeocode(param.getAreaCode());

            resultUrl = resultUrl.contains("{areaCode}") ? resultUrl.replace("{areaCode}", Objects.toString(param.getAreaCode())) : resultUrl;
            if (areaCodeInfo != null) {
                resultUrl = resultUrl.contains("{areaName}") ? resultUrl.replace("{areaName}", Objects.toString(areaCodeInfo.getName())) : resultUrl;
            } else {
                resultUrl = resultUrl.contains("{areaName}") ? resultUrl.replace("{areaName}", "") : resultUrl;
            }
            resultUrl = resultUrl.contains("{userId}") ? resultUrl.replace("{userId}", Objects.toString(param.getUserId())) : resultUrl;
            merchantStatusDto.setProtocolAddress(resultUrl);
        }

        return merchantStatusDto;
    }

    private String fillLocationParam(String url, String areaCode) {

        try {
            LocationDTO locationByGeocode = locationIntegrationService.getLocationByGeocode(areaCode);
            if (log.isDebugEnabled()) {
                log.debug("根据code: {} 获取到的城市信息: {}", areaCode, JSON.toJSONString(locationByGeocode));
            }

            boolean flag = (isNull(locationByGeocode) || (isBlank(locationByGeocode.getCode()) && isBlank(locationByGeocode.getLastCode())));
            if (flag) {
                log.warn("根据code: {} 无法获取到自动填充的城市", areaCode);
                return url;
            }

            String fullCode = isNotBlank(locationByGeocode.getLastCode()) ? locationByGeocode.getLastCode() : locationByGeocode.getCode();

            // 省
            String provinceCode = join(fullCode.substring(0, 2), "0000");
            // 市
            String cityCode = join(fullCode.substring(0, 4), "00");
            // 区
            String countyCode = fullCode.substring(0, 6);

            if (log.isDebugEnabled()) {
                log.debug("根据code: {} 获取到的省: {} 市: {} 区: {}", fullCode, provinceCode, cityCode, countyCode);
            }

            if (isBlank(provinceCode) || isBlank(cityCode) || isBlank(countyCode)) {
                return url;
            }

            return join(url, "&provinceCode=", provinceCode, "&cityCode=", cityCode, "&countyCode=", countyCode);
        } catch (Exception e) {
            log.error("处理商家入驻的城市信息出错", e);
        }

        return url;
    }

    private String replaceUserIdAndMerchant(String url, Long userId, Long merchantId) {
        logger.debug(url);
        String replace = url.replace("{userId}", Objects.toString(userId));
        return replace.replace("{merchantId}", Objects.toString(merchantId));
    }


    private String replaceMerchant(String url, Long merchantId) {
        return url.contains("{merchantId}") ? url.replace("{merchantId}", Objects.toString(merchantId)) : url;
    }

    private String replacePhone(String url, String phone) {
        return url.contains("{phone}") ? url.replace("{phone}", phone) : url;
    }


    @Override
    public void clickCenter(Long merchantId) {
        //判断redis是否有商户id
        String string = redisStringAdapter.getString(RedisConfig.MERCHANT_ID_KEY.copy().appendKey(merchantId));
        if (isBlank(string)) {
            redisStringAdapter.set(RedisConfig.MERCHANT_ID_KEY.copy().appendKey(merchantId), merchantId);
        }
    }

    @Override
    public MerchantInfoSimpleVO getSimpleMerchantInfoByGoodsId(Long goodsId) {
        return merchantInfoMapper.getMerchantByGoodsId(goodsId);
    }
}