package com.bxm.localnews.merchants.service.impl;

import com.bxm.localnews.merchant.common.config.RedisConfig;
import com.bxm.localnews.merchant.coupon.service.CouponInfoService;
import com.bxm.localnews.merchant.domain.MerchantWorkMapper;
import com.bxm.localnews.merchant.dto.UserInfoDTO;
import com.bxm.localnews.merchant.dto.coupon.MerchantCouponOverviewDTO;
import com.bxm.localnews.merchant.integration.UserIntegrationService;
import com.bxm.localnews.merchant.param.coupon.MerchantCouponStatusPageParam;
import com.bxm.localnews.merchant.service.goods.GoodsService;
import com.bxm.localnews.merchants.cache.collected.MerchantCollectRedisRefresh;
import com.bxm.localnews.merchants.cache.detail.MerchantInfoRedisRefresh;
import com.bxm.localnews.merchants.dto.*;
import com.bxm.localnews.merchants.param.MerChantListParam;
import com.bxm.localnews.merchants.param.MerchantDetailParam;
import com.bxm.localnews.merchants.service.MerchantListService;
import com.bxm.localnews.merchants.vo.MerchantChannelQuVo;
import com.bxm.localnews.merchants.vo.MerchantWorkListVo;
import com.bxm.localnews.merchants.vo.UserForMerchantInfo;
import com.bxm.newidea.component.geo.dto.Coordinate;
import com.bxm.newidea.component.geo.service.GeoService;
import com.bxm.newidea.component.redis.KeyGenerator;
import com.bxm.newidea.component.redis.RedisListAdapter;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.bxm.newidea.component.vo.PageWarper;
import com.fasterxml.jackson.core.type.TypeReference;
import com.github.pagehelper.Page;
import lombok.AllArgsConstructor;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.BeanUtils;
import org.springframework.stereotype.Service;
import org.springframework.util.CollectionUtils;

import java.util.*;
import java.util.stream.Collectors;

/**
 * @author pf.w
 * @date 2020/7/13 13:58
 */
@Service
@Log4j2
@AllArgsConstructor
public class MerchantWorkServiceImpl implements MerchantListService {

    private GeoService geoService;

    private MerchantWorkMapper merchantWorkMapper;

    private UserIntegrationService userIntegrationService;

    private GoodsService goodsService;

    private RedisListAdapter redisListAdapter;

    private MerchantCollectRedisRefresh merchantCollectRedisRefresh;

    private final MerchantInfoRedisRefresh merchantInfoRedisRefresh;

    private CouponInfoService couponInfoService;

    /**
     * 一个全局默认比对值
     */
    private static int DEFAULT_NUM = 0;

    /**
     * 距离比较值
     */
    private static int DISTANCE_LEVE = 1000;

    private static int DISTANCE_LEVE2 = 500;

    @Override
    public PageWarper<MerchantWorkListDTO> getMerchantList(MerChantListParam param) {
        if (log.isDebugEnabled()) {
            log.debug("param : {}", param);
        }
        //获取商家列表
        List<MerchantWorkListDTO> resultList = this.loadMerchantListInfo(param);

        if (resultList.size() <= 0) {
            return new PageWarper<>();
        }

        if (log.isDebugEnabled()) {
            log.debug("result : {}", resultList);
        }

        //排序
        resultList = sorted(resultList, param);

        //分页
        int start = (param.getPageNum() - 1) * param.getPageSize();
        int end = param.getPageNum() * param.getPageSize();
        if (start < 0) {
            start = 0;
        }
        if (end > resultList.size()) {
            end = resultList.size();
        }
        List<MerchantWorkListDTO> pageRecords = resultList.subList(start, end);

        //商品列表
        if (!CollectionUtils.isEmpty(resultList)) {
            pageRecords.forEach(merchantWorkListDTO -> {
                if (merchantWorkListDTO != null) {
                    List<MerchantWorkGoodsDTO> merchantWorkGoodsDtoS = goodsService.queryWorkGoods(merchantWorkListDTO.getMerchantId());
                    merchantWorkListDTO.setGoodsDTOList(merchantWorkGoodsDtoS);
                }
            });
        }

        Page<MerchantWorkListDTO> page = new Page<>(param.getPageNum(), param.getPageSize());
        page.setTotal(resultList.size());
        page.addAll(pageRecords);

        return new PageWarper<>(page);
    }

    @Override
    public MerchantDetailDTO getMerchantDetail(MerchantDetailParam param) {
        if (log.isDebugEnabled()) {
            log.debug("param : {},merchantId：{}", param, param.getMerchantId());
        }

        //获取详情
        MerchantInfoCacheDTO merchantInfo = merchantInfoRedisRefresh.getMerchantInfoCache(param.getMerchantId());

        if (Objects.isNull(merchantInfo)) {
            log.error("获取详情时 商家无数据  merchantId : {}", param.getMerchantId());
            return new MerchantDetailDTO();
        }

        if (log.isDebugEnabled()) {
            log.debug("merchantInfo : {}", merchantInfo);
        }
        //数据转化
        MerchantDetailDTO merchantDetailDTO = convertToDetail(merchantInfo, param);

        //填充距离字段
        this.distanceCalculationDetail(param, merchantDetailDTO);
        return merchantDetailDTO;
    }

    /**
     * 填充商品详情中的距离
     *
     * @param param             详情参数
     * @param merchantDetailDTO 详情返回实体
     */
    private void distanceCalculationDetail(MerchantDetailParam param, MerchantDetailDTO merchantDetailDTO) {
        if (param.getLatitude() != null && param.getLongitude() != null) {
            Coordinate userLngAndLat = new Coordinate();
            userLngAndLat.setLongitude(param.getLongitude());
            userLngAndLat.setLatitude(param.getLatitude());
            if (merchantDetailDTO.getLat() != null && merchantDetailDTO.getLng() != null) {
                Long distance = fillDistance(userLngAndLat, merchantDetailDTO.getLat(), merchantDetailDTO
                        .getLng());
                merchantDetailDTO.setDistance(distance(distance));
            }
        }
    }

    private MerchantDetailDTO convertToDetail(MerchantInfoCacheDTO merchantInfo, MerchantDetailParam param) {
        MerchantDetailDTO result = new MerchantDetailDTO();

        BeanUtils.copyProperties(merchantInfo, result);
        result.setMerchantId(merchantInfo.getId());
        result.setIsVip(merchantInfo.getJudgeMarker());
        result.setShortDesc(merchantInfo.getDes());
        result.setTownName(merchantInfo.getAddress());
        result.setBusinessArea(Objects.nonNull(merchantInfo.getAddress())
                ? merchantInfo.getAddress()
                : null);
        result.setWorkTime(merchantInfo.getOpenType().equals(DEFAULT_NUM)
                ? "全天"
                : merchantInfo.getOpenTime() + "-" + merchantInfo.getCloseTime());
        result.setFacadeUrl(StringUtils.isNotBlank(merchantInfo.getHeadPics())
                ? Arrays.asList(merchantInfo.getHeadPics().split(","))
                : new ArrayList<>());
        result.setImgUrls(StringUtils.isNotBlank(merchantInfo.getAlbum())
                ? Arrays.asList(merchantInfo.getAlbum().split(","))
                : new ArrayList<>());
        result.setServicePhone(StringUtils.isNotBlank(merchantInfo.getMobile())
                ? merchantInfo.getMobile()
                : null);
        result.setDescription(StringUtils.isNotBlank(merchantInfo.getDetailContent())
                ? merchantInfo.getDetailContent()
                : null);

        if (merchantInfo.getLat() != null) {
            result.setLat(merchantInfo.getLat().doubleValue());
        }
        if (merchantInfo.getLng() != null) {
            result.setLng(merchantInfo.getLng().doubleValue());
        }

        //填充收藏信息
        boolean collected = merchantCollectRedisRefresh.checkMerchantIsCollected(param.getUserId(), param.getMerchantId());
        result.setHasCollected(collected ? 1 : DEFAULT_NUM);

        //填充分类名称
        result.setCategoryName(Objects.nonNull(merchantInfo.getCategoryName()) ? merchantInfo.getCategoryName() : null);

        //填充user信息
        if (merchantInfo.getUserId() != null) {
            UserInfoDTO userInfoDTO = userIntegrationService.getUserFromRedisDB(merchantInfo.getUserId());
            if (Objects.nonNull(userInfoDTO)) {
                UserForMerchantInfo userInfo = UserForMerchantInfo.builder()
                        .userId(merchantInfo.getUserId().intValue())
                        .headimgurl(StringUtils.isNotBlank(userInfoDTO.getHeadImg()) ? userInfoDTO.getHeadImg() : "")
                        .nickName(StringUtils.isNotBlank(userInfoDTO.getNickname()) ? userInfoDTO.getNickname() : "")
                        .userJobDesc(StringUtils.isNotBlank(userInfoDTO.getJobTitle()) ? userInfoDTO.getJobTitle() : "")
                        .build();
                result.setUserForMerchantInfo(userInfo);
            }
        }
        return result;
    }

    /**
     * 默认展示综合排序，展示顺序优先按后台位置顺序设置（只显示同一站
     * 点内商家），
     * 未设置顺序的商家按商家按距离用户位置从近到远进行排序，
     * 若用户未授权地理位置，按商家成为会员的时间先后排序，先成为会员
     * 排前面
     * 离我最近:按距离从近到远进行排序，未授权地理位置，点击离我最近，申请获取用户地理位置，拒绝则留着综合排序
     *
     * @param list  商家列表list
     * @param param 列表接口参数
     * @return 经过排序后的商家列表
     */
    private List<MerchantWorkListDTO> sorted(List<MerchantWorkListDTO> list, MerChantListParam param) {
        //区别total和category进行排序
        if (Objects.isNull(param.getCategoryId()) || param.getCategoryId() == DEFAULT_NUM) {
            //截取未过期置顶商家-total
            List<MerchantWorkListDTO> topForBmTotal = list.stream().filter(item -> item.getBianMinOrder() > 0 && (DateUtils.before
                    (item.getBianMinStartTime()) && DateUtils.after(item.getBianMinEndTime()))).sorted(Comparator.comparing(MerchantWorkListDTO::getBianMinOrder))
                    .collect(Collectors.toList());

            //设置便民首页置状态
            topForBmTotal.forEach(i -> i.setBianMinTopStatus(1));
            //最终排序
            return sortedCommon(list, topForBmTotal, param);
        } else {
            //截取未过期置顶商家-category
            List<MerchantWorkListDTO> topForChannel = list.stream().filter(item -> item.getChannelOrder() > 0 && (DateUtils.before
                    (item.getChannelStartTime()) && DateUtils.after(item.getChannelEndTime()))).sorted(Comparator.comparing(MerchantWorkListDTO::getChannelOrder))
                    .collect(Collectors.toList());

            //设置分类列表置顶过期状态
            topForChannel.forEach(i -> i.setChannelTopStatus(1));
            //最终排序
            return sortedCommon(list, topForChannel, param);
        }
    }

    private List<MerchantWorkListDTO> sortedCommon(List<MerchantWorkListDTO> list, List<MerchantWorkListDTO> listTop, MerChantListParam param) {
        //未获得位置权限时
        if (param.getLongitude() == 0 && param.getLatitude() == 0) {
            list.removeAll(listTop);
            list.sort(Comparator.comparing(MerchantWorkListDTO::getToVipDate, Comparator.nullsLast(Date::compareTo)).reversed());
            listTop.addAll(list);
            //屏蔽缓存里面的位置信息
            listTop.forEach(item -> {
                item.setDistance("");
                item.setDistanceNum(null);
            });
            return listTop;
        }

        if (param.getSortType() == DEFAULT_NUM) {
            list.removeAll(listTop);
            list.sort(Comparator.comparing(MerchantWorkListDTO::getDistanceNum, Comparator.nullsLast(Long::compareTo)));
            listTop.addAll(list);
            return listTop;
        } else {
            list.sort(Comparator.comparing(MerchantWorkListDTO::getDistanceNum, Comparator.nullsLast(Long::compareTo)));
            return list;
        }
    }

    /**
     * 获取商家列表信息  优先取缓存数据，没有从数据库获取
     */
    private List<MerchantWorkListDTO> loadMerchantListInfo(MerChantListParam param) {
        List<MerchantWorkListDTO> list;
        //优先从缓存获取
        KeyGenerator key = getListKey(param.getAreaCode(), param.getCategoryId());
        //获取缓存中的所有商家
        list = redisListAdapter.leftIndex(key, -1, new TypeReference<MerchantWorkListDTO>() {
        });
        if (CollectionUtils.isEmpty(list)) {
            list = getFromDb(param);
        }
        //填充距离字段
        this.distanceCalculation(param, list);
        return list;
    }

    @Override
    public List<MerchantWorkListDTO> getFromDb(MerChantListParam param) {
        List<MerchantWorkListVo> listFromDb = merchantWorkMapper.selectByCategoryId(param.getCategoryId()
                , param.getAreaCode());

        KeyGenerator listKey = this.getListKey(param.getAreaCode(), param.getCategoryId());

        //既然刷新缓存，啥也别问，有没有都先清一下
        redisListAdapter.remove(listKey);

        if (listFromDb.size() > 0) {
            List<MerchantWorkListDTO> list = listFromDb.stream().map(this::convert).collect(Collectors.toList());
            redisListAdapter.leftPush(listKey, list);
            return list;
        } else {
            return new ArrayList<>();
        }
    }

    @Override
    public void removeCache(String areaCode, Long categoryId) {
        redisListAdapter.remove(this.getListKey(areaCode, categoryId));
        redisListAdapter.remove(this.getListKey(areaCode, 0L));
    }

    @Override
    public List<MerchantChannelInfo> getChannelQuMerchantList(MerChantListParam param) {
        List<MerchantChannelQuVo> merchantChannelQuVos = merchantWorkMapper.selectForQuByCategoryId(param
                .getCategoryId(), param.getAreaCode());

        return merchantChannelQuVos.stream().map(s -> {
            MerchantChannelInfo result = new MerchantChannelInfo();
            BeanUtils.copyProperties(s, result);
            result.setIcon(s.getLogo());
            return result;
        }).collect(Collectors.toList());
    }

    private MerchantWorkListDTO convert(MerchantWorkListVo merchantWorkListVo) {
        MerchantWorkListDTO merchantWorkListDTO = new MerchantWorkListDTO();
        BeanUtils.copyProperties(merchantWorkListVo, merchantWorkListDTO);
        merchantWorkListDTO.setLat(merchantWorkListVo.getLat().doubleValue());
        merchantWorkListDTO.setLng(merchantWorkListVo.getLng().doubleValue());
        merchantWorkListDTO.setMerchantId(merchantWorkListVo.getId());
        merchantWorkListDTO.setShortDesc(Objects.nonNull(merchantWorkListVo.getDes())
                ? merchantWorkListVo.getDes()
                : null);
        merchantWorkListDTO.setTownName(Objects.nonNull(merchantWorkListVo.getAddress())
                ? merchantWorkListVo.getAddress()
                : null);
        merchantWorkListDTO.setHeadImg(Objects.nonNull(merchantWorkListVo.getHeadPics())
                ? Arrays.asList(merchantWorkListVo.getHeadPics().split(",")).get(0)
                : null);
        merchantWorkListDTO.setWorkTime(merchantWorkListVo.getOpenType().equals(DEFAULT_NUM) ?
                "全天"
                : merchantWorkListVo.getOpenTime() + "-" + merchantWorkListVo.getCloseTime());
        merchantWorkListDTO.setIsVip(merchantWorkListVo.getJudgeMarker());
        return merchantWorkListDTO;
    }

    /**
     * 填充商家列表中的“距离”数据
     */
    private void distanceCalculation(MerChantListParam param, List<MerchantWorkListDTO> list) {
        if (param.getLatitude() != null && param.getLongitude() != null) {
            Coordinate userLngAndLat = new Coordinate();
            userLngAndLat.setLongitude(param.getLongitude());
            userLngAndLat.setLatitude(param.getLatitude());

            for (MerchantWorkBase merchantListDTO : list) {
                if (merchantListDTO.getLat() != null && merchantListDTO.getLng() != null) {
                    Long distance = fillDistance(userLngAndLat, merchantListDTO.getLat(), merchantListDTO.getLng());
                    merchantListDTO.setDistance(distance(distance));
                    merchantListDTO.setDistanceNum(distance);
                }
            }
        }
    }

    /**
     * 距离转化
     * 小于500m统一显示<500m,500~999m:显示具距离
     * 1公里及以上：显示X.Xkm，进给到小数点一位
     */
    private String distance(Long distance) {
        if (distance >= DISTANCE_LEVE) {
            return (Math.round(distance / 100d) / 10d) + "km";
        }
        if (distance >= DISTANCE_LEVE2) {
            return distance + "m";
        }
        return "<500m";
    }

    /**
     * 根据经纬度计算距离
     */
    private Long fillDistance(Coordinate userLngAndLat, Double geoLat, Double geoLng) {
        Coordinate lngAndLat = new Coordinate();
        lngAndLat.setLongitude(geoLng);
        lngAndLat.setLatitude(geoLat);
        return geoService.getDistance(userLngAndLat, lngAndLat);
    }

    /**
     * 根据地区码和分类id定义缓存key
     */
    private KeyGenerator getListKey(String areaCode, Long categoryId) {
        return RedisConfig.MERCHANT_LIST_WORK_KEY.copy()
                .appendKey(StringUtils.isNotBlank(areaCode) ? areaCode : "all")
                .appendKey(categoryId);
    }

    @Override
    public PageWarper<MerchantWorkV2ListDTO> merchantListInfoV2(MerChantListParam param) {
        if (log.isDebugEnabled()) {
            log.debug("param : {}", param);
        }
        //获取商家列表
        List<MerchantWorkListDTO> resultList = this.loadMerchantListInfo(param);
        if (resultList.isEmpty()) {
            return new PageWarper<>();
        }
        if (log.isDebugEnabled()) {
            log.debug("resultList : {}", resultList);
        }
        //排序，规则还按照原来的
        resultList = sorted(resultList, param);

        List<MerchantWorkV2ListDTO> workV2ListDTOS = new ArrayList<>();

        //分页
        int start = (param.getPageNum() - 1) * param.getPageSize();
        int end = param.getPageNum() * param.getPageSize();
        if (start < 0) {
            start = 0;
        }
        if (end > resultList.size()) {
            end = resultList.size();
        }
        List<MerchantWorkListDTO> pageRecords = resultList.subList(start, end);
        //商品列表
        if (!CollectionUtils.isEmpty(resultList)) {
            pageRecords.forEach(merchantWorkListDTO -> {
                if (merchantWorkListDTO != null) {
                    MerchantWorkV2ListDTO merchantWorkV2ListDTO = new MerchantWorkV2ListDTO();
                    BeanUtils.copyProperties(merchantWorkListDTO, merchantWorkV2ListDTO);
                    List<MerchantWorkGoodsV2DTO> fixGoodsList = getFixGoodsList(merchantWorkListDTO.getMerchantId());
                    merchantWorkV2ListDTO.setGoodsDTOList(fixGoodsList);
                    workV2ListDTOS.add(merchantWorkV2ListDTO);
                }
            });
        }
        Page<MerchantWorkV2ListDTO> page = new Page<>(param.getPageNum(), param.getPageSize());
        page.setTotal(resultList.size());
        page.addAll(workV2ListDTOS);

        return new PageWarper<>(page);
    }

    private List<MerchantWorkGoodsV2DTO> getFixGoodsList(Long merchantId) {
        List<MerchantWorkGoodsV2DTO> merchantWorkGoodsV2DTOS = new ArrayList<>();
        //增加优惠券显示逻辑 优惠券+团购最多显示两条，出售中的商品少于两个时，显示优惠券（优先显示商品），无可领取优惠券则不显示
        int maxLength = 2;
        List<MerchantWorkGoodsDTO> merchantWorkGoodsDtoS = goodsService.queryWorkGoods(merchantId);
        int merchantWorkGoodsNums = 0;
        if (merchantWorkGoodsDtoS != null) {
            merchantWorkGoodsNums = merchantWorkGoodsDtoS.size();
            merchantWorkGoodsDtoS.forEach(merchantWorkGoodsDTO -> {
                MerchantWorkGoodsV2DTO merchantWorkGoodsV2DTO = new MerchantWorkGoodsV2DTO();
                merchantWorkGoodsV2DTO.setType(1);
                MerchantWorkGroupBuyDTO merchantWorkGroupBuyDTO = MerchantWorkGroupBuyDTO.builder()
                        .goodsId(merchantWorkGoodsDTO.getGoodsId())
                        .name(merchantWorkGoodsDTO.getName())
                        .originalPrice(merchantWorkGoodsDTO.getOriginalPrice())
                        .price(merchantWorkGoodsDTO.getPrice())
                        .vipDiscount(merchantWorkGoodsDTO.getVipDiscount())
                        .vipPrice(merchantWorkGoodsDTO.getVipPrice())
                        .build();
                merchantWorkGoodsV2DTO.setGroupBuyDTO(merchantWorkGroupBuyDTO);
                merchantWorkGoodsV2DTOS.add(merchantWorkGoodsV2DTO);
            });
        }
        if (maxLength - merchantWorkGoodsNums > 0) {
            MerchantCouponStatusPageParam pageParam = new MerchantCouponStatusPageParam();
            pageParam.setMerchantId(merchantId);
            //优惠券+团购最多显示两条,获取剩下的数据
            pageParam.setPageSize(maxLength - merchantWorkGoodsNums);
            PageWarper<MerchantCouponOverviewDTO> query = couponInfoService.query(pageParam);
            List<MerchantCouponOverviewDTO> list = query.getList();
            list.forEach(merchantCouponOverviewDTO -> {
                MerchantWorkGoodsV2DTO merchantWorkGoodsV2DTO = new MerchantWorkGoodsV2DTO();
                merchantWorkGoodsV2DTO.setType(2);
                MerchantWorkCouponDTO merchantWorkCouponDTO = MerchantWorkCouponDTO.builder()
                        .conditionAmount(merchantCouponOverviewDTO.getConditionAmount() ? 1 : 0)
                        .couponId(merchantCouponOverviewDTO.getCouponId())
                        .discount(merchantCouponOverviewDTO.getDiscount())
                        .totalAmount(merchantCouponOverviewDTO.getTotalAmount())
                        .build();
                merchantWorkGoodsV2DTO.setCouponDTO(merchantWorkCouponDTO);
                merchantWorkGoodsV2DTOS.add(merchantWorkGoodsV2DTO);
            });
        }
        return merchantWorkGoodsV2DTOS;
    }
}
