package com.bxm.localnews.msg.sender;

import com.alibaba.fastjson.JSON;
import com.bxm.localnews.mq.common.model.dto.MessageFilterInfo;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.common.model.dto.PushPayloadInfo;
import com.bxm.localnews.mq.common.param.DingtalkMessage;
import com.bxm.localnews.msg.facade.DingtalkFacadeService;
import com.bxm.localnews.msg.facade.MessageFacadeService;
import com.bxm.localnews.msg.facade.PushFacadeService;
import com.bxm.localnews.msg.facade.UserEventFacadeService;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.scheduling.concurrent.ThreadPoolTaskExecutor;
import org.springframework.stereotype.Component;

import java.util.Objects;

import static org.apache.commons.lang.StringUtils.isBlank;

/**
 * APP应用消息推送，仅支持单条消息推送，不支持全量推送
 * 此服务面向的是其他服务
 *
 * @author liujia
 * @date 2020-07-03 15:58
 **/
@Component
@Slf4j
public class MessageSender {

    private final ThreadPoolTaskExecutor asyncTheadPoolExecutor;

    private final PushFacadeService pushFacadeService;

    private final UserEventFacadeService userEventFacadeService;

    private final MessageFacadeService messageFacadeService;

    private final DingtalkFacadeService dingtalkFacadeService;

    @Autowired
    public MessageSender(ThreadPoolTaskExecutor asyncTheadPoolExecutor,
                         PushFacadeService pushFacadeService,
                         UserEventFacadeService userEventFacadeService,
                         MessageFacadeService messageFacadeService,
                         DingtalkFacadeService dingtalkFacadeService) {
        this.asyncTheadPoolExecutor = asyncTheadPoolExecutor;
        this.pushFacadeService = pushFacadeService;
        this.userEventFacadeService = userEventFacadeService;
        this.messageFacadeService = messageFacadeService;
        this.dingtalkFacadeService = dingtalkFacadeService;
    }

    /**
     * 推送消息给用户
     *
     * @param message 推送消息内容，不支持群推，仅支持单个用户推送
     */
    public void sendPushMessage(PushMessage message) {
        if (null == message) {
            return;
        }

        asyncTheadPoolExecutor.execute(() -> pushFacadeService.pushMsg(message));
    }

    /**
     * 添加消息到用户消息中心，但是不会进行推送
     *
     * @param message 消息内容,消息内容中必须有用户ID
     */
    public void sendUserMessage(PushMessage message) {
        if (null == message) {
            return;
        }

        if (message.getPushReceiveScope() == null || message.getPushReceiveScope().getUserId() == null) {
            log.info("消息接收人为空，不予处理，消息内容为：{}", JSON.toJSONString(message));
        }
        asyncTheadPoolExecutor.execute(() -> {
            Long userId = message.getPushReceiveScope().getUserId();
            messageFacadeService.addMsg(message, userId);
        });
    }

    /**
     * 发送用户事件
     *
     * @param payloadInfo 用户事件信息
     */
    public void sendUserEvent(PushPayloadInfo payloadInfo) {
        if (null == payloadInfo || payloadInfo.getTargetUserId() == null) {
            log.warn("用户事件对象不存在或者接收用户ID为空，载体信息为：{}", JSON.toJSONString(payloadInfo));
            return;
        }

        asyncTheadPoolExecutor.execute(() -> userEventFacadeService.add(payloadInfo));
    }

    /**
     * 推送消息的召回处理
     *
     * 说明 ：因对后续召回逻辑预判有限，目前该方法处理比较简陋，只处理了评论消息不可见后的删除逻辑
     *         如后续有相关逻辑，可在此基础上进行再开发
     * @param info
     */
    public void sendMsgFilterEvent(MessageFilterInfo info){
        if(Objects.isNull(info)){
            log.warn("用户上报的消息召回信息有误  : {}",JSON.toJSONString(info));
        }
        asyncTheadPoolExecutor.execute(() -> messageFacadeService.addMsgFilter(info));
    }

    /**
     * 发送钉钉消息
     *
     * @param message 钉钉消息内容
     */
    public void sendDingtalk(DingtalkMessage message) {
        if (null == message || isBlank(message.getContent())) {
            log.warn("钉钉推送消息为空或内容为空");
            return;
        }

        asyncTheadPoolExecutor.execute(() -> dingtalkFacadeService.pushDingtalk(message));
    }
}
