package com.bxm.localnews.mq.consume.push.platform.impl;

import java.util.Calendar;
import java.util.Date;
import java.util.List;

import javax.annotation.Resource;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.bxm.component.httpclient.service.HttpClientService;
import com.bxm.localnews.mq.common.constant.PlatformTypeEnum;
import com.bxm.localnews.mq.common.constant.TemplateTypeEnum;
import com.bxm.localnews.mq.common.model.dto.PushMessage;
import com.bxm.localnews.mq.config.AppPushProperties;
import com.bxm.localnews.mq.consume.umeng.AndroidNotification;
import com.bxm.localnews.mq.consume.umeng.UmengNotification;
import com.bxm.localnews.mq.consume.umeng.android.AndroidUnicast;
import com.bxm.localnews.mq.consume.umeng.ios.AlertMsg;
import com.bxm.localnews.mq.consume.umeng.ios.IOSUnicast;
import com.bxm.localnews.msg.vo.UserDeviceBean;
import com.bxm.newidea.component.tools.DateUtils;
import com.bxm.newidea.component.tools.StringUtils;

import org.apache.commons.codec.digest.DigestUtils;
import org.springframework.cloud.context.config.annotation.RefreshScope;
import org.springframework.stereotype.Component;

import static com.bxm.localnews.mq.common.constant.PushSoundEnum.IOS_MUTE;

@Component
@RefreshScope
public class UmengPushPlatform extends AbstractPushPlatform {

    // The host
    protected static final String host = "http://msg.umeng.com";

    // The post path
    protected static final String postPath = "/api/send";

    @Resource
    private HttpClientService httpClientService;

    @Resource
    private AppPushProperties appPushProperties;

    @Override
    public PlatformTypeEnum getType() {
        return PlatformTypeEnum.UMENG;
    }

    @Override
    void singlePush(PushMessage message, UserDeviceBean userDevice) {
        logger.debug("消息推送类型为：[{}]", message.getType());
        UmengNotification unicast;
        if (isIOS(userDevice)) {
            logger.debug("推送ios消息");
            unicast = createIOSUnicast(message, userDevice);
        } else {
            logger.debug("推送android消息");
            unicast = createAndroidUnicast(message, userDevice);
        }
        String postUrl = host + postPath;
        String postBody = unicast.getPostBody();
        String sign = StringUtils.EMPTY;
        logger.info("推送消息：[{}]", postBody);
        try {
            sign = DigestUtils.md5Hex(("POST" + postUrl + postBody + unicast.getAppMasterSecret()).getBytes("utf8"));
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
        }
        postUrl = postUrl + "?sign=" + sign;

        String response = httpClientService.doPost(postUrl, postBody);
        logger.info("umeng push result:[{}]", response);
        //推送完成回调
        callback(message, userDevice, response);

    }

    @Override
    void groupPush(PushMessage message, List<UserDeviceBean> userDevice) {
        for (UserDeviceBean userDeviceBean : userDevice) {
            logger.debug("当前推送的用户信息为：[{}]", userDevice.toString());
            singlePush(message, userDeviceBean);
        }
    }

    /**
     * 创建ios单播推送消息
     *
     * @param message
     * @param userDevice
     * @return
     */
    private IOSUnicast createIOSUnicast(PushMessage message, UserDeviceBean userDevice) {
        try {
            IOSUnicast unicast = new IOSUnicast();

            String timestamp = Integer.toString((int) (System.currentTimeMillis() / 1000));
            unicast.setPredefinedKeyValue("timestamp", timestamp);

            unicast.setDeviceToken(userDevice.getPushToken());
            AlertMsg alertMsg = new AlertMsg(message.getTitle(), message.getContent());
            unicast.setAlert(JSONObject.toJSON(alertMsg));

            unicast.setContentAvailable(0);
            unicast.setBadge(1);

            if (appPushProperties.isUmengProductionMode()) {
                unicast.setProductionMode();
            } else {
                unicast.setTestMode();
            }

            if (message.isMute()) {
                unicast.setSound(IOS_MUTE.getIosSound());
            } else {
                unicast.setSound(message.getSound().getIosSound());
            }

            //组装payload
            unicast.setCustomizedField("extend", JSONObject.toJSON(message.getPayloadInfo().getExtend()));
            unicast.setCustomizedField("type", message.getPayloadInfo().getType());
            unicast.setCustomizedField("title", message.getTitle());
            unicast.setCustomizedField("content", message.getContent());
            unicast.setCustomizedField("msgId", message.getPayloadInfo().getMsgId());
            if (message.getPeriodTime() != null) {
                Date now = new Date();
                Date expireDate = DateUtils.addField(now, Calendar.HOUR, message.getPeriodTime());
                unicast.setExpireTime(DateUtils.formatAtWill(expireDate, DateUtils.DATE_TIME_FORMAT));
            }

            return unicast;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }

    /**
     * 创建安卓单播消息
     *
     * @param message
     * @param userDevice
     * @return
     */
    private AndroidUnicast createAndroidUnicast(PushMessage message, UserDeviceBean userDevice) {
        try {
            AndroidUnicast unicast = new AndroidUnicast();
            String timestamp = Integer.toString((int) (System.currentTimeMillis() / 1000));
            unicast.setPredefinedKeyValue("timestamp", timestamp);

            unicast.setDeviceToken(userDevice.getPushToken());
            //设置通知内容
            unicast.setBody(message.getTitle(), message.getTitle(), message.getContent());

            //设置提醒方式
            unicast.setPlay(message.isShock(), message.isMute());
            if (message.isMute()) {
                unicast.setSound(IOS_MUTE.getAndriodSound());
            } else {
                unicast.setSound(message.getSound().getAndriodSound());
            }

            if (TemplateTypeEnum.TRANSMISSION.equals(message.getType())) {
                unicast.setDisplayType(AndroidNotification.DisplayType.MESSAGE);
            } else {
                unicast.setDisplayType(AndroidNotification.DisplayType.NOTIFICATION);
            }
//            Map<String, Object> extraMap = Maps.newHashMap();
//            extraMap.put("extra", message.getPayloadInfo());
//            unicast.setCustomField(JSONObject.parseObject(JSON.toJSONString(extraMap)));
            unicast.setCustomField(JSONObject.parseObject(JSONObject.toJSONString(message.getPayloadInfo())));
            //设置默认打开app
            unicast.goAppAfterOpen();

            if (appPushProperties.isUmengProductionMode()) {
                unicast.setProductionMode();
            } else {
                unicast.setTestMode();
            }
            unicast.setExtra(JSONObject.parseObject(JSON.toJSONString(message.getPayloadInfo())));
//            JSONObject object = JSONObject.parseObject(JSONObject.toJSONString(message.getPayloadInfo().getExtend()));
//            unicast.setExtraField("extend", object);
            if (message.getPeriodTime() != null) {
                Date now = new Date();
                Date expireDate = DateUtils.addField(now, Calendar.HOUR, message.getPeriodTime());
                unicast.setExpireTime(DateUtils.formatAtWill(expireDate, DateUtils.DATE_TIME_FORMAT));
            }
            return unicast;
        } catch (Exception e) {
            logger.error(e.getMessage(), e);
            return null;
        }
    }
}
