package com.bxm.localnews.news.image;

import com.alibaba.fastjson.JSONObject;
import com.bxm.localnews.news.config.OssProperties;
import com.bxm.localnews.news.vo.PostImgVo;
import com.bxm.newidea.component.tools.FileUtils;
import com.bxm.newidea.component.tools.StringUtils;
import com.google.common.collect.Lists;
import org.springframework.stereotype.Component;
import org.springframework.util.CollectionUtils;

import javax.annotation.Resource;
import java.util.ArrayList;
import java.util.List;
import java.util.stream.Collectors;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.replace;

/**
 * 图片处理辅助类,所有图片相关的处理逻辑均聚合在此处
 *
 * @author liujia
 * @date 1/14/21 11:39 AM
 **/
@Component
public class ImageHelper {

    @Resource
    private OssProperties ossProperties;

    /**
     * 从图片数组中选取封面图
     *
     * @param coverStr    封面图数组
     * @param imgList     内容中的图片数组
     * @param coverSelect 1.取视频封面 2.取图片封面 3.取图片内容
     */
    public List<PostImgVo> converSelect(String coverStr, List<PostImgVo> imgList, Byte coverSelect) {
        List<PostImgVo> coverArray = new ArrayList<>();
        //视频图片数量
        Integer videoSize = 1;
        //内容图片数量
        Integer imageSize = 3;
        if (coverSelect == 0) {
            coverArray = getCoverByCoverStr(coverStr);
            if (CollectionUtils.isEmpty(coverArray)) {
                coverArray = getCoverByVideo(imgList, videoSize);
            }
            if (CollectionUtils.isEmpty(coverArray)) {
                coverArray = getCoverByImgUrl(imgList, imageSize);
            }
        } else if (coverSelect == 1) {
            coverArray = getCoverByVideo(imgList, videoSize);
            if (CollectionUtils.isEmpty(coverArray)) {
                coverArray = getCoverByImgUrl(imgList, imageSize);
            }
        } else if (coverSelect == 2) {
            videoSize = 3;
            coverArray = getCoverByVideo(imgList, videoSize);
            if (coverArray.size() < videoSize) {
                imageSize = videoSize - coverArray.size();
                List<PostImgVo> coverByImgUrl = getCoverByImgUrl(imgList, imageSize);
                coverArray.addAll(coverByImgUrl);
            }
        }

        return coverArray;
    }

    public List<PostImgVo> getDetailFromPost(List<PostImgVo> imgList) {
        List<PostImgVo> coverList = new ArrayList<>();

        List<PostImgVo> videoCoverTempList = new ArrayList<>();
        List<PostImgVo> imgCoverTempList = new ArrayList<>();
        if (!CollectionUtils.isEmpty(imgList)) {
            for (PostImgVo postImgVo : imgList) {
                if (isBlank(postImgVo.getType()) || isBlank(postImgVo.getImgUrl())) {
                    continue;
                }
                if ("VIDEO".equals(postImgVo.getType())) {
                    postImgVo.setImgUrl(buildVideoCover(postImgVo.getImgUrl()));
                    postImgVo.setVideoUrl(buildVideoUrlCover(postImgVo.getVideoUrl()));
                    videoCoverTempList.add(postImgVo);
                } else if ("IMG".equals(postImgVo.getType())) {
                    postImgVo.setImgUrl(buildImgCover(postImgVo.getImgUrl()));
                    imgCoverTempList.add(postImgVo);
                }
            }
        }

        if (!CollectionUtils.isEmpty(videoCoverTempList)) {
            coverList.addAll(videoCoverTempList);
        }
        if (!CollectionUtils.isEmpty(imgCoverTempList)) {
            coverList.addAll(imgCoverTempList);
        }

        return coverList.stream().limit(9).collect(Collectors.toList());
    }

    /**
     * 内容图转封面
     *
     * @param imgList 图片列表
     * @param size    个数
     * @return 封面图
     */
    private List<PostImgVo> getCoverByImgUrl(List<PostImgVo> imgList, Integer size) {
        List<PostImgVo> result = Lists.newArrayList();
        if (!CollectionUtils.isEmpty(imgList)) {
            for (PostImgVo postImgVo : imgList) {
                if (isBlank(postImgVo.getType()) || isBlank(postImgVo.getImgUrl())) {
                    continue;
                }
                if ("IMG".equals(postImgVo.getType())) {
                    postImgVo.setImgUrl(buildImgCover(postImgVo.getImgUrl()));
                    result.add(postImgVo);
                }
                if (result.size() >= size) {
                    break;
                }
            }
        }

        return result;
    }

    /**
     * 视频图转封面
     *
     * @param videoList 视频图列表
     * @param size      个数
     * @return 封面图
     */
    private List<PostImgVo> getCoverByVideo(List<PostImgVo> videoList, Integer size) {
        List<PostImgVo> result = Lists.newArrayList();
        if (!CollectionUtils.isEmpty(videoList)) {
            for (PostImgVo postImgVo : videoList) {
                if (isBlank(postImgVo.getType()) || isBlank(postImgVo.getImgUrl())) {
                    continue;
                }
                if ("VIDEO".equals(postImgVo.getType())) {
                    postImgVo.setImgUrl(buildVideoCover(postImgVo.getImgUrl()));
                    result.add(postImgVo);
                }
                if (result.size() >= size) {
                    break;
                }
            }
        }
        return result;
    }

    private List<PostImgVo> getCoverByCoverStr(String coverStr) {
        if (StringUtils.isNotBlank(coverStr)) {
            List<String> coverList = JSONObject.parseArray(coverStr, String.class);
            return coverList.stream()
                    .filter(StringUtils::isNotBlank)
                    .map(this::buildImgCover).map(PostImgVo::buildImg)
                    .limit(3)
                    .collect(Collectors.toList());
        }
        return Lists.newArrayList();
    }

    public String buildImgCover(String imgUrl) {
        int lastIndex = imgUrl.lastIndexOf("?");
        if (-1 != lastIndex) {
            imgUrl = imgUrl.substring(0, lastIndex);
        }

        String extend = FileUtils.getFilextension(imgUrl);

        if ("gif".equalsIgnoreCase(extend)) {
            imgUrl += "?x-oss-process=style/mcompress";
        } else {
            imgUrl += "?x-oss-process=style/thumbnail";
        }

        return imgUrl;
    }

    public String buildVideoUrlCover(String videoUrl) {
        if (isBlank(videoUrl)) {
            return videoUrl;
        }

        return replace(videoUrl, ossProperties.getUrlPrefix(), ossProperties.getMediaUrlPrefix());
    }

    public String buildVideoCover(String imgUrl) {
        int lastIndex = imgUrl.lastIndexOf("?");
        if (-1 != lastIndex) {
            imgUrl = imgUrl.substring(0, lastIndex);
        }
        imgUrl += "?x-oss-process=style/vthumbnail";
        return imgUrl;
    }
}
